/*
	extras.cpp - Extras
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "extras.h"
#include "miscutils.h"
#include "mmainwindow.h"

#include <qdir.h>

#include <kdebug.h>
#include <kdesktopfile.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kpopupmenu.h>
#include <kpushbutton.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kstringhandler.h>

Extras *Extras::_instance = 0;

// public

Extras::~Extras()
{
	if (_idToFileMap)
		delete _idToFileMap;
	if (_menu)
		delete _menu;
}

void Extras::createButton(QWidget *parent)
{
	_button = new KPushButton(parent, "KPushButton::_button");
	_button->hide();
	_button->setPopup(_menu);
	MiscUtils::setHint(_button, i18n("More actions..."));
	setAction(QString::null, QString::null); /* evil */
}

bool Extras::execAction() const
{
	if (_fileToExecute.isNull())
		return false; // error

	if (!execFile())
	{
		MiscUtils::showRunErrorMessage(_fileToExecute);

		return false; // error
	}

	return true; // ok
}

QString Extras::getActionDescription() const
{
	return
		_description.isEmpty()
			? QString::null
			// "Substitute characters at the middle of a string by \"...\""
			: KStringHandler::csqueeze(_description, 100);
}

// private

Extras::Extras()
	: QObject(ks_main, "Extras"),
	_idToFileMap(0) // initialized in getIdToFileMap()
{
	_menu = new KPopupMenu(0, "KPopupMenu::_menu");
	connect(_menu, SIGNAL(aboutToShow()), SLOT(slotShowMenu()));
	connect(_menu, SIGNAL(activated(int)), SLOT(slotActivated(int)));
}

// TODO: 0.9: recent items
void Extras::buildMenu(KPopupMenu *parentMenu, const QString &subdir)
{
	int id;
	QDir d(subdir);
	QFileInfo fi;
	QString
		description,
		icon,
		mimeType,
		name,
		path;

	QStringList list = d.entryList(QDir::DefaultFilter, QDir::Name | QDir::DirsFirst);
	list.remove(".");
	list.remove("..");

	uint count = list.count();
	for (uint i = 0; i < count; i++)
	{
		fi.setFile(d, list[i]);
		path = fi.filePath();

		// dir
		if (fi.isDir())
		{
			KPopupMenu *submenu = new KPopupMenu(parentMenu, "KPopupMenu::submenu");
			connect(submenu, SIGNAL(activated(int)), SLOT(slotActivated(int)));

			KDesktopFile *desktopFile = new KDesktopFile(path + "/.directory", true);
			if (desktopFile->getConfigState() != KConfigBase::NoAccess)
			{
				icon = desktopFile->readIcon();
				if (icon.isEmpty())
					icon = "folder";
				name = desktopFile->readName();
				if (name.isEmpty())
					name = fi.baseName();
				description = desktopFile->readComment();
				// add item
				parentMenu->insertItem(
					SmallIcon(icon),
					description.isEmpty() ? name : description,
					submenu
				);
			}
			else
			{
				// add item
				parentMenu->insertItem(SmallIcon("folder"), fi.baseName(), submenu);
			}
			delete desktopFile;

			// recurse
			buildMenu(submenu, fi.filePath());
		}
		// file
		else if (fi.isFile())
		{
			mimeType = KMimeType::findByPath(path)->name();

			if (mimeType == "application/x-trash")
				continue;

			if (mimeType != "application/x-desktop")
			{
				// add item
				id = parentMenu->insertItem(
					KMimeType::pixmapForURL(KURL::fromPathOrURL(path)),
					fi.baseName()
				);
				// map item
				getIdToFileMap()->insert(id, path);

				continue;
			}

			// application/x-desktop
			KDesktopFile *desktopFile = new KDesktopFile(path, true);
			if (desktopFile->getConfigState() != KConfigBase::NoAccess)
			{
				name = desktopFile->readName();
				if (name.isEmpty())
					name = desktopFile->readEntry("Exec");
				description = desktopFile->readComment();
				// add item
				id = parentMenu->insertItem(
					SmallIcon(desktopFile->readIcon()),
					description.isEmpty()
						? name
						: (description + " (" + name + ")")
				);
				// map item
				getIdToFileMap()->insert(id, path);
			}
			delete desktopFile;
		}
	}
}

bool Extras::execFile() const
{
	if (_fileToExecute.isNull())
	{
		// kdDebug() << "Extras::execFile(): No file to execute" << endl;

		return false;
	}

	pid_t pid = KRun::runURL(
		KURL::fromPathOrURL(_fileToExecute),
		KMimeType::findByPath(_fileToExecute)->name()
	);

	return pid;
}

QMap<int, QString> *Extras::getIdToFileMap()
{
	if (!_idToFileMap)
		_idToFileMap = new QMap<int, QString>;

	return _idToFileMap;
}

void Extras::setAction(const QString &fileToExecute, const QString &description)
{
	_description = description;
	_fileToExecute = fileToExecute;
	if (_fileToExecute.isNull())
	{
		_button->setText(i18n("Select a command..."));
		_button->setIconSet(ks_actions->getIcon(Action::Extras));
	}
	else
	{
		_button->setText(getActionDescription());
// TODO: 0.9: selected item icon
		_button->setIconSet(ks_actions->getIcon(Action::Extras));
	}
}

// public slots

void Extras::slotModify()
{
	KMessageBox::information(
		0,
		"<qt>" +
		i18n("Use context menu to add/edit/remove links.") +
		"<ul>" +
		"<li>" + i18n("Use <b>Context Menu</b> " \
			"to create a new link to application") + "</li>" +
		"<li>" + i18n("Use <b>Create New|Folder...</b> " \
			"to create a new submenu") + "</li>" +
		"<li>" + i18n("Use <b>Properties</b> " \
			"to change icon, name, or comment") + "</li>" +
		"</ul></qt>",
		i18n("Extras"), // title
		"ModifyExtras" // config key
	);

	// launch Konqueror as an item editor
	MiscUtils::runCommand("konqueror \"" + KGlobal::dirs()->saveLocation("data", "kshutdown/extras") + "\"");
}

// private slots

void Extras::slotActivated(int id)
{
	QMap<int, QString>::iterator i = getIdToFileMap()->find(id);
	if (i == getIdToFileMap()->end())
	{
		setAction(QString::null, QString::null);
		KMessageBox::error(
			0,
			i18n("Internal error!\nSelected menu item is broken."),
			i18n("Extras")
		);
	}
	else
	{
		setAction(i.data(), _menu->text(id));
	}
}

void Extras::slotShowMenu()
{
	setAction(QString::null, QString::null);
	_menu->clear(); // reset menu
	getIdToFileMap()->clear(); // reset map
	QStringList extrasDirs(KGlobal::dirs()->findDirs("data", "kshutdown/extras"));
	QStringList::ConstIterator
		begin = extrasDirs.begin(),
		end = extrasDirs.end();
	for (QStringList::ConstIterator it = begin; it != end; ++it)
	{
		if ((it != begin) && (_menu->count() > 0))
			_menu->insertSeparator();
		buildMenu(_menu, *it);
	}
}
