/***************************************************************************
                          jabbercore.cpp  -  description
                             -------------------
    begin                : Thu Sep 30 1999
    copyright            : (C) 1999 by Seth Hartbecke
    email                : gandalf@netins.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include "jabbercore.h"
#include "user.h"
#include "group.h"
#include "agent.h"

JabberCore *jabber;

JabberCore::JabberCore(QWidget *parent, const char *name )
	: XMLStream(parent,name)
{
	jabber = this;

	connectStatusDlg = 0;
	messageStore = new MessageBin(0, "Message Store");
	infoQueryManager = new QueryManager(0, "Info Query manager");
        _prefs = new PrefsDB( this, "Preferences Database" );


	connect( this, SIGNAL(connected()), messageStore, SLOT(systemOnline()) );
	// This will regester the xmlns jabber:iq:roster
	queryManager()->registerNS( "jabber:iq:roster", this, SLOT(rosterPushed(InfoQuery *)) );
	queryManager()->registerNS( "jabber:iq:version", this, SLOT(versionRequest(InfoQuery *)) );
}

JabberCore::~JabberCore()
{
	delete messageStore;
	delete infoQueryManager;
}


void JabberCore::connectToServer(QString server, int port, QString user, QString pass, QString nick, bool create )
{
	int retVal;

	connectStatusDlg = new DlgConnectInProgress( 0, 0 );
        connect( connectStatusDlg, SIGNAL(cancelConnect()), this, SLOT(connectCanceled()) );
	QString caption;
	caption.sprintf("Connecting to %s", (const char *) server);
	connectStatusDlg->setCaption( caption );
	connectStatusDlg->show();
	/* We are about to do many things that may not allow the screen to update for some time
	   this informs QT to let any screen updates that are pending (like hiding the login dialog)
	   to happen now */
	qApp->processEvents();

	retVal = connectXMLStream( server, port, "jabber:client", "http://etherx.jabber.org/streams" );

	if (retVal != 0)
	{
		KMessageBox errDlg;
		// Setup of the new stream failed
		switch (retVal)
		{
		 	case DNS_NOT_FOUND:
		 		errDlg.sorry( 0, "Could not find a DNS entry for the server", "Konverse Error");
	 			break;
		 	case CONNECTION_REJECTED:
		 		errDlg.sorry( 0, "The server rejected the connection", "Konverse Error");
	 			break;
		 	case NO_RESPONCE:
		 		errDlg.sorry( 0, "The server did not respond", "Konverse Error");
	 			break;
		 	default:
		 		errDlg.sorry( 0, "Could not connect to the server...for some reason", "Konverse Error");
	 			break;
		 };

		if (connectStatusDlg != 0)
		{
			delete connectStatusDlg;
			connectStatusDlg = 0;
		}

	}

	_user = user;
	_pass = pass;
	_nick = nick;
	_server = server;
	createAccount = create;

	return;
}

void JabberCore::connectCanceled()
{
        disconnect();

	if (connectStatusDlg != 0)
	{
		delete connectStatusDlg;
		connectStatusDlg = 0;
	}
}

void JabberCore::rootTagRecieved( XMLNode * )
{
	connectStatusDlg->setCaption("Logging in");

	// Now create the login packet or create packet or whatever.
	InfoQuery *loginRequest = queryManager()->createQuery();
	if (!createAccount)
	{
		loginRequest->setNameSpace("jabber:iq:auth");
		loginRequest->getChildNode("query")->getChildNode("resource")->setValue(_nick);
	}
	else
		loginRequest->setNameSpace("jabber:iq:register");

	loginRequest->getChildNode("query")->getChildNode("password")->setValue(_pass);
	loginRequest->getChildNode("query")->getChildNode("username")->setValue(_user);
	loginRequest->setTo(_server);
	loginRequest->setType("set");
	connect( loginRequest, SIGNAL(responceRecieved(InfoQuery *, InfoQuery *)), this, SLOT(loginResults(InfoQuery *, InfoQuery *)) );
	loginRequest->send();
}

void JabberCore::setStatus( StatusType show, int priority, QString description )
{
	QString txtPriority;
        txtPriority.setNum( priority );
 	XMLNode presence;
        presence.setName( "presence" );
        presence.getChildNode( "status" )->setValue( description );
        presence.getChildNode( "priority" )->setValue( txtPriority );
 	switch( show )
        {
         	case Chat:
                	presence.getChildNode( "show" )->setValue( "chat" );
                 	break;
           	case Away:
                	presence.getChildNode( "show" )->setValue( "away" );
                        break;
            	case ExtendedAway:
                	presence.getChildNode( "show" )->setValue( "xa" );
                        break;
             	case DND:
                	presence.getChildNode( "show" )->setValue( "dnd" );
                        break;
                default:
                	/* We dont send a show for normal */
                        break;
        }
        sendTag( presence );

        _show = show;
        _priority = priority;
        _description = description;

        emit( changedStatus() );
}

void JabberCore::loginResults(InfoQuery *, InfoQuery *responce)
{

	if (strcmp(responce->getAttribute("type"), "result") == 0)
	{
		// Yea...we connected!
		// Say we are availabe
                setStatus( Normal, 10, "Online with Konverse jabber client");

		connectStatusDlg->setCaption("Downloading Preferences");

                connect( _prefs, SIGNAL(readCompleted()), this, SLOT(settingsRecieved()) );
                _prefs->read();

		// Tell the rest of the client
		emit(connected());
	}

	if (strcmp(responce->getAttribute("type"), "error") == 0)
	{
		// Drat
		KMessageBox errDlg;
		if ( createAccount )
			errDlg.sorry( 0, "Could not create your account...that user name may be in use already", "Konverse Error");
		else
			errDlg.sorry( 0, "Could not login.\nCheck your user name and password", "Konverse Error");

		if (connectStatusDlg != 0)
		{
			delete connectStatusDlg;
			connectStatusDlg = 0;
		}


		disconnect();
	}

}

void JabberCore::settingsRecieved()
{
        QObject::disconnect( _prefs, SIGNAL(readCompleted()), this, SLOT(settingsRecieved()) );

	// Get the roster
	connectStatusDlg->setCaption("Requesting roster");
	InfoQuery *getRoster;
	getRoster = queryManager()->createQuery();
	getRoster->setType("get");
	getRoster->setNameSpace("jabber:iq:roster");
	connect( getRoster, SIGNAL(responceRecieved(InfoQuery *, InfoQuery *)), this, SLOT(rosterRecieved(InfoQuery *, InfoQuery *)) );
	getRoster->send();

}

void JabberCore::connectionLost()
{
	emit(disconnected());

        /* Zap all the current users...we don't really know about them anymore */
        while( users.count() > 0 )
        {
                User *u = users.at( 0 );
                users.remove( u );
                delete u;
        }

        _agents.clear();
}

JabberID JabberCore::whoAmI()
{
	JabberID me( _user, _server, _nick );
	return me;
}

// These functions provide access to the various "managers"
MessageBin *JabberCore::messageBin()
{
	return messageStore;
}

QueryManager *JabberCore::queryManager()
{
	return infoQueryManager;
}


int JabberCore::numberUsers()
{
	return users.count();
}

User *JabberCore::getUser(int n)
{
	return users.at(n);
}

User *JabberCore::getUser(QString s)
{
        JabberID jid( s );

        return getUser( jid );
}

User *JabberCore::getUser(JabberID jid)
{
 	unsigned int count_i;
 	User *item;

        if (jid.getDomain().isNull() || jid.getNode().isNull())
                return 0;

 	for (count_i=0; count_i<users.count(); count_i++)
 	{
 		item = users.at(count_i);
 		if (strcasecmp(item->getDomain(), jid.getDomain()) == 0
                        && strcasecmp(item->getNode(), jid.getNode()) == 0)
 		{
 			return item;
 		}
 	}

 	return 0;
}

User *JabberCore::fetchUser(QString jid)
{
	User *anewUser;
	anewUser = new User(jid);

	users.append(anewUser);

        connect( anewUser, SIGNAL( userDeleted(User *) ), this, SLOT(userDeleted(User *)) );

	emit( newUser(anewUser) );

	return anewUser;
}

void JabberCore::userDeleted( User *u )
{
        users.remove( u );
}

void JabberCore::recievedTag(XMLNode *t)
{
        printf("Recieved From Server:\n");
        printf("\e[1m\e[34m");  // Make things blue
	XMLNode::printXML( t );
        printf("\e[0m");

	if (strcmp(t->getName(), "message") == 0)
	 	messageStore->incomingPacket( t );

	if (strcmp(t->getName(), "iq") == 0)
		infoQueryManager->incomingPacket( t );

	if (strcmp(t->getName(), "presence") == 0)
	{
                User *usr;
		JabberID jid; jid = t->getAttribute("from");
                QString searchBy;
                if (jid.getNode().isEmpty())
                        searchBy = jid.getDomain();
                else
                        searchBy.sprintf("%s@%s", (const char *)jid.getNode(), (const char *)jid.getDomain() );

		usr = getUser(searchBy);
		if (usr == 0)
			usr = fetchUser(searchBy);

		usr->processPresence(t);
	}

	delete t;
}

void JabberCore::rosterPushed(InfoQuery *result)
{
 	int count_i;
 	XMLNode *item;
 	User *usr;

 	for (count_i=0; count_i<result->getChildNode("query")->numberChildNodes(); count_i++)
 	{
 	 	item = result->getChildNode("query")->getChildNode(count_i);
                if( item->getAttribute("subscription") != "remove" )
                {
         	 	JabberID jid; jid = item->getAttribute("jid");
                        QString searchBy;
                        if (jid.getNode().isEmpty())
                                searchBy = jid.getDomain();
                        else
                                searchBy.sprintf("%s@%s", (const char *)jid.getNode(), (const char *)jid.getDomain() );


        		usr = getUser(searchBy);
		        if (usr == 0)
        			usr = fetchUser(searchBy);

		        usr->processItemTag(item);
                }
	}

}

void JabberCore::rosterRecieved(InfoQuery *, InfoQuery *result)
{
	rosterPushed(result);

	if (connectStatusDlg != 0)
		connectStatusDlg->setCaption("Requesting agents list");

        // Now...request the agents list
	InfoQuery *getAgents = queryManager()->createQuery();
	getAgents->setType("get");
	getAgents->setNameSpace("jabber:iq:agents");
	connect( getAgents, SIGNAL(responceRecieved(InfoQuery *, InfoQuery *)), this, SLOT(agentsRecieved(InfoQuery *, InfoQuery *)) );
	getAgents->send();
}

void JabberCore::agentsRecieved(InfoQuery *, InfoQuery *result)
{
	if (connectStatusDlg != 0)
	{
		delete connectStatusDlg;
		connectStatusDlg = 0;
	}

        int i;
        for ( i=0; i<result->getChildNode("query")->numberChildNodes(); i++ )
        {
        	Agent *agent = new Agent( result->getChildNode("query")->getChildNode( i ) );
                if ( _agents.contains( agent->jid() ) )
                	delete agent;
                else
                	_agents.insert( agent->jid(), agent );
	}


}

Group *JabberCore::getGroup( QString name, bool create )
{
 	if ( _groups.contains( name ) )
        	return _groups[ name ];
  	else
        {
        	if (create == true)
                {
                	Group *ng = new Group( name );
                        _groups.insert( name, ng );
                        emit( newGroup( ng ) );
                        connect( ng, SIGNAL(deletedGroup(Group *)), this, SLOT(deleteGroupNotifiy(Group *)) );
                        return ng;
                }
                else
                	return 0;
        }
}

Group *JabberCore::getGroup( int n )
{
 	QMap<QString, Group *>::Iterator it;
        int i=0;

        for( it = _groups.begin(); it != _groups.end(); ++it )
        {
               if (i == n)
        		return it.data();

        	i++;
        }

        return 0;
}

void JabberCore::deleteGroupNotifiy(Group *g)
{
	_groups.remove( g->getName() );
	emit( deletedGroup(g) );
}

Agent *JabberCore::getAgent( int n )
{
 	QMap<QString, Agent *>::Iterator it;
        int i=0;

        for( it = _agents.begin(); it != _agents.end(); ++it )
        {
               if (i == n)
        		return it.data();

        	i++;
        }

        return 0;
}

void JabberCore::versionRequest( InfoQuery *request )
{
        XMLNode responce("iq");
        responce.setAttribute( "type", "result" );
        if ( !request->getID().isNull() )
                responce.setAttribute( "id", request->getID() );
        responce.setAttribute( "to", request->getAttribute( "from" ) );
        responce.getChildNode( "query" )->setAttribute( "xmlns", "jabber:iq:version" );
        responce.getChildNode( "query" )->getChildNode( "name" )->setValue( "Konverse" );
        responce.getChildNode( "query" )->getChildNode( "version" )->setValue( "0.2" );
        responce.getChildNode( "query" )->getChildNode( "os" )->setValue( "Some Unix" );

        sendTag( responce );
}

PrefsDB *JabberCore::prefs()
{
        return _prefs;
}

QDateTime JabberCore::convertFromJabberTime( QString &source )
{
        QDateTime dt;
        /* We have to take the time apart first */
        QDate date( source.mid( 0, 4 ).toInt(), source.mid( 4, 2 ).toInt(), source.mid( 6, 2 ).toInt() );
        dt.setDate( date );
        QTime t( source.mid( 9, 2 ).toInt(), source.mid( 12, 2 ).toInt(), source.mid( 15, 2 ).toInt() );
        dt.setTime( t );


        /* This is called just to get the extern int timezone populated */
        time_t dummy = 0;
        struct tm *local = localtime( &dummy );

        QDateTime retVal = dt.addSecs( -1 * timezone );

        return retVal;
}

QString JabberCore::convertToJabberTime( QDateTime &source )
{
        /* This is called just to get the extern int timezone populated */
        time_t dummy = 0;
        struct tm *local = localtime( &dummy );

        QDateTime dt = source.addSecs( timezone );

        /* create the string */
        QString retVal;
        retVal.sprintf("%04d%02d%02dT%02d:%02d:%02d", dt.date().year(), dt.date().month(), dt.date().day(), dt.time().hour(), dt.time().minute(), dt.time().second() );

        return retVal;
}


