/***************************************************************************
                          kobjectimpl.h  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.12 $
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KOBJECTIMPL_H
#define KOBJECTIMPL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qdom.h>
#include <qstring.h>
//#include <qdragobject.h>
#include <qobject.h>
#include "km.h"
//#include "kobject.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif


using namespace KM;

class KContainerImpl;
class KRootContainerImpl;
class KObject;

/** 
 * @short Abstract base class for all implementation.
 * 
 * @author Sergio Moretti
 * 
 * @version $Id: kobjectimpl.h,v 1.12 2001/03/18 16:31:03 sergio Exp $
 * 
 * An object has an identity, (@ref KObjectImpl#id), a state and a
 * configuration.
 * 
 * State of object:
 * @li id()     :     object identifier unique in document
 * @li type()   :     object type
 * @li dom()    :     dom element associated to object
 * 
 */
class KObjectImpl : public QObject 
{
   Q_OBJECT

   friend class KFactoryImpl;
   friend class KObject;

   /** document identifier in XML prolog */
   static const char DOCID[];

   /** xml document identifier string */
   static const char DOCTYPE[];

public:
   virtual ~KObjectImpl();

   /** load state and identity from QDomElement */
   virtual void load(KContainerImpl *container, const QDomElement &e);

   // Public State

   /** identity of object */
   int id() const { return _id; }

   /** get type of object */
   int type() const { return _type; }

   /** DOM Element associated to object */
   QDomElement dom() const { return _dom; }

   const ModFlags & modState() const { return _modState; }

   /** type string */
   QString typeStr() const;

   /** document name */
   virtual const char * docId() const { return DOCID; }

   /** dom document type */
   virtual const char * docType() const { return DOCTYPE; }

   /** return the root container (redefine in root) */
   virtual KRootContainerImpl * root() const { return _root; }

   /** return true if object don't have a parent */
   bool isRoot() const;

   /** return true if this object is a global instance for a manager */
   bool isGlobal() const { return id() == 0; }

   /** return true if this object is a default instance for the document */
   bool isDefault() const { return id() == 1; }

   /** return true if this is a normal object */
   bool isNormal() const { return id() > 1; }

   /** get/set use of global configuration */
   bool useGlobal() const { return (isNormal() && _cfg_global); }

   bool getGlobal() const { return _cfg_global; }

   void setGlobal(bool g);

   /** start up the object */
   virtual bool start() = 0;

   /** stop the object */
   virtual bool kill(bool = false) = 0;

   /** return true if object is active */
   virtual bool isRunning() const { return false; }

   /** run this procedure periodically */
   virtual void runPeriodically();

   /** update the view, called from root container */
   virtual void update() {}

   /** get priority */
   int priority() const { return _priority; }

   /** get priority of object + priority of all its tree */
   int prioritySum() const;

   /** set priority */
   void setPriority(int p);

   /** get message */
   QString message() const { return _msg; }

   /** set message */
   void setMessage(const QString &msg);
   
   /** true if a fatal error has occurred */
   bool fatalError() const { return _fatalError; }
   
   void setFatalError(bool v);

   /** 
    * compare priority
    * call cmpPriorityXXXX to follow selected policy
    * @returns this > t => 1, this == t => 0, this < t => -1
    */
   int cmpPriority(const KObjectImpl *obj) const;

   /**
    * set one modification flag
    * @param p identifier of modification
    * @param v state of modification
    */
   void setMod(ModIndex p);

   /** 
    * set one or more modification flags
    * @param f OR-ed list of flags to set
    */
   void setMods(const ModFlags &f);

   /**
    * get state from one modification flag
    * @param p identifier or modification flag
    * 
    */
   bool isMod(ModIndex p) const;

   /**
    * return true if at least one flags is set
    * @param OR-ed list of flags to test
    */
   bool isMods(const ModFlags &f) const;

   /** reset modification flags */
   void resetMods();

   /**
    * signal the modification of this object
    * an object call this function without parameter, the object function
    * call root's signalMod with its addr, which is responsible to manage
    * the signalling, and to reset the object's mod flag
    */
   void emitMod();

   /** modification string */
   QString modString() const { return bitsToString(_modState); }

   // config entries

protected:
   /** constructor for cloning */
   KObjectImpl(int type);

   /** abstract clone operation */
   virtual KObjectImpl * clone() const = 0;

   /** read object data from QDomElement */
   virtual void loadData();

   /** check and update state on loading */
   virtual void checkState() {}

   /** wait for kill operation to complete */
   void waitForKill();

   /** parent container */
   KContainerImpl * container() const { return _container; }

   /** create a DOM from object */
   virtual QDomDocument toDom() const;

   /** global instance reference */
   const KObjectImpl * global() const { return _global; }

   /** compare policy function */
   virtual int cmpPriorityFifo(const KObjectImpl *obj) const;
   virtual int cmpPriorityLifo(const KObjectImpl *obj) const;
   virtual int cmpPriorityShort(const KObjectImpl *obj) const;
   virtual int cmpPriorityLong(const KObjectImpl *obj) const;

private:
   /** root container */
   KRootContainerImpl *_root;
   /** parent container */
   KContainerImpl *_container;
   /** object identifier */
   int _id;
   /** global instance reference */
   KObjectImpl *_global;
   /** use of global configuration */
   bool _cfg_global;
   /** DOM element */
   QDomElement _dom;
   /** type of object */
   int _type;
   /** modification flags */
   ModFlags _modState;
   /** priority */
   int _priority;
   /** fatal error flag */
   bool _fatalError;
   /** message */
   QString _msg;

   // config entries
};

#endif // KOBJECTIMPL_H

/* Local Variables: */
/* mode: c++ */
/* End: */
