/*
 * dvbstream.h
 *
 * Copyright (C) 2003-2006 Christophe Thommeret <hftom@free.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifndef DVBSTREAM_H
#define DVBSTREAM_H

#include <sys/poll.h>
#include <sys/stat.h>

#include <qthread.h>
#include <qptrlist.h>
#include <qfile.h>
#include <qobject.h>
#include <qtimer.h>
#include <qstringlist.h>

#include "channeldesc.h"
#include "dvbout.h"
#include "ts2rtp.h"
#include "dvbconfig.h"

#define MAX_CHANNELS 8

class DVBevents;
class DvbCam;
class KaffeineDvbPlugin;



struct diseqc_cmd {
	struct dvb_diseqc_master_cmd cmd;
	uint32_t wait;
};



class DvbStream : public QObject, public QThread
{
	Q_OBJECT

public :

	enum { ErrDontSwitchAudio=-1, ErrCantTune=1, ErrCantSetPids=2, ErrIsRecording=3, ErrNoLive=4,
		ErrCantOpenFile=5, ErrIsBroadcasting=6, ErrCamUsed=7 };

	DvbStream( int anum, int tnum, fe_type_t type, const QString &src, bool as, const QString &charset );
	~DvbStream();
	void setPlug( KaffeineDvbPlugin *p );
	void setSources( const QString &src );
	void setLnb( int pos, LNB b );
	QStringList getSources( bool all=false );
	bool canSource( ChannelDesc *chan );
	bool tuneDvb( ChannelDesc *chan, bool dvr=true );
	void stopFrontend();
	virtual void run();
	int goLive( ChannelDesc *chan, const QString &pipeName, int &napid );
	void preStopLive();
	void stopLive( ChannelDesc *chan );
	void stop();
	void stopScan();
	void setScanning( bool b );
	bool hasVideo();
	bool doPause( const QString &name );
	bool timeShiftMode();
	bool running() const;
	ChannelDesc getLiveChannel();
	Transponder getCurrentTransponder();
	bool isTuned();
	bool startTimer( ChannelDesc *chan, QString path, RecTimer *t );
	int canStartTimer( bool &live, ChannelDesc *chan );
	int canStartBroadcast( bool &live, ChannelDesc *chan );
	bool startBroadcast( QPtrList<ChannelDesc> *list, Ts2Rtp *rtp );
	void stopBroadcast();
	bool hasRec();
	bool hasBroadcast();
	bool hasLive();
	bool liveIsRecording();
	int getSNR();

	unsigned char thBuf[188*10];
	struct pollfd pfd;
	DVBevents *dvbEvents;
	int adapter, tuner;
	fe_type_t dvbType;
	bool canAutoscan;

public slots:

	bool checkStatus();
	void receivePlayDvb();
	void recordEnded( DVBout *o, RecTimer *t, bool kill );
	void receiveShifting( bool b );
	void updateTimer( RecTimer *t, int ms );

protected:

	void timerEvent( QTimerEvent* ev );

private :

	int setupSwitch( int switchPos, int voltage18, int hiband );
	int diseqc( fe_sec_voltage_t v, struct diseqc_cmd **cmd, fe_sec_tone_mode_t t, fe_sec_mini_cmd_t b);
	int getSatPos( const QString &src );
	bool setPids( DVBout *o, int napid=0 );
	void removePids( DVBout *o );
	void removeOut( DVBout *o );
	void recordingState();
	void startReading();
	void probeCam();

	QFile liveFile;
	bool timeShifting;
	QPtrList<DVBout> out;
	LNB lnb[4];
	bool openFe();
	bool closeFe();
	void connectStatus( bool con );
	int waitPause;
	DVBout *delOut;
	Transponder currentTransponder;

	QString frontendName;
	QString dvrName;
	QString demuxName;
	int fdFrontend;
	int ndmx;
	int fdDvr, fdPipe;
	bool isRunning;
	QTimer statusTimer;
	QString timeShiftFileName;
	QStringList source;
	DvbCam *cam;
	bool camProbed;
	KaffeineDvbPlugin *plug;

signals:

	void playDvb();
	void isRecording(bool);
	void isBroadcasting(bool);
	void timerEnded(RecTimer*);

	void errorMsg( QString );
	void snrStatus( int );
	void signalStatus( int );
	void lockStatus( bool );
	void shifting( bool );

};

#endif /* DVBSTREAM_H */
