
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
#include <stdarg.h>
#include <errno.h>

#include "typedefs.h"
#include "utils.h"
#include "config.h"
#include "leds.h"
#include "log.h"

			/***************************/
			/******** VARIABLES ********/
			/***************************/

/* configuration file */
static const char *configfile = CFGFILE;

/* indicate when we're finished */
static volatile bool finished = false;

/* indicate when we have to reload configuration file */
static volatile bool reload = false;

/* indicate when we have to reset the logs */
static volatile bool reset_logs = false;

/* if this flag is set, bail() don't exit directly, but rather
 * set some flags to be interpreted later on */
static bool im_a_daemon = false;

/* Mailboxes */
static const char *mbox[3];

/* do we want some cool keyboard led eyecandy on startup? */
static bool led_startup;

/* do we want some cool keyboard led eyecandy when we shutdown? */
static bool led_shutdown;

/* if false, the keyboard leds will stop blinking if your mailbox
 * access times and modification times differ. in other words,
 * the keyboard leds will stop blinking after you've read your mail,
 * even if you haven't deleted the mail. */
static bool leds_on_after_accessed;

/* do we use the Maildir format or standard unix mbox format for mail storage? */
static bool use_maildir_format;

/* led blink rate in milliseconds */
static int led_blink_rate;

/* name of the mail client executable */
static char mail_client[strsize];

/* if true, ixbiff will warn you (by blinking the leds in
 * a certain sequence) if the mail client is active for
 * more than ``warn_delay'' minutes and every ``warn_delay''th
 * minute thereafter. */
static bool warn_if_client_active;

/* the delay in minutes between each mail client launch. this
 * option have no effect if `warn_if_client_active'' is set
 * to false. */
static int warn_delay;

/* if false, the leds will stop blinking when your mail client is
 * active (except for the warning, if enabled, above). */
static bool leds_on_client_launch;

/* Mailboxes assigned to leds */
static char led1[strsize],led2[strsize],led3[strsize];

/* default LED status */
static bool led[3] = { false, false, false };

/* sequence arrays. they're all -1 terminated, hence the extra integer  */
static int startup_sequence[arraysize + 1];
static int shutdown_sequence[arraysize + 1];
static int client_active_sequence[arraysize + 1];

			/***************************/
			/******** FUNCTIONS ********/
			/***************************/

inline void setled (int n) { led[n] = true; }
inline void clrled (int n) { led[n] = false; }

static void bail (const char *fmt, ...)
{
   va_list ap;
   va_start (ap,fmt);
   log_vprintf (LOG_ERROR,fmt,ap);
   va_end (ap);
   if (!im_a_daemon) exit (EXIT_FAILURE); else finished = true;
}

static bool isdir (const char *name)
{
   struct stat filestat;
   if (stat (name,&filestat) < 0) log_printf (LOG_ERROR,"stat(%s): %s\n",name,strerror (errno));
   return (S_ISDIR (filestat.st_mode));
}

static bool newmail (const char *filename)
{
   struct stat filestat;
   DIR *dir;
   struct dirent *entry;
   size_t i = 0;
   char tmp[PATH_MAX];
   if (!use_maildir_format)
	 {
		if (stat (filename,&filestat) < 0) log_printf (LOG_ERROR,"stat(%s): %s\n",filename,strerror (errno));
		if (filestat.st_size > 0 && (leds_on_after_accessed || filestat.st_mtime >= filestat.st_atime)) return (true);
		return (false);
	 }
   else
	 {
		if ((dir = opendir (filename)) == NULL) log_printf (LOG_ERROR,"opendir(%s): %s\n",filename,strerror (errno));
		while ((entry = readdir (dir)) != NULL)
		  {
			 strcpy (tmp,filename);
			 strcat (tmp,"/");
			 strcat (tmp,entry->d_name);
			 if (!isdir (tmp))
			   {
				  if (stat (tmp,&filestat) < 0) log_printf (LOG_ERROR,"stat(%s): %s\n",tmp,strerror (errno));
				  if (filestat.st_size > 0 && (leds_on_after_accessed || filestat.st_mtime >= filestat.st_atime)) i++;
			   }
		  }
		if (closedir (dir) < 0) bail ("unable to close directory %s\n",filename);
		return (i);
	 }
}

static const int xlat[3] = { LED_NUM, LED_CAP, LED_SCR };

static void flashleds ()
{
   int i,mask = 0;

   for (i = 0; i < 3; i++) if (led[i]) mask |= xlat[i];
   led_set (mask);

   usleep (led_blink_rate * 1000);

   led_clear (mask);
   for (i = 0; i < 3; i++) clrled (i);
}

static void show_sequence (int *seq)
{
   int i,j,mask;

   for (i = 1; seq[i] >= 0; i++)
	 {
		mask = 0;
		for (j = 0; j < 3; j++) if (seq[i] & (1 << j)) mask |= xlat[j];
		led_set (mask);

		usleep (seq[0] * 1000);

		led_clear (LED_NUM | LED_CAP | LED_SCR);
	 }
}

static void load_settings ()
{
   static struct
	 {
		int type;
		void *var;
		const char *str;
	 } required_settings[] =
	 {
		{ CFG_TYPE_BOOL, &led_startup, "led_startup" },
		{ CFG_TYPE_BOOL, &led_shutdown, "led_shutdown" },
		{ CFG_TYPE_BOOL, &leds_on_after_accessed, "leds_on_after_accessed" },
		{ CFG_TYPE_BOOL, &use_maildir_format, "use_maildir_format" },
		{ CFG_TYPE_INT, &led_blink_rate, "led_blink_rate" },
		{ CFG_TYPE_STRING, &mail_client, "mail_client" },
		{ CFG_TYPE_BOOL, &warn_if_client_active, "warn_if_client_active" },
		{ CFG_TYPE_INT, &warn_delay, "warn_delay" },
		{ CFG_TYPE_BOOL, &leds_on_client_launch, "leds_on_client_launch" },
		{ -1, NULL, NULL }
	 };
   static struct
	 {
		int *array;
		const char *str;
	 } sequences[] =
	 {
		{ startup_sequence, "startup_sequence" },
		{ shutdown_sequence, "shutdown_sequence" },
		{ client_active_sequence, "client_active_sequence" },
		{ NULL, NULL },
	 };
   int n;
   register int i;
   if (!cfg_open (configfile)) bail ("couldn't open %s for reading\n",configfile);
   for (i = 0; required_settings[i].var != NULL; i++)
	 if (!cfg_get (required_settings[i].type,required_settings[i].var,required_settings[i].str))
	   {
		  cfg_close ();
		  bail ("unable to extract ``%s'' from %s\n",required_settings[i].str,configfile);
	   }
   for (i = 0; sequences[i].str != NULL; i++)
	 {
		if (!cfg_get_int_array (sequences[i].array,&n,sequences[i].str))
		  {
			 cfg_close ();
			 bail ("unable to extract ``%s'' from %s\n",sequences[i].str,configfile);
		  }
		sequences[i].array[n] = -1;
	 }
   mbox[0] = cfg_get_string (led1,"led1") ? led1 : NULL;
   mbox[1] = cfg_get_string (led2,"led2") ? led2 : NULL;
   mbox[2] = cfg_get_string (led3,"led3") ? led3 : NULL;
   cfg_close ();
   if (mbox[0] == NULL && mbox[1] == NULL && mbox[2] == NULL) bail ("no LED's defined in %s\n",configfile);
}

static void signal_handler (int sig)
{
   switch (sig)
	 {
	  case SIGINT:
	  case SIGTERM:
		finished = true;
		break;
	  case SIGHUP:
		reload = true;
		break;
	  case SIGUSR1:
		reset_logs = true;
		break;
	  default:
	 }
}

			/***************************/
			/********** MAIN ***********/
			/***************************/

int main (int argc,char *argv[])
{
   int i;
   const char *errmsg,*progname;
   bool client_is_alive;
   float started = 0;
   log_init (LOGLEVEL);
   (progname = strrchr (argv[0],'/')) ? progname++ : (progname = argv[0]);
   if (geteuid ()) bail ("%s must be run as root\n",progname);
   load_settings ();
   if ((errmsg = led_open ()) != NULL) bail (errmsg);
   if (daemon (0,0) < 0) bail ("unable to succesfully daemonize myself\n");
   im_a_daemon = true;
   signal (SIGINT,signal_handler);
   signal (SIGTERM,signal_handler);
   signal (SIGHUP,signal_handler);
   signal (SIGUSR1,signal_handler);
   if (led_startup) show_sequence (startup_sequence);
   while (!finished)
	 {
		if (reset_logs)
		  {
			 log_reset ();
			 reset_logs = false;
		  }
		if (reload)
		  {
			 load_settings ();
			 reload = false;
		  }
		client_is_alive = alive (mail_client);
		if (warn_if_client_active && client_is_alive)
		  {
			 if (!started) started = time (NULL);
			 if (time (NULL) - started >= warn_delay * 60)
			   {
				  started = 0;
				  show_sequence (client_active_sequence);
			   }
		  }
		else started = 0;
		if (leds_on_client_launch || !client_is_alive)
		  {
			 for (i = 0; i < 3; i++) if (mbox[i] != NULL)
			   if (newmail (mbox[i])) setled (i);
		  }
		flashleds ();
		usleep (500000);
	 }
   if (led_shutdown) show_sequence (shutdown_sequence);
   led_close ();
   log_close ();
   exit (EXIT_SUCCESS);
}

