#include <gnome.h>
#include "gipsc.h"
#include "../../../prips/prips.h"

void gipsc_update_general(const GipscGeneral *gw, const Network *n)
{
	GString *host_max = g_string_new(NULL);
	GString *netmask_hex = g_string_new(NULL);
	GString *subnet_mask_hex = g_string_new(NULL);
	GString *subnet_max = g_string_new(NULL);

	int full_mask = ipsc_network_get_full_mask(n);
	
	g_string_sprintf(subnet_max, "%d", n->subnet_max);
	g_string_sprintf(host_max, "%d", n->host_max);
	g_string_sprintf(netmask_hex, "%X", n->mask);
	g_string_sprintf(subnet_mask_hex, "%X", full_mask);
	
 	gtk_entry_set_text(gw->subnet_max, subnet_max->str);
	gtk_entry_set_text(gw->host_max, host_max->str);
	gtk_entry_set_text(gw->bitmap, (gchar *) n->bitmap);
	gtk_entry_set_text(gw->subnet_mask, (gchar *) denumberize(full_mask)); 
	gtk_entry_set_text(gw->subnet_mask_hex, subnet_mask_hex->str);
	gtk_entry_set_text(gw->netmask, (gchar *) denumberize(n->mask));
	gtk_entry_set_text(gw->netmask_hex, netmask_hex->str);
	
	g_string_free(host_max, TRUE);
	g_string_free(netmask_hex, TRUE);
	g_string_free(subnet_mask_hex, TRUE);
	g_string_free(subnet_max, TRUE);
}

void gipsc_update_subnets(const GipscSubnets *sw, const Network *n)
{
	unsigned int i, network = n->addr, network_end;
	gchar *cols[2];
	
	GString *subnet_number = g_string_new(NULL);
	GString *host_range = g_string_new(NULL);
	
	gtk_clist_freeze( GTK_CLIST(sw->subnet_list));
	gtk_clist_clear( GTK_CLIST(sw->subnet_list));

	for(i = 0; i < n->subnet_max; i++)
	{
		network_end = network + n->host_max -1;

   		g_string_sprintf(subnet_number, "%d", i + 1);
 		g_string_sprintf(host_range, "%s - ", denumberize(network));
		g_string_append(host_range, (gchar *) denumberize(network_end));

		/* put and asteristk next to the current network */ 
		if(n->host_addr >= network && n->host_addr < (network_end +1))
                        g_string_append(host_range, " *");
	
 		cols[0] = subnet_number->str;
 		cols[1] = host_range->str;
  		
      		gtk_clist_append( GTK_CLIST(sw->subnet_list), cols);
		network += n->host_max;
	}
	gtk_clist_thaw( GTK_CLIST(sw->subnet_list));

	g_string_free(subnet_number, TRUE);
	g_string_free(host_range, TRUE);
}

void gipsc_update_host(const GipscHost *hw, const Network *n)
{
	GString *addr_hex = g_string_new(NULL);
	GString *subnet_range = g_string_new(NULL);

	g_string_sprintf(addr_hex, "%X", n->host_addr);
	g_string_sprintf(subnet_range, "%s - ", denumberize(n->host_subnet_lbound)); 
	g_string_append(subnet_range, (gchar *) denumberize(n->host_subnet_ubound));

	gtk_entry_set_text(hw->addr, (gchar *) denumberize(n->host_addr));
 	gtk_entry_set_text(hw->addr_hex, addr_hex->str);
	gtk_entry_set_text(hw->subnet_range, subnet_range->str);
	gtk_entry_set_text(hw->subnet_mask, (gchar *) denumberize(n->subnet_mask));
	gtk_entry_set_text(hw->subnet_id, (gchar *) 
		denumberize(n->subnet_mask & n->host_addr)); 
	gtk_entry_set_text(hw->network_id, (gchar *)
		denumberize(n->mask & n->host_addr));
	gtk_entry_set_text(hw->host_id, (gchar *) 
		denumberize(n->host_id));

	g_string_free(addr_hex, TRUE);
	g_string_free(subnet_range, TRUE);
}

void gipsc_update_cidr(const GipscCidr *cw, const Network *n)
{
	GString *cidr_notation = g_string_new(NULL);
	GString *route_mask = g_string_new(NULL);
	GString *route_mask_hex = g_string_new(NULL);
	GString *supernet_max = g_string_new(NULL);
	GString *cidr_network = g_string_new(NULL);

	g_string_sprintf(cidr_notation, "%s/%d", denumberize(n->addr), 
			ipsc_network_get_prefix_bits(n));
	g_string_sprintf(route_mask, "%s / ", denumberize(n->addr)); 
	g_string_append(route_mask, (gchar *) denumberize(ipsc_network_get_full_mask(n)));
	g_string_sprintf(route_mask_hex, "%X / %X", n->addr, 
			ipsc_network_get_full_mask(n));
	g_string_sprintf(supernet_max, "%d", ipsc_network_get_supernet_max(n));
	g_string_sprintf(cidr_network, "%s/%d", denumberize(n->addr), n->bits);

	gtk_entry_set_text(cw->cisco_wildcard, (gchar *) 
		denumberize(n->cisco_wildcard));
	gtk_entry_set_text(cw->cidr_notation, cidr_notation->str);
	gtk_entry_set_text(cw->route_mask, route_mask->str);
	gtk_entry_set_text(cw->route_mask_hex, route_mask_hex->str);
	gtk_entry_set_text(cw->supernet_max, supernet_max->str);
	gtk_entry_set_text(cw->cidr_network, cidr_network->str);
	
	g_string_free(cidr_notation, TRUE);
	g_string_free(route_mask, TRUE);
	g_string_free(route_mask_hex, TRUE);
	g_string_free(supernet_max, TRUE);
	g_string_free(cidr_network, TRUE);
}

GtkWidget *gipsc_create_subnets(GipscSubnets *si)
{
	gchar *titles[] = {"Subnet", "Host Range"};
	GtkScrolledWindow *scrolled_window;

	si->subnet_list = GTK_CLIST( gtk_clist_new_with_titles(2, titles));
	gtk_clist_set_column_width(si->subnet_list, 0, 60);
	gtk_clist_column_titles_passive(si->subnet_list);

	scrolled_window = GTK_SCROLLED_WINDOW( gtk_scrolled_window_new(NULL, NULL));
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW(scrolled_window), 
					GTK_POLICY_AUTOMATIC, 
		 			GTK_POLICY_AUTOMATIC);

	si->tab_label = GTK_LABEL( gtk_label_new("Subnets/Hosts"));
	
	gtk_container_add( GTK_CONTAINER(scrolled_window), 
		GTK_WIDGET(si->subnet_list));

	return GTK_WIDGET(scrolled_window);
}

GtkWidget *gipsc_create_host(GipscHost *hi)
{
	GtkTable *table;
	GtkWidget *vbox;
	GtkWidget *label;
	
	table = GTK_TABLE( gtk_table_new(4, 2, TRUE));
	gtk_container_border_width(GTK_CONTAINER(table), 5);

 	hi->addr = GTK_ENTRY( gtk_entry_new());
	hi->addr_hex = GTK_ENTRY( gtk_entry_new());
	hi->subnet_range = GTK_ENTRY( gtk_entry_new());
	hi->subnet_mask	= GTK_ENTRY( gtk_entry_new());
	hi->subnet_id = GTK_ENTRY( gtk_entry_new());
	hi->network_id = GTK_ENTRY( gtk_entry_new());
	hi->host_id = GTK_ENTRY( gtk_entry_new());
	
	gtk_entry_set_editable(hi->addr, FALSE);
	gtk_entry_set_editable(hi->addr_hex, FALSE);
	gtk_entry_set_editable(hi->subnet_range, FALSE);
	gtk_entry_set_editable(hi->subnet_mask, FALSE);
	gtk_entry_set_editable(hi->subnet_id, FALSE);
	gtk_entry_set_editable(hi->network_id, FALSE);
	gtk_entry_set_editable(hi->host_id, FALSE);
			
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("IP address");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->addr),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 0, 1);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Hexadecimal IP address");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->addr_hex),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 0, 1);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Address allocation range");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->subnet_range),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 2, 1, 2);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Subnet mask");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->subnet_mask), 
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 2, 3);

 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Subnet ID");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->subnet_id),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 2, 3);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Network ID");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->network_id),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 3, 4);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Host ID");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(hi->host_id),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 3, 4);

	hi->tab_label = GTK_LABEL( gtk_label_new("Host"));
	
	return GTK_WIDGET(table);
}

GtkWidget *gipsc_create_cidr(GipscCidr *ci)
{
	GtkTable *table;
	GtkWidget *vbox;
	GtkWidget *label;
	
	table = GTK_TABLE( gtk_table_new(4, 2, TRUE));
	gtk_container_border_width(GTK_CONTAINER(table), 5);

 	ci->cisco_wildcard = GTK_ENTRY( gtk_entry_new());
	ci->cidr_notation  = GTK_ENTRY( gtk_entry_new());
	ci->route_mask = GTK_ENTRY( gtk_entry_new());
	ci->route_mask_hex = GTK_ENTRY( gtk_entry_new());
	ci->supernet_max = GTK_ENTRY( gtk_entry_new());
	ci->cidr_network = GTK_ENTRY( gtk_entry_new());
 
	gtk_entry_set_editable(ci->cisco_wildcard, FALSE);
	gtk_entry_set_editable(ci->cidr_notation, FALSE);
	gtk_entry_set_editable(ci->route_mask, FALSE);
	gtk_entry_set_editable(ci->route_mask_hex, FALSE);
	gtk_entry_set_editable(ci->supernet_max, FALSE);
	gtk_entry_set_editable(ci->cidr_network, FALSE);
			
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("CIDR notation");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ci->cidr_notation),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 0, 1);
 	
	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Supernet maximum");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ci->supernet_max),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 0, 1);
	
	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Network");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ci->cidr_network),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 1, 2);
	
	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Cisco wildcard");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ci->cisco_wildcard),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 1, 2);
	
	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Route/Mask");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ci->route_mask),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 2, 2, 3);
	
	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Hexadecimal route/mask");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ci->route_mask_hex),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 2, 3, 4);
	
	ci->tab_label = GTK_LABEL( gtk_label_new("CIDR"));
	
	return GTK_WIDGET(table);
}

GtkWidget *gipsc_create_general(GipscGeneral *gi)
{
	GtkTable *table;
	GtkWidget *vbox;
	GtkWidget *label;
	
	table = GTK_TABLE( gtk_table_new(4, 2, TRUE));
	gtk_container_border_width( GTK_CONTAINER(table), 5);

 	gi->subnet_max 	= GTK_ENTRY( gtk_entry_new());
	gi->host_max 	= GTK_ENTRY( gtk_entry_new());
	gi->bitmap 	= GTK_ENTRY( gtk_entry_new());
	gi->netmask 	= GTK_ENTRY( gtk_entry_new());
	gi->netmask_hex	= GTK_ENTRY( gtk_entry_new());
	gi->subnet_mask	= GTK_ENTRY( gtk_entry_new());
	gi->subnet_mask_hex = GTK_ENTRY( gtk_entry_new());
	
	gtk_entry_set_editable(gi->subnet_max, FALSE);
	gtk_entry_set_editable(gi->host_max, FALSE);
	gtk_entry_set_editable(gi->bitmap, FALSE);
	gtk_entry_set_editable(gi->netmask, FALSE);
	gtk_entry_set_editable(gi->netmask_hex, FALSE);
	gtk_entry_set_editable(gi->subnet_mask, FALSE);
	gtk_entry_set_editable(gi->subnet_mask_hex, FALSE);
			
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Subnet maximum");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->subnet_max),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 0, 1);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Hosts per subnet");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->host_max),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 0, 1);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Bitmap (n = network, s = subnet, h = host)");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->bitmap),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 2, 1, 2);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Class mask");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->netmask), 
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 2, 3);

 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Hexadecimal class mask");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->netmask_hex),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 2, 3);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Full subnet mask");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->subnet_mask),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 0, 1, 3, 4);
	
 	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new("Full hexadecimal subnet mask");
	gtk_box_pack_start( GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), GTK_WIDGET(gi->subnet_mask_hex),
		FALSE, FALSE, 0);
	gtk_table_attach_defaults(table, vbox, 1, 2, 3, 4);

	gi->tab_label = GTK_LABEL( gtk_label_new("General"));
	
	return GTK_WIDGET(table);
}
