// --------------------------------------------------------------------
// The reference object.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "iperef.h"
#include "ipevisitor.h"
#include "ipestyle.h"
#include "ipepainter.h"

/*! \class IpeReference
  \ingroup obj
  \brief The reference object.

  A reference reuses an object defined in an Ipe style sheet. The
  object is defined as a named template in the style sheet, and can be
  reused arbitrarily often in the document.  This can, for instance,
  be used for backgrounds on multi-page documents.

  It is admissible to refer to an undefined object (that is, the
  current style sheet cascade does not define a template with the
  given name).

*/

//! Create from XML stream.
IpeReference::IpeReference(IpeRepository *rep, const IpeXmlAttributes &attr,
			   IpeString /* data */)
  : IpeObject(rep, attr)
{
  SetStroke(IpeAttribute()); // null stroke
  iName = rep->MakeSymbol(IpeAttribute::ETemplate, attr["name"]);
  iObject = 0;
}

//! Create a reference to the named object in stylesheet.
IpeReference::IpeReference(const IpeStyleSheet *sheet, IpeAttribute name)
  : IpeObject() // null stroke, identity matrix
{
  assert(name.Kind() == IpeAttribute::ETemplate);
  iName = name;
  iObject = sheet->FindTemplate(iName);
}

//! Clone object
IpeObject *IpeReference::Clone() const
{
  return new IpeReference(*this);
}

//! Copy constructor.
IpeReference::IpeReference(const IpeReference &rhs)
  : IpeObject(rhs)
{
  iName = rhs.iName;
  iObject = rhs.iObject;
}

//! Return pointer to this object.
IpeReference *IpeReference::AsReference()
{
  return this;
}

void IpeReference::Accept(IpeVisitor &visitor) const
{
  visitor.VisitReference(this);
}

//! Save in XML format.
void IpeReference::SaveAsXml(IpePainter &painter, IpeStream &stream,
			     IpeString layer) const
{
  stream << "<ref";
  SaveAttributesAsXml(painter, stream, layer);
  stream << " name=\"" << painter.Repository()->String(Name()) << "\"/>\n";
}

//! Draw reference.
/*! If the symbolic attribute is not defined in the current style sheet,
  nothing is drawn at all. */
void IpeReference::Draw(IpePainter &painter) const
{
  if (iObject) {
    painter.Push();
    painter.Transform(Matrix());
    iObject->Draw(painter);
    painter.Pop();
  }
}

void IpeReference::AddToBBox(IpeRect &box, const IpeMatrix &m) const
{
  if (iObject)
    iObject->AddToBBox(box, m * Matrix());
}

void IpeReference::CheckStyle(const IpeStyleSheet *sheet,
			       IpeAttributeSeq &seq) const
{
  iObject = sheet->FindTemplate(iName);
  if (!iObject && std::find(seq.begin(), seq.end(), iName) == seq.end())
    seq.push_back(iName);
}

double IpeReference::Distance(const IpeVector &v, const IpeMatrix &m,
			      double bound) const
{
  if (iObject)
    return iObject->Distance(v, m * Matrix(), bound);
  else
    return bound;
}

void IpeReference::SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
			   IpeVector &pos, double &bound) const
{
  if (iObject)
    iObject->SnapVtx(mouse, m * Matrix(), pos, bound);
}

void IpeReference::SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
			   IpeVector &pos, double &bound) const
{
  if (iObject)
    iObject->SnapBnd(mouse, m * Matrix(), pos, bound);
}

// --------------------------------------------------------------------
