/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#if !defined(WIN32) && !defined(MACINTOSH)
#define _GNU_SOURCE
#include <getopt.h>
#endif

#include "iiwusynth.h"

#ifdef WIN32
#include "config_win32.h"
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#if WITH_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

/** string tokenizer */
typedef struct {
  char* string;
  char* delimiters;
  int offset;
  int len;
} iiwu_strtok;

iiwu_strtok* new_iiwu_strtok(char* s, char* d);
int delete_iiwu_strtok(iiwu_strtok* st);
int iiwu_strtok_set(iiwu_strtok* st, char* s, char* d);
char* iiwu_strtok_next_token(iiwu_strtok* st);
int iiwu_strtok_has_more(iiwu_strtok* st);
int iiwu_strtok_char_index(char c, char* s);

/** some help functions */
int iiwu_is_number(char* a);
int iiwu_is_empty(char* a);

/** the handlers for the command lines */
void iiwu_handle_help(char* a, char* b, char* c);
void iiwu_handle_quit(char* a, char* b, char* c);
void iiwu_handle_noteon(char* a, char* b, char* c);
void iiwu_handle_noteoff(char* a, char* b, char* c);
void iiwu_handle_cc(char* a, char* b, char* c);
void iiwu_handle_prog(char* a, char* b, char* c);
void iiwu_handle_select(char* a, char* b, char* c);
void iiwu_handle_inst(char* a, char* b, char* c);
void iiwu_handle_channels(char* a, char* b, char* c);
void iiwu_handle_load(char* a, char* b, char* c);
void iiwu_handle_fonts(char* a, char* b, char* c);
void iiwu_handle_mstat(char* a, char* b, char* c);

/** the command structures */
typedef struct {
  char* name;                                   /* the name of the command, as typed in on the command line */
  void (*handler)(char* a, char* b, char* c);   /* pointer to the handler for this command */
  char* help;                                   /* a help string */
} iiwu_cmd;

/** the table of all handled commands */
iiwu_cmd iiwu_command[] = {
  { "help", iiwu_handle_help,         "help                    Print command summary" },
  { "quit", iiwu_handle_quit,         "quit                    Quit the synthesizer" },
  { "noteon", iiwu_handle_noteon,     "noteon chan key vel     Send noteon" },
  { "noteoff", iiwu_handle_noteoff,   "noteoff chan key        Send noteoff"  },
  { "cc", iiwu_handle_cc,             "cc chan ctrl value      Send control-change message" },
  { "prog", iiwu_handle_prog,         "prog chan num           Send program-change message" },
  { "select", iiwu_handle_select,     "select chan bank prog   Combination of bank-select and program-change" },
  { "load", iiwu_handle_load,         "load file               Load a SoundFont" },
  { "fonts", iiwu_handle_fonts,       "fonts                   Display the list of loaded SoundFonts" },
  { "inst", iiwu_handle_inst,         "inst font               Print out the available instruments for the font" },
  { "channels", iiwu_handle_channels, "channels                Print out preset of all channels" },
  { "mstat", iiwu_handle_mstat,       "mstat                   Print out the status of the MIDI driver" },
  { NULL, NULL, NULL }
};


void print_usage(void);
void print_help(void);

/*
 * the globals
 */
iiwu_synth_t* synth = NULL;
iiwu_midi_handler_t* midi = NULL;
int iiwu_continue = 1;
char* appname = NULL;
char* midi_driver = NULL;
char* midi_device = NULL;
char* audio_driver = NULL;
char* audio_device = NULL;

/*
 * macros to wrap readline functions
 */
#define MAX_TOKENS 4

#if WITH_READLINE
#define IIWU_MALLOCLINE(n)     NULL
#define IIWU_READLINE(s,n)     (s = readline(NULL))
#define IIWU_RESETLINE(s)      free(s)
#define IIWU_FREELINE(s)     
#define IIWU_HISTLINE(s)       add_history(s)
#define IIWU_CPYLINE(dst,src)  strcpy(dst,src)
#else
#define IIWU_MALLOCLINE(n)     malloc(n)
#define IIWU_READLINE(s,n)     (s = fgets(s,n,stdin))
#define IIWU_RESETLINE(s)     
#define IIWU_FREELINE(s)       free(s)     
#define IIWU_HISTLINE(s)     
#define IIWU_CPYLINE(dst,src)  strcpy(dst,src)
#endif

/*
 * support for the getopt function
 */
#if !defined(WIN32) && !defined(MACINTOSH)
int getopt(int argc, char * const argv[], const char *optstring);
extern char *optarg;
extern int optind, opterr, optopt;
#endif


int main(int argc, char** argv) 
{
  int i, c;
  char* line;
  char* workline;
  iiwu_strtok* st;
  char* token[MAX_TOKENS];
  int arg1 = 1;

  appname = argv[0];

#if !defined(WIN32) && !defined(MACINTOSH)
  /* handle command options. on posix machines only */
  opterr = 0;
  while (1) {
    int option_index = 0;
    static struct option long_options[] = {
      {"midi-driver", 1, 0, 'm'},
      {"midi-device", 1, 0, 'M'},
      {"audio-driver", 1, 0, 'a'},
      {"audio-device", 1, 0, 'A'},
      {"help", 0, 0, 'h'},
      {0, 0, 0, 0}
    };
    c = getopt_long(argc, argv, "hm:M:a:A:", long_options, &option_index);
    if (c == -1) {
      break;
    }
    switch (c) {
    case 0:
      printf ("option %s", long_options[option_index].name);
      if (optarg) {
	printf (" with arg %s", optarg);
      }
      printf ("\n");
      break;
    case 'm':
      midi_driver = optarg;
      break;
    case 'M':
      midi_device = optarg;
      break;
    case 'a':
      audio_driver = optarg;
      break;
    case 'A':
      audio_device = optarg;
      break;
    case 'h':
      print_help();
      exit(0);
    case '?':
      printf ("Unknown option %c\n", optopt);
      print_usage();
      exit(0);
      break;
    default:
      printf ("?? getopt returned character code 0%o ??\n", c);
    }
  }
  arg1 = optind;
#endif

  /* the creation of the synthesizer starts the synthesis thread */
  synth = new_iiwu_synth(128, 1, audio_driver, audio_device);
  if (synth == NULL) {
    printf("Failed to create the synthesizer\n");
    exit(-1);
  }

  /* load the sfont into the synthesizer */
  while (--argc >= arg1) {
    if (iiwu_synth_sfload(synth, argv[argc]) != 0) {
      printf("Failed to load the soundfont\n");
      goto cleanup;
    }
  }

  /* start the midi handler and link it to the synth */
  midi = new_iiwu_midi_handler(synth, midi_driver, midi_device);
  if (midi == NULL) {
    printf("Failed to create the MIDI input handler\n");
    goto cleanup;
  }

  st = new_iiwu_strtok(NULL, NULL);
  if (st == NULL) {
    goto cleanup;
  }
  line = IIWU_MALLOCLINE(256);
  workline = malloc(256);

  /* handle user input */
  while (iiwu_continue) {
    line = IIWU_READLINE(line, 256);
    if (line == NULL) {
      /* break out of the interpreter loop when we read an EOF */
      break;
    }

    /* skip empty and comment lines */
    if ((line[0] == '#') || iiwu_is_empty(line)) {
      IIWU_RESETLINE(line);
      continue;
    }

    /* keep the line in history (readline only) */
    IIWU_HISTLINE(line);

    /* make a working copy of the line */
    IIWU_CPYLINE(workline, line);

    /* tokenize the input line */
    iiwu_strtok_set(st, workline, " \t\n\r");
    for (i = 0; i < MAX_TOKENS; i++) {
      token[i] = iiwu_strtok_has_more(st)? iiwu_strtok_next_token(st) : NULL;
    }

    /* handle the command */
    for (i = 0; iiwu_command[i].name != NULL; i++) {
      if ((strcmp(iiwu_command[i].name, token[0]) == 0)
	  && (iiwu_command[i].handler != NULL)) {
	iiwu_command[i].handler(token[1], token[2], token[3]);
	break;
      }
    }
    if (iiwu_command[i].name == NULL) {
      printf("Unknown command: %s (Try help)\n", token[0]);
    }
    IIWU_RESETLINE(line);
  }

  /* if iiwu_continue is still set, don't exit but do a dummy wait */
  if (iiwu_continue) {
    iiwu_midi_handler_join(midi);
  }

 cleanup:
  IIWU_FREELINE(line);
  if (midi) {
    delete_iiwu_midi_handler(midi);
  }
  if (synth) {
    delete_iiwu_synth(synth);
  }
  return 0;
}


void print_usage() 
{
  printf("Usage: %s [options] [soundfonts]\n", appname);
  printf("Try -h for help.\n");
  exit(0);
}

void print_help() 
{
  printf("Usage: \n");
  printf("  iiwusynth [options] soundfont1 soundfont2 ... \n");
  printf("Possible options:\n");
  printf(" -m, --midi-driver=[label]\n    the midi driver [oss,alsa,winmidi,...]\n\n");
  printf(" -M, --midi-device=[device]\n    the midi device\n\n");
  printf(" -a, --audio-driver=[label]\n    the audio driver [oss,alsa,winmidi,...]\n\n");
  printf(" -A, --audio-device=[device]\n    the audio device\n\n");
  printf(" -h, --help\n    print out this help summary\n\n");
  exit(0);
}

/*      
 *  handlers 
 */
void iiwu_handle_noteon(char* a, char* b, char* c)
{
  if ((a == NULL) || (b == NULL) || (c == NULL)) {
    printf("noteon: too few arguments\n");
    return;
  }
  if (!iiwu_is_number(a) || !iiwu_is_number(b) || !iiwu_is_number(c)) {
    printf("noteon: invalid argument\n");
    return;
  }
  iiwu_synth_noteon(synth, atoi(a), atoi(b), atoi(c));  
}

void iiwu_handle_noteoff(char* a, char* b, char* c)
{
  if ((a == NULL) || (b == NULL)) {
    printf("noteoff: too few arguments\n");
    return;
  }
  if (!iiwu_is_number(a) || !iiwu_is_number(b)) {
    printf("noteon: invalid argument\n");
    return;
  }
  iiwu_synth_noteoff(synth, atoi(a), atoi(b));  
}

void iiwu_handle_cc(char* a, char* b, char* c)
{
  if ((a == NULL) || (b == NULL) || (c == NULL)) {
    printf("cc: too few arguments\n");
    return;
  }
  if (!iiwu_is_number(a) || !iiwu_is_number(b) || !iiwu_is_number(c)) {
    printf("cc: invalid argument\n");
    return;
  }
  iiwu_synth_cc(synth, atoi(a), atoi(b), atoi(c));  
}

void iiwu_handle_prog(char* a, char* b, char* c)
{
  if ((a == NULL) || (b == NULL)) {
    printf("prog: too few arguments\n");
    return;
  }
  if (!iiwu_is_number(a) || !iiwu_is_number(b)) {
    printf("prog: invalid argument\n");
    return;
  }
  iiwu_synth_program_change(synth, atoi(a), atoi(b));  
}

void iiwu_handle_select(char* a, char* b, char* c)
{
  int chan;
  if ((a == NULL) || (b == NULL) || (c == NULL)) {
    printf("preset: too few arguments\n");
    return;
  }
  if (!iiwu_is_number(a) || !iiwu_is_number(b) || !iiwu_is_number(c)) {
    printf("preset: invalid argument\n");
    return;
  }
  chan = atoi(a);
  iiwu_synth_bank_select(synth, chan, (unsigned int) atoi(b));  
  iiwu_synth_program_change(synth, chan, atoi(c));  
}

void iiwu_handle_inst(char* a, char* b, char* c)
{
  char* s;
  int font;
  if (a == NULL) {
    printf("inst: too few arguments\n");
    return;
  }
  if (!iiwu_is_number(a)) {
    printf("inst: invalid argument\n");
    return;
  }
  font = atoi(a);
  s = iiwu_synth_first_preset(synth, font);
  while (s != NULL) {
    printf("%s\n", s);
    s = iiwu_synth_next_preset(synth, font);
  }
}

void iiwu_handle_channels(char* a, char* b, char* c)
{
  int i;
  char* s;
  for (i = 0; i < 16; i++) {
    s = iiwu_synth_get_channel_preset(synth, i);
    if (s != NULL) {
      printf("chan %d, %s\n", i, s);
    } else {
      printf("chan %d, no preset\n", i);
    }
  }
}

void iiwu_handle_load(char* a, char* b, char* c)
{
  if (a == NULL) {
    printf("load: too few arguments\n");
    return;
  }
  if (iiwu_synth_sfload(synth, a) != 0) {
    printf("Failed to load the soundfont\n");
  }
}

void iiwu_handle_fonts(char* a, char* b, char* c)
{
  int i;
  int num = iiwu_synth_sfcount(synth);
  if (num == 0) {
    printf("no SoundFont loaded (try load)\n");
  }
  for (i = 0; i < num; i++) {
    printf("%d %s\n", i, iiwu_synth_sfname(synth, i));
  }
}

void iiwu_handle_mstat(char* a, char* b, char* c)
{
  printf("Dvr=%s, Dev=%s\n", 
	 iiwu_midi_handler_get_driver_name(midi),
	 iiwu_midi_handler_get_device_name(midi));
  printf("Stat=%s, On=%d, Off=%d, Prog=%d, Pbend=%d, Err=%d\n", 
	 iiwu_midi_handler_get_status(midi),
	 iiwu_midi_handler_get_event_count(midi, 0x90),
	 iiwu_midi_handler_get_event_count(midi, 0x80),
	 iiwu_midi_handler_get_event_count(midi, 0xc0),
	 iiwu_midi_handler_get_event_count(midi, 0xe0),
	 iiwu_midi_handler_get_event_count(midi, 0));
}

void iiwu_handle_quit(char* a, char* b, char* c)
{
  iiwu_continue = 0;
}

void iiwu_handle_help(char* a, char* b, char* c)
{
  int i;
  for (i = 0; iiwu_command[i].name != NULL; i++) {
    if (iiwu_command[i].help != NULL) {
      printf("%s\n", iiwu_command[i].help);
    }
  }
}

int iiwu_is_number(char* a)
{
  while (*a != 0) {
    if ((*a < '0') || (*a > '9')) {
      return 0;
    }
    a++;
  }
  return 1;
}

int iiwu_is_empty(char* a)
{
  while (*a != 0) {
    if ((*a != ' ') && (*a != '\t') && (*a != '\n') && (*a != '\r')) {
      return 0;
    }
    a++;
  }
  return 1;
}

/*
 * new_iiwu_strtok
 */
iiwu_strtok* new_iiwu_strtok(char* s, char* d) 
{
  iiwu_strtok* st;
  st = (iiwu_strtok*) malloc(sizeof(iiwu_strtok));
  if (st == NULL) {
    printf("Out of memory");
    return NULL;
  }
  /* Careful! the strings are not copied for speed */
  st->string = s;
  st->delimiters = d;
  st->offset = 0;
  st->len = (s == NULL)? 0 : strlen(s);
  return st;
}

int delete_iiwu_strtok(iiwu_strtok* st) 
{
  if (st == NULL) {
    printf("Null pointer");
    return 0;
  }
  free(st);
  return 0;
}

int iiwu_strtok_set(iiwu_strtok* st, char* s, char* d)
{
  /* Careful! the strings are not copied for speed */
  st->string = s;
  st->delimiters = d;
  st->offset = 0;
  st->len = (s == NULL)? 0 : strlen(s);
  return 0;
}

char* iiwu_strtok_next_token(iiwu_strtok* st) 
{
  int start = st->offset;
  int end;
  if ((st == NULL) || (st->string == NULL) || (st->delimiters == NULL)) {
    printf("Null pointer");
    return NULL;
  }
  if (start >= st->len) {
    return NULL;
  }
  while (iiwu_strtok_char_index(st->string[start], st->delimiters) >= 0) {
    if (start == st->len) {
      return NULL;
    }
    start++;
  }
  end = start + 1;
  while (iiwu_strtok_char_index(st->string[end], st->delimiters) < 0) {
    if (end == st->len) {
      break;
    }
    end++;
  }
  st->string[end] = 0;
  st->offset = end + 1;
  return &st->string[start];
}

int iiwu_strtok_has_more(iiwu_strtok* st) 
{
  int cur = st->offset;
  if ((st == NULL) || (st->string == NULL) || (st->delimiters == NULL)) {
    printf("Null pointer");
    return -1;
  }
  while (cur < st->len) {
    if (iiwu_strtok_char_index(st->string[cur], st->delimiters) < 0) {
      return -1;
    }
    cur++;
  }
  return 0;
}

int iiwu_strtok_char_index(char c, char* s) 
{
  int i;
  if (s == NULL) {
    printf("Null pointer");
    return -1;
  }
  for (i = 0; s[i] != 0; i++) {
    if (s[i] == c) {
      return i;
    }
  }
  return -1;
}
