/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#include "idatareader.h"


#include "ibasicdatasubjects.h"
#include "ibasicparticlesfileloader.h"
#include "ibuffer.h"
#include "icommoneventobservers.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatasubject.h"
#include "idirectory.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iobjectfactory.h"
#include "iparallelmanager.h"
#include "iparallelworker.h"
#include "iuniformgriddata.h"
#include "iuniformgridfileloader.h"
#include "iviewmodule.h"

#include <vtkArrayCalculator.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


//
//  Define static members
//
IOBJECT_DEFINE_TYPE(iDataReader,DataReader,dr,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iDataReader,BoundaryConditions,bc,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,EraseData,e,String,1);
IOBJECT_DEFINE_KEY(iDataReader,ErrorMessage,em,String,1); 
IOBJECT_DEFINE_KEY(iDataReader,IsSet,is,Bool,1); 
IOBJECT_DEFINE_KEY(iDataReader,LoadData,l,String,2); 
IOBJECT_DEFINE_KEY(iDataReader,ShiftData,s,Double,3);
IOBJECT_DEFINE_KEY(iDataReader,ScaledDimension,sd,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,VoxelLocation,vl,Int,1);

IOBJECT_DEFINE_KEY(iDataReader,RecordLength,-l,Int,1);  // hidden key

//
//  Main class
//
iDataReader* iDataReader::New(iViewModule* vm)
{
	int i, j, k;

	IERROR_ASSERT(vm);
	iDataReader *tmp = new iDataReader(vm); IERROR_ASSERT(tmp);
	iObjectFactory::InstallExtensions(tmp);

	iFileLoader *s;
	for(i=0; i<tmp->mExtensions.Size(); i++)
	{
		j = 0;
		while((s = iRequiredCast<iDataReaderExtension>(INFO,tmp->mExtensions[i])->GetLoader(j++)) != 0)
		{
			for(k=0; k<s->NumStreams(); k++)
			{
				iEdition::ApplySettings(s->GetStream(k)->Subject,s->GetStream(k)->Subject->GetObjectType());
			}
			tmp->mLoaders.Add(s);
		}
	}

	iEdition::ApplySettings(tmp,iDataReader::Type());

	return tmp;
}


iDataReader::iDataReader(iViewModule *vm) : iExtendableObject("Data Reader"), mViewModule(vm)
{
	mRecordLength = 4; 

	mTwoCopies = mInExtensionUpdate = false;

	mCellToPointMode = 2;
	mVoxelLocation = VoxelLocation::Vertex;
	mBoundaryConditions = BoundaryConditions::Wall;

    mUniformVectorsLoader = new iUniformVectorsFileLoader(this); IERROR_ASSERT(mUniformVectorsLoader);
    mUniformScalarsLoader = new iUniformScalarsFileLoader(this,mUniformVectorsLoader); IERROR_ASSERT(mUniformScalarsLoader);
    mUniformTensorsLoader = new iUniformTensorsFileLoader(this); IERROR_ASSERT(mUniformTensorsLoader);
    mBasicParticlesLoader = new iBasicParticlesFileLoader(this); IERROR_ASSERT(mBasicParticlesLoader);

	iUniformScalarsDataSubject *us = new iUniformScalarsDataSubject(mUniformScalarsLoader); IERROR_ASSERT(us);
	iUniformVectorsDataSubject *uv = new iUniformVectorsDataSubject(mUniformVectorsLoader); IERROR_ASSERT(uv);
	iUniformTensorsDataSubject *ut = new iUniformTensorsDataSubject(mUniformTensorsLoader); IERROR_ASSERT(ut);
	iBasicParticlesDataSubject *bp = new iBasicParticlesDataSubject(mBasicParticlesLoader); IERROR_ASSERT(bp);

	iEdition::ApplySettings(us,us->GetObjectType());
	iEdition::ApplySettings(uv,uv->GetObjectType());
	iEdition::ApplySettings(ut,ut->GetObjectType());
	iEdition::ApplySettings(bp,bp->GetObjectType());

	mLoaders.Add(mUniformScalarsLoader);
	mLoaders.Add(mUniformVectorsLoader);
	mLoaders.Add(mUniformTensorsLoader);
	mLoaders.Add(mBasicParticlesLoader);

	mCurrentRecord = -1;
	mPeriodicityLeader = 0;

	mDataShift[0] = mDataShift[1] = mDataShift[2] = 0.0f;

	mScaledDim = -1; //max dim

//	mLastFileName = mLastAttemptedFileName = "";

	mDirectory = new iDirectory; IERROR_ASSERT(mDirectory);
}


iDataReader::~iDataReader()
{
	int i;

	delete mDirectory;

	for(i=0; i<mLoaders.Size(); i++) mLoaders[i]->Delete();
}


bool iDataReader::IsAborted(const iDataType &type) const
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type) && mLoaders[i]->IsAborted()) return true;
	}
	return false;
}


void iDataReader::EraseData(const iDataType &type)
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type) && mLoaders[i]->IsThereData(type)) mLoaders[i]->EraseData();
	}
	this->ClearCache();
}


void iDataReader::SetTwoCopies(bool s)
{
	int i;

	if(mTwoCopies != s)
	{
		mTwoCopies = s;

		for(i=0; i<mLoaders.Size(); i++)
		{
			mLoaders[i]->SetTwoCopies(s);
		}
		this->ClearCache();
	}
}


void iDataReader::ReloadData(const iDataType &type)
{
	int i;

	this->GetErrorStatus()->Clear();
	for(i=0; this->GetErrorStatus()->NoError() && i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type) && mLoaders[i]->IsThereData(type))
		{
			this->GetErrorStatus()->Monitor(mLoaders[i]->GetErrorStatus());
			mLoaders[i]->ReadFile(mLoaders[i]->GetLastFileName());
			if(mLoaders[i]->GetObserver()->IsAborted()) this->GetErrorStatus()->SetAbort();

			if(this->GetErrorStatus()->NoError()) mLoaders[i]->Finalize();
			if(mLoaders[i]->GetObserver()->IsAborted()) this->GetErrorStatus()->SetAbort();

			if(this->GetErrorStatus()->NoError()) mLoaders[i]->ShiftData(mDataShift);
			if(mLoaders[i]->GetObserver()->IsAborted()) this->GetErrorStatus()->SetAbort();

			if(this->GetErrorStatus()->IsError())
			{
				mLoaders[i]->EraseData();
				this->ClearCache();
			}
		}
	}
}


void iDataReader::ResetPipeline()
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		mLoaders[i]->Reset();
	}
	this->ClearCache();
}


bool iDataReader::IsThereData(const iDataInfo &info) const
{
	int i;

	for(i=0; i<info.Count(); i++)
	{
		if(this->IsThereData(info.Type(i))) return true;
	}
	return false;
}

	
bool iDataReader::IsThereData(const iDataType &type) const
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type) && mLoaders[i]->IsThereData(type)) return true;
	}
	return false;
}


const iString& iDataReader::GetLastFileName(const iDataType &type) const
{
	static const iString null;
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type)) return mLoaders[i]->GetLastFileName();
	}
	return null;
}


iDataLimits* iDataReader::GetLimits(const iDataType &type) const
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type)) return mLoaders[i]->GetSubject(type)->GetLimits();
	}
	return 0;
}


iDataSubject* iDataReader::GetSubject(const iDataType &type) const
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type)) return mLoaders[i]->GetSubject(type);
	}
	return 0;
}


bool iDataReader::IsSetLeader(const iFileLoader *loader) const
{
	return (mSetLoaders.Size()>0 && mSetLoaders[0]==loader);
}


void iDataReader::LoadFile(const iDataType &type, const iString &fname, bool dryrun, bool savename)
{
	int i;
	
	this->GetErrorStatus()->Clear();

	//
	//  Find a loader that accepts this type.
	//
	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type))
		{
			this->LoadFile(mLoaders[i],fname,dryrun,savename);
			return;
		}
	}

	this->GetErrorStatus()->Set("There is no loader that supports this data type.");
}


void iDataReader::LoadFile(iFileLoader *loader, const iString &name, bool dryrun, bool savename)
{
	int i;

	this->ClearCache();

	if(loader == 0)
	{
		this->GetErrorStatus()->Set("Trying to load file by a non-existent loader.");
		return;
	}

	iString fname(name);
	iDirectory::ExpandFileName(fname);

	mLastAttemptedFileName = fname;

	if(dryrun)
	{
		if(!iFile::IsReadable(fname)) this->GetErrorStatus()->Set("File does not exist.");
		return;
	}
	else
	{
		this->GetErrorStatus()->Monitor(loader->GetErrorStatus());
		loader->ReadFile(fname);
		if(this->GetErrorStatus()->IsError())
		{
			if(!mTwoCopies) loader->EraseData(); // mark data as absent if we keep only one copy
			return;
		}
	}

	iString root, suffix;
	int newrec = loader->DissectFileName(fname,root,suffix);

	//
	//  Update the set
	//
	if(newrec >= 0)
	{
		if(newrec == mCurrentRecord)
		{
			//
			//  Adding to the existent set if the subject is not there yet
			//
			if(mSetLoaders.Find(loader) < 0) mSetLoaders.Add(loader);
		}
		else
		{
			if(savename) this->InsertIntoVisitedFilesList(loader,fname);
			//
			//  Re-establish the set if we are not loading a new one
			//
			if(!this->IsSetLeader(loader))
			{
				this->BreakSet();
				mSetLoaders.Add(loader);
			}
			mCurrentRecord = newrec;
		}
	}
	else
	{
		if(this->IsSetLeader(loader))
		{
			this->BreakSet();
		}
		if(savename) this->InsertIntoVisitedFilesList(loader,fname);
	}

	if(mPeriodicityLeader==0 || loader->GetPriority()<mPeriodicityLeader->GetPriority()) mPeriodicityLeader = loader;
	mLastFileName = fname;
	
	if(dryrun) return; // we are done in that case

	//
	//  Do we have a set? If yes, load the rest of it too
	//
	if(this->IsSetLeader(loader))
	{
		//
		//Is the filename consistent?
		//
		if(!mSetLoaders[0]->IsSeriesFileName(fname))
		{
			this->BreakSet();
		}
		else
		{
			//
			//  Load the data
			//
			for(i=1; this->GetErrorStatus()->NoError() && i<mSetLoaders.Size(); i++) // start at 1 since the leader has been loaded already
			{
				this->GetErrorStatus()->Monitor(mSetLoaders[i]->GetErrorStatus());
				mSetLoaders[i]->ReadFile(mSetLoaders[i]->GetFileName(newrec));
				if(this->GetErrorStatus()->IsError() && !mTwoCopies) mSetLoaders[i]->EraseData(); // mark data as absent if we keep only one copy
			}
		}
	}

	this->GetViewModule()->GetParallelManager()->StartCounters();

	//
	//  Finish updating the data
	//
	loader->Finalize();
	if(loader->GetObserver()->IsAborted()) { loader->EraseData(); this->GetErrorStatus()->SetAbort(); }
	if(this->IsSetLeader(loader))
	{
		for(i=1; this->GetErrorStatus()->NoError() && i<mSetLoaders.Size(); i++) // the leader has been loaded already
		{
			mSetLoaders[i]->Finalize();
			if(mSetLoaders[i]->GetObserver()->IsAborted()) { mSetLoaders[i]->EraseData(); this->GetErrorStatus()->SetAbort(); }
		}
	}
		
	//
	//  Shift the data
	//
	loader->ShiftData(mDataShift);
	if(loader->GetObserver()->IsAborted()) { loader->EraseData(); this->GetErrorStatus()->SetAbort(); }
	if(this->IsSetLeader(loader))
	{
		for(i=1; this->GetErrorStatus()->NoError() && i<mSetLoaders.Size(); i++) // the leader has been loaded already
		{
			mSetLoaders[i]->ShiftData(mDataShift);
			if(mSetLoaders[i]->GetObserver()->IsAborted()) { mSetLoaders[i]->EraseData(); this->GetErrorStatus()->SetAbort(); }
		}
	}

	this->GetViewModule()->GetParallelManager()->StopCounters();
	this->GetViewModule()->UpdateAfterFileLoad();

	//
	//  If this loader has an extension, ask it do to its share of work, but guard against loops if it calls this function
	//
	if(loader->GetReaderExtension() != 0)
	{
		if(mInExtensionUpdate) return;
		mInExtensionUpdate = true;
		loader->GetReaderExtension()->AfterLoadFile(loader,fname,dryrun);
		mInExtensionUpdate = false;
	}
}


void iDataReader::BreakSet()
{
	mCurrentRecord = -1;
	mSetLoaders.Clear();
	mDirectory->Close();
	this->ClearCache();
}


void iDataReader::LoadRecord(int rec, int skip, bool dryrun, bool savename)
{
	if(rec < 0) rec = mCurrentRecord;

	this->GetErrorStatus()->Clear();

	this->ClearCache();
	if(mSetLoaders.Size() == 0)
	{
		this->GetErrorStatus()->Set("File set has not been established.");
		return;
	}

	iString fname = mSetLoaders[0]->GetFileName(rec);
	if(!iFile::IsReadable(fname))
	{
		this->GetErrorStatus()->Set("File is not accessible.");
		return;
	}

	if(skip != 0)
	{
		if(!mDirectory->IsOpened() && !mDirectory->Open(fname))
		{
			this->GetErrorStatus()->Set("Unable to read the directory.");
			return;
		}
	
		while(skip>0 && !fname.IsEmpty())
		{
			fname = mDirectory->GetFile(fname,1);
			if(this->IsAnotherSetLeader(fname)) skip--;
		}

		//
		//  Is this filename from the same series?
		//
		if(!this->IsAnotherSetLeader(fname))
		{
			this->GetErrorStatus()->Set("File does not exist.",10);
			return;
		}
	}

	if(!fname.IsEmpty())
	{
		this->LoadFile(mSetLoaders[0],fname,dryrun,savename);
	}
	else
	{
		this->GetErrorStatus()->Set("File does not exist.",10);
	}
}


const iDataType& iDataReader::GetFileSetDataType(int member) const
{
	if(member<0 || member>=mSetLoaders.Size())
	{
		return iDataType::Null();
	}
	else
	{
		return mSetLoaders[member]->GetDataType(0);
	}
}


const iString& iDataReader::GetLastFileSetName() const
{
	static const iString none;

	if(mSetLoaders.Size() == 0) return none; else return mSetLoaders[0]->GetLastFileName();
}


const iString iDataReader::GetFileSetName(int rec) const
{
	static const iString none;
	
	if(mSetLoaders.Size()>0 && rec>=0) return mSetLoaders[0]->GetFileName(rec); else return none;
}


const iString iDataReader::GetFileSetRoot() const
{
	static const iString none;
	
	if(mSetLoaders.Size() > 0) return mSetLoaders[0]->GetFileRoot(); else return none;
}


bool iDataReader::IsCurrentSetMember(const iString &fname) const
{
	int i;
	for(i=0; i<mSetLoaders.Size(); i++)
	{
		if(mSetLoaders[i]->GetLastFileName() == fname) return true;
	}
	return false;
}


bool iDataReader::IsAnotherSetLeader(const iString &fname) const
{
	if(mSetLoaders.Size() > 0)
	{
		return mSetLoaders[0]->IsSeriesFileName(fname);
	}
	else return false;
}


void iDataReader::SetDataShift(int d, double dx)
{
	if(d>=0 && d<=2) mDataShift[d] = dx;
	this->ClearCache();
}


void iDataReader::ShiftData(double *dr)
{
	int i;

	if(dr == 0) dr = mDataShift;

	this->GetViewModule()->GetParallelManager()->StartCounters();

	for(i=0; i<mLoaders.Size(); i++) if(mLoaders[i]->IsThereData())
	{	
		mLoaders[i]->ShiftData(dr);
	}

	this->GetViewModule()->GetParallelManager()->StopCounters();
}


float iDataReader::GetMemorySize() const
{
	int i;
    float s = 0.0;
	
	for(i=0; i<mLoaders.Size(); i++)
	{	
		s += mLoaders[i]->GetMemorySize();
	}

	return s;
}


void iDataReader::ExtendableObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyScaledDimension(),mScaledDim);
	this->PackValue(s,KeyVoxelLocation(),mVoxelLocation);
	this->PackValue(s,KeyBoundaryConditions(),mBoundaryConditions);

	this->PackValue(s,KeyShiftData(),mDataShift,3);

	//
	//  Hidden key
	//
	this->PackValue(s,KeyRecordLength(),mRecordLength);

	//
	//  Special "query" keys
	//
	this->PackValue(s,KeyErrorMessage(),this->GetErrorStatus()->Message());
	this->PackValue(s,KeyIsSet(),this->IsSet());
}


void iDataReader::ExtendableObjectUnPackStateBody(const iString &s)
{
	int i; double ds[3]; iString s1;

	if(this->UnPackValue(s,KeyScaledDimension(),i)) this->SetScaledDimension(i);
	if(this->UnPackValue(s,KeyVoxelLocation(),i)) this->SetVoxelLocation(i);
	if(this->UnPackValue(s,KeyBoundaryConditions(),i)) this->SetBoundaryConditions(i);

	for(i=0; i<3; i++) ds[i] = mDataShift[i];
	if(this->UnPackValue(s,KeyShiftData(),ds,3)) 
	{
		for(i=0; i<3; i++) this->SetDataShift(i,ds[i]);
		this->ShiftData();
	}

	//
	//  Hidden key
	//
	iObject::ReportMissingKeys(false);
	if(this->UnPackValue(s,KeyRecordLength(),i))
	{
		mRecordLength = i;
		this->ClearCache();
	}
	iObject::ReportMissingKeys(true);

	//
	//  Special "action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeyEraseData(),s1))
	{
		const iDataType &type(iDataType::FindTypeByName(s1));
		if(!type.IsNull())
		{
			this->EraseData(type); 
		}
	}

	iString ws[2];
	if(this->UnPackValue(s,KeyLoadData(),ws,2))
	{
		const iDataType &type(iDataType::FindTypeByName(ws[0]));
		if(type.IsNull())
		{
			mUnPackedSomething = false;
		}
		else
		{
			iDirectory::ExpandFileName(ws[1],type.GetEnvironment(this->GetViewModule()->GetControlModule()->GetShell()));
			this->LoadFile(type,ws[1]);
		}
	}

	iObject::ReportMissingKeys(true);
}


void iDataReader::PackCompleteState(iString &s) const
{
	int i, k; iString s1;

	this->PackState(s);

	//
	//  Pack all subjects for state saving
	//
	for(i=0; i<mLoaders.Size(); i++)
	{
		for(k=0; k<mLoaders[i]->NumStreams(); k++)
		{
			mLoaders[i]->GetSubject(k)->PackCompleteState(s1);
			s += iObjectKey::SpecialSeparator() + mLoaders[i]->GetSubject(k)->GetDataType().GetName() + iObjectKey::Delimiter() + s1;
		}
	}
}


void iDataReader::UnPackCompleteState(const iString &s)
{
	int i, j, k, n;
	iString s1, name;

	this->UnPackState(s);

	//
	//  UnPack all subjects for state restoring
	//
	n = s.Contains(iObjectKey::SpecialSeparator());
	for(j=0; j<n; j++)
	{
		s1 = s.Section(iObjectKey::SpecialSeparator(),j+1,j+1);
		name = s1.Section(iObjectKey::Delimiter(),0,0);
		for(i=0; i<mLoaders.Size(); i++)
		{
			for(k=0; k<mLoaders[i]->NumStreams(); k++) if(mLoaders[i]->GetSubject(k)->GetDataType().IsOfType(name))
			{
				mLoaders[i]->GetSubject(k)->UnPackCompleteState(s1);
			}
		}
	}
}
	

void iDataReader::InsertIntoVisitedFilesList(const iFileLoader *loader, const iString &name)
{
	VisitedFile v;
	v.Name = name;
	v.Loader = loader;
	mVisitedFilesList.Add(v);
}


iProgressEventObserver* iDataReader::GetProgressEventObserver(const iDataType &type)
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type)) return mLoaders[i]->GetObserver();
	}

	return 0;
}


//
//  Outputs
//
vtkDataSet* iDataReader::GetData(const iDataType &type) const
{
	int i;

	for(i=0; i<mLoaders.Size(); i++)
	{
		if(mLoaders[i]->IsUsingData(type))
		{
			return mLoaders[i]->GetData(type);
		}
	}

	return 0;
}


//
//  Decorator functions
//
void iDataReader::SetBoundaryConditions(int v)
{
	int i;

	if(BoundaryConditions::IsValid(v))
	{
		mBoundaryConditions = v;
		for(i=0; i<mLoaders.Size(); i++)
		{
			mLoaders[i]->SetBoundaryConditions(v);
		}
		this->ClearCache();
	}
}


//
void iDataReader::SetScaledDimension(int v)
{
	int i;

	iUniformGridFileLoader *ds;
	for(i=0; i<mLoaders.Size(); i++)
	{
		ds = iUniformGridFileLoader::SafeDownCast(mLoaders[i]);
		if(ds != 0)
		{
			ds->SetScaledDimension(v);
			mScaledDim = ds->GetScaledDimension();
		}
	}
	this->ClearCache();
}


//
void iDataReader::SetVoxelLocation(int v)
{
	int i;

	iUniformGridFileLoader *ds;
	for(i=0; i<mLoaders.Size(); i++)
	{
		ds = iUniformGridFileLoader::SafeDownCast(mLoaders[i]);
		if(ds != 0)
		{
			ds->SetVoxelLocation(v);
			mVoxelLocation = ds->GetVoxelLocation();
		}
	}
	this->ClearCache();
}


//
//  Extension
//
IOBJECTEXTENSION_DEFINE_ABSTRACT(iDataReader);


void iDataReaderExtension::Define()
{
}


void iDataReaderExtension::AfterLoadFile(iFileLoader *loader, const iString &fname, bool dryrun)
{
	if(loader->GetReaderExtension() !=  this)
	{
		IERROR_HIGH("iDataReaderExtension is configured incorrectly.");
		return;
	}

	mLastFileName = fname;
	this->AfterLoadFileBody(loader,fname,dryrun);
}


void iDataReaderExtension::AfterLoadFileBody(iFileLoader *, const iString &, bool)
{
	//
	//  By default do nothing
	//
}

