/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  Commonly used observers
//

#ifndef ICOMMONEVENTOBSERVERS_H
#define ICOMMONEVENTOBSERVERS_H


#include "ieventobserver.h"


class iControlModule;
class iScript;
class iViewModule;

class vtkProcessObject;
class vtkTimerLog;


class iProgressEventObserver : public iEventObserver
{

	IPOINTER_AS_PART(ViewModule);

public:
	
	enum ProgressMode
	{
		_Default = 0,
		_Reading = 1,
		_Projecting = 2,
		_Rendering = 3,
		_Writing = 4,
		_Operating = 5,
		_Formatting = 6,
		_Shifting = 7,
		_Computing = 8
	};

	vtkTypeMacro(iProgressEventObserver,iEventObserver);
	static iProgressEventObserver* New(iViewModule *vm = 0);

	//
	//  Public for manual driving
	//
	virtual void Started(ProgressMode mode);
	virtual void SetProgress(float fraction);
	virtual void Finished();

	void AttachScript(iScript *s);

	void SetMode(ProgressMode m);
	inline int GetMode() const { return mMode; }
	const char* GetLabel() const;

	inline bool IsAborted() const { return mAborted; }

protected:

	iProgressEventObserver(iViewModule *vm);

	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);

	virtual void ChildStarted() = 0;
	virtual void ChildFinished() = 0;
	virtual bool CheckAbort() = 0;
	virtual void SetProgressBody(float fraction) = 0;

private:

	iScript *mScript;
	ProgressMode mMode;
	bool mAborted;
	float mProgressValue;
};


class iAbortRenderEventObserver : public iEventObserver
{
	
	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iAbortRenderEventObserver,iEventObserver);
	static iAbortRenderEventObserver* New(iViewModule *vm = 0);

	inline bool IsCancelled() const { return mCancelled; }
	inline float GetRenderTime() const { return mRenderTime; }

	void SetInteractive(bool s){ mInteractive = s; }
	bool IsInteractive() const { return mInteractive; }

	static void BlockAbortRenderEventObservers(bool s) { mTheseAreBlocked = s; }
	static bool AbortRenderEventObserversBlocked(){ return mTheseAreBlocked; }

	virtual void PostFinished() = 0; // can be manually driven

protected:

	iAbortRenderEventObserver(iViewModule *vm);
	~iAbortRenderEventObserver();
	
	vtkTimerLog* Timer() const { return mRenderTimer; }

	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);

	virtual void Started() = 0;
	virtual void Finished() = 0;
	virtual bool CheckAbort() = 0;

private:

	vtkTimerLog *mRenderTimer;
	float mRenderTime;
	static bool mTheseAreBlocked;
	bool mCancelled, mInProgress, mInteractive;
};


class iSlaveAbortRenderEventObserver : public iEventObserver
{
	
	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iSlaveAbortRenderEventObserver,iEventObserver);
	static iSlaveAbortRenderEventObserver* New(iAbortRenderEventObserver *m = 0);

protected:

	iSlaveAbortRenderEventObserver(iAbortRenderEventObserver *m);
	virtual ~iSlaveAbortRenderEventObserver();
	
	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);

private:

	iAbortRenderEventObserver *mMaster;
};


class iPickEventObserver : public iEventObserver
{
	
	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iPickEventObserver,iEventObserver);
	static iPickEventObserver* New(iViewModule *vm = 0);

protected:

	iPickEventObserver(iViewModule *vm);

	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);
	virtual void Started() = 0;
	virtual void Finished() = 0;

};


class iParallelUpdateEventObserver : public iEventObserver
{
	
	IPOINTER_AS_PART(ControlModule);

public:
	
	vtkTypeMacro(iParallelUpdateEventObserver,iEventObserver);
	static iParallelUpdateEventObserver* New(iControlModule *cm = 0);

	virtual void UpdateInformation() = 0;

protected:

	iParallelUpdateEventObserver(iControlModule *cm);

	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);
};


class iAnimatorEventObserver : public iEventObserver
{
	
	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iAnimatorEventObserver,iEventObserver);
	static iAnimatorEventObserver* New(iViewModule *vm = 0);

	virtual void OnCameraPath(int step) = 0;

protected:

	iAnimatorEventObserver(iViewModule *vm);

	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);
};


class iMarkerEventObserver : public iEventObserver
{
	
	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iMarkerEventObserver,iEventObserver);
	static iMarkerEventObserver* New(iViewModule *vm = 0);

protected:

	iMarkerEventObserver(iViewModule *vm);

	virtual void ExecuteBody(vtkObject *caller, unsigned long event, void *data);
};

#endif // ICOMMONEVENTOBSERVERS_H

