/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icommandinterpreter.h"


#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "idirectory.h"
#include "ifile.h"
#include "ierrorstatus.h"
#include "ishell.h"


using namespace iParameter;


iCommandInterpreter::iCommandInterpreter(iShell *s) : mShell(s), mErrorStatus("Command Interpreter")
{
	mReturnState = 0;
	mExecuted = false;
}


iCommandInterpreter::~iCommandInterpreter()
{
}


void iCommandInterpreter::Start()
{
	this->DislayLineOfText("IFrIT Command Interpreter.",CommandInterpreterType::Result);
	this->DislayLineOfText("Type commands one per line.",CommandInterpreterType::Result);
	this->DislayLineOfText("Type 'help' for a list of available commands.",CommandInterpreterType::Result);

	this->Initialize();
}


void iCommandInterpreter::Stop()
{
	this->Finalize();
}


void iCommandInterpreter::Append(const iString &text, char exec_char)
{
	int i, n;
	bool exec = false;
	iString input, output, line;
	
	mExecuted = false;

	input = text;
	if(exec_char != 0)
	{
		int io = input.Length() - 1;
		while(io>=0 && input.IsWhiteSpace(io)) io--;
		if(io>=0 && input[io]==exec_char)
		{
			//
			//  Remove the semi-colon
			//
			input.Replace(io,' ');
			exec = true;
		}
	}

	//
	//  Clear the error status
	//
	mReturnState = 0;
	this->GetErrorStatus()->Clear();

	n = input.Contains('\n');
	for(i=0; i<n; i++)
	{
		line = input.Section("\n",i,i);
		line.ReduceWhiteSpace();

		//
		//  Shell-specific commands
		//
		this->AnalyseSpecialCommands(line);

		//
		//  Redirection sign reads from an external file
		//
		if(!line.IsEmpty() && line[0]=='<')
		{
			iString s = line.Part(1);
			s.ReduceWhiteSpace();
			iDirectory::ExpandFileName(s,this->GetShell()->GetEnvironment(Environment::Base));

			iFile F(s);
			if(F.Open(iFile::_Read,iFile::_Text))
			{
				iString s;
				while(F.ReadLine(s)) if(!s.IsEmpty())
				{
					this->DislayLineOfText("==> "+s,CommandInterpreterType::Code);
					output += s + "\n";
				}
				F.Close();
			}
			else
			{
				this->DislayLineOfText("File does not exist.",CommandInterpreterType::Error);
				mReturnState = -1;
				return;
			}
			line.Clear();
		}

		//
		//  Help, print, and list are executed at once
		//
		if(this->IsHelpRequest(line))
		{
			this->RunScript(line);
			if(!this->GetErrorStatus()->Message().IsEmpty())
			{
				this->DislayLineOfText("ERROR: "+this->GetErrorStatus()->Message(),CommandInterpreterType::Error);
				mReturnState = -1;
				return;
			}
			line.Clear();
		}

		//
		// Form a script
		//
		if(!line.IsEmpty()) output += line + "\n";
	}

	mBufferedScript += output;

	if(exec) this->Exec();
}


void iCommandInterpreter::Exec()
{
	//
	//  Clear the error status
	//
	mReturnState = 0;
	this->GetErrorStatus()->Clear();

	//
	//  Execute the script
	//
	this->RunScript(mBufferedScript);

	if(this->GetErrorStatus()->NoError())
	{
		mExecutedScript += mBufferedScript;
		mLastExecutedScriptPiece = mBufferedScript;
	}
	mBufferedScript.Clear();
	mExecuted = true;
}


void iCommandInterpreter::RunScript(const iString &text, bool reset)
{
	this->GetErrorStatus()->Monitor(this->Script()->GetErrorStatus());

	this->Script()->SetText(text);
	this->Script()->Execute(!reset); // do not reset the script!!!

	if(this->GetErrorStatus()->IsError())
	{
		this->DislayLineOfText("ERROR: "+this->GetErrorStatus()->Message(),CommandInterpreterType::Error);
		mReturnState = -1;
	}
}


iControlScript* iCommandInterpreter::Script() const
{
	return this->GetShell()->GetControlModule()->GetControlScript();
}


bool iCommandInterpreter::IsHelpRequest(const iString &line) const
{
	return (line.Part(0,4)=="help" && (line.Length()==4 || line[4]==' ')) ||
		line.Part(0,5)=="list " ||
//		line.Part(0,6)=="print " ||
		(line.Part(0,2)==".h" && (line.Length()==2 || line[2]==' ')) ||
		line.Part(0,4)==".hp " || 
		line.Part(0,4)==".ho " || 
//		line.Part(0,3)==".p " ||
		line.Part(0,4)==".lp " || 
		(line.Part(0,3)==".lo" && (line.Length()==3 || line[3]==' '));
}

