/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ianimatorscript.h"


#include "ianimator.h"
#include "icamera.h"
#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "icrosssectionviewsubject.h"
#include "idatareader.h"
#include "idirectory.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "irendertool.h"
#include "iscriptkit.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "isurfaceviewsubject.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkRenderer.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "icalculatortemplate.h"
#include "iscriptkittemplate.h"
#include "iviewfamilytemplate.h"

//
//  macros
//
#include "ianimatorscriptmacro.h"

using namespace iParameter;

#define curScript	iRequiredCast<iAnimatorScript>(INFO,self)


namespace iAnimatorScript_Private
{
	DEFINE_WORKER_REL(Animator,NumberOfFrames,int);
	DEFINE_WORKER_REL(Animator,NumberOfBlendedFrames,int);
	DEFINE_WORKER_REL(Animator,NumberOfTransitionFrames,int);
	DEFINE_WORKER_REL(Animator,FlybySpeed,float);
	DEFINE_WORKER_REL(Animator,Phi,float);
	DEFINE_WORKER_REL(Animator,Theta,float);
	DEFINE_WORKER_REL(Animator,Roll,float);
	DEFINE_WORKER_REL(Animator,Zoom,float);
	DEFINE_WORKER_REL(Animator,CrossSectionSpeed,float);
	DEFINE_WORKER_REL(Animator,LogoOpacity,float);

	DEFINE_WORKER_ABS(Animator,SetStyle,int);
	DEFINE_WORKER_ABS(Animator,SetNumberOfTitlePageFrames,int);
	DEFINE_WORKER_ABS(Animator,SetNumberOfTitlePageBlendedFrames,int);
	DEFINE_WORKER_ABS(Animator,SetLogoPosition,int);
	DEFINE_WORKER_ABS(Camera,SetParallelScale,double);
	DEFINE_WORKER_ABS(Camera,SetParallelProjection,bool);
	DEFINE_WORKER_ABS(Visual,ShowLabel,bool); 
	DEFINE_WORKER_ABS(Visual,ShowColorBars,bool); 
	DEFINE_WORKER_ABS(Visual,ShowBoundingBox,bool);

	DEFINE_WORKER_ABSPTR(Visual,UpdateLabel,float); 

	DEFINE_WORKER_ABS_FILE(Animator,SetTitlePageFile,"Unable to load title page image.");
	DEFINE_WORKER_ABS_FILE(Animator,SetLogoFile,"Unable to load logo image.");

	//
	//  Object showing/hiding functions - for batch operations
	//
	DEFINE_WORKER_SHOW(Surface); 
	DEFINE_WORKER_SHOW(CrossSection); 
	DEFINE_WORKER_SHOW(Volume); 
	DEFINE_WORKER_SHOW(Particles); 
	DEFINE_WORKER_SHOW(VectorField); 
	DEFINE_WORKER_SHOW(TensorField); 

#ifdef ISCRIPT_BACKWARD_COMPATIBLE
//	DEFINE_WORKER_LIMIT(iUniformScalars,LowerLimit);
//	DEFINE_WORKER_LIMIT(iUniformScalars,UpperLimit);
//	DEFINE_WORKER_LIMIT(iBasicParticles,LowerLimit);
//	DEFINE_WORKER_LIMIT(iBasicParticles,UpperLimit);
//	DEFINE_WORKER_LIMIT(iUniformVectors,UpperLimit);
//	DEFINE_WORKER_LIMIT(iUniformTensors,UpperLimit);

	DEFINE_WORKER_SUBJECT_BEGIN(Surface,IsoSurfaceLevel,float,true);
		sub->SetIsoSurfaceLevel(val);
	DEFINE_WORKER_SUBJECT_END;

	DEFINE_WORKER_SUBJECT_BEGIN(CrossSection,Location,double,true);
		sub->SetLocation(val);
	DEFINE_WORKER_SUBJECT_END;
#endif

	//
	//  focal point & camera location
	//
	bool CameraSetFocalPoint(iScript *self, short at, int num, const double *value, int index)
	{
		VECTOR_HELPER_ASSERT(3);

		double p[3];
		curCamera->GetFocalPoint(p);

		VECTOR_HELPER_ASSIGN(3,double);

		curCamera->SetFocalPoint(p);
		curCamera->OrthogonalizeViewUp();
		return true;
	}

	bool CameraSetPosition(iScript *self, short at, int num, const double *value, int index)
	{
		VECTOR_HELPER_ASSERT(3);

		double p[3];
		curCamera->GetPosition(p);

		VECTOR_HELPER_ASSIGN(3,double);

		curCamera->SetPosition(p);
		curCamera->OrthogonalizeViewUp();
		return true;
	}

	bool CameraSetViewUp(iScript *self, short at, int num, const double *value, int index)
	{
		VECTOR_HELPER_ASSERT(3);

		double p[3];
		curCamera->GetViewUp(p);

		VECTOR_HELPER_ASSIGN(3,double);

		double *dop = curCamera->GetDirectionOfProjection();
		double c[3];
		vtkMath::Cross(dop,p,c);
		if(vtkMath::Norm(p)>1.0e-8 && vtkMath::Norm(c)>1.0e-8)
		{
			curCamera->SetViewUp(p);
			curCamera->OrthogonalizeViewUp();
			return true;
		}
		else
		{
			self->GetErrorStatus()->Set("The ViewUp vector cannot coincide with the direction of projection.");
			return false;
		}
	}
};


using namespace iAnimatorScript_Private;


iAnimatorScript* iAnimatorScript::New(iAnimator *an)
{
	IERROR_ASSERT(an);
	return new iAnimatorScript(an);
}


iAnimatorScript::iAnimatorScript(iAnimator *an) : iBasicScript(an->GetViewModule()->GetControlModule()->GetControlScript()) // always embedded
{
	mEmbeddedControlScript = iShellFactory::CreateControlScript(an->GetViewModule()->GetControlModule(),this);
	mAnimator = an;
	mInRender = false;

	//
	//  Alias words (mostly for backward compatibility)
	//
	this->CreateAliasWord("camera-scale","camera-parallel-scale");
	this->CreateAliasWord("frames-per-file","number-of-frames");
	this->CreateAliasWord("blended-frames","number-of-blended-frames");
	this->CreateAliasWord("transition-frames","number-of-transition-frames");

#ifdef ISCRIPT_BACKWARD_COMPATIBLE
	this->CreateAliasWord("x-section-speed","cross-section-speed"); 
	this->CreateAliasWord("xsection-speed","cross-section-speed"); 
	this->CreateAliasWord("fly-by-speed","flyby-speed"); 

	this->CreateAliasWord("mesh-file","uniform-scalars-file");
	this->CreateAliasWord("particle-file","basic-particles-file");
	this->CreateAliasWord("vector-file","uniform-vectors-file");
	this->CreateAliasWord("tensor-file","uniform-tensors-file");
	
	this->CreateAliasWord("xsection","cross-section"); 
	this->CreateAliasWord("vector","vector-field"); 
	this->CreateAliasWord("tensor","tensor-field"); 

	this->CreateAliasWord("mesh-variable-lower-limit","uniform-scalars-lower-limit");
	this->CreateAliasWord("mesh-variable-upper-limit","uniform-scalars-upper-limit");
	this->CreateAliasWord("particle-attribute-lower-limit","basic-particles-lower-limit");
	this->CreateAliasWord("particle-attribute-upper-limit","basic-particles-upper-limit");
	this->CreateAliasWord("vector-upper-limit","uniform-vectors-upper-limit");
	this->CreateAliasWord("tensor-upper-limit","uniform-tensors-upper-limit");
	this->CreateAliasWord("scalar-field-lower-limit","uniform-scalars-lower-limit");
	this->CreateAliasWord("scalar-field-upper-limit","uniform-scalars-upper-limit");
	this->CreateAliasWord("particle-set-lower-limit","basic-particles-lower-limit");
	this->CreateAliasWord("particle-set-upper-limit","basic-particles-upper-limit");
	this->CreateAliasWord("vector-field-upper-limit","uniform-vectors-upper-limit");
	this->CreateAliasWord("tensor-field-upper-limit","uniform-tensors-upper-limit");

	this->CreateAliasWord("xsection-var","cross-section-var");
	this->CreateAliasWord("xsection-dir","cross-section-dir");
	this->CreateAliasWord("xsection-position","cross-section-position");

	this->CreateAliasWord("rotate-scale","rotate");
	this->CreateAliasWord("scale","zoom");
	this->CreateAliasWord("fly-by","flyby");
	this->CreateAliasWord("camera-path","camerapath");
	this->CreateAliasWord("upper-left-corner","ulcorner");
	this->CreateAliasWord("upper-right-corner","urcorner");
	this->CreateAliasWord("lower-left-corner","llcorner");
	this->CreateAliasWord("lower-right-corner","lrcorner");
#endif

	//
	//  Parameter words
	//
	this->AddConstant(new iScriptKit::Constant<int>(this,"static",0));
	this->AddConstant(new iScriptKit::Constant<int>(this,"rotate",1));
	this->AddConstant(new iScriptKit::Constant<int>(this,"tumble",2));
	this->AddConstant(new iScriptKit::Constant<int>(this,"flyby",3));
	this->AddConstant(new iScriptKit::Constant<int>(this,"path",4));
	this->AddConstant(new iScriptKit::Constant<int>(this,"next",-1));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"parallel",true));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"perspective",false));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"hidden",false));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"visible",true));
	this->AddConstant(new iScriptKit::Constant<int>(this,"current",iMath::_LargeInt+1));
	this->AddConstant(new iScriptKit::Constant<int>(this,"ulcorner",0));
	this->AddConstant(new iScriptKit::Constant<int>(this,"urcorner",1));
	this->AddConstant(new iScriptKit::Constant<int>(this,"llcorner",2));
	this->AddConstant(new iScriptKit::Constant<int>(this,"lrcorner",3));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"off",false));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"on",true));

	//
	//  Command words
	//

	//
	//  Assignment operations hidden as statements
	//
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("render",Render);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("render-all",RenderAll);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("render-set",RenderSet);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("load",Load);

	this->AddPrototype< iScriptKit::SimpleStatement >("reset",Reset);
		
	this->AddPrototype< iScriptKit::FunctionCallStatement >("execute control-script",ExecuteControlScript);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("embed control-script",EmbedControlScript);

	//
	//  Genuine assignment operations to pre-defined variables
	//
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("style",AnimatorSetStyle);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("number-of-frames",AnimatorSetNumberOfFrames);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("number-of-blended-frames",AnimatorSetNumberOfBlendedFrames);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("number-of-transition-frames",AnimatorSetNumberOfTransitionFrames);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("flyby-speed",AnimatorSetFlybySpeed);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("rotation-phi",AnimatorSetPhi);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("rotation-theta",AnimatorSetTheta);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("rotation-roll",AnimatorSetRoll);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("zoom",AnimatorSetZoom);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("cross-section-speed",AnimatorSetCrossSectionSpeed);

	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<double> >("camera-parallel-scale",CameraSetParallelScale);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<double> >("camera-view-up",CameraSetViewUp);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<double> >("camera-focal-point",CameraSetFocalPoint);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<double> >("camera-position",CameraSetPosition);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("projection",CameraSetParallelProjection);

	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("record-label",VisualShowLabel);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("color-bars",VisualShowColorBars);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("bounding-box",VisualShowBoundingBox);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("record-label-value",VisualUpdateLabel);

	this->AddPrototype< iScriptKit::FunctionCallStatement >("title-page-file",AnimatorSetTitlePageFile);
	this->AddPrototype< iScriptKit::FunctionCallStatement >("logo-file",AnimatorSetLogoFile);

	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("title-page-number-of-frames",AnimatorSetNumberOfTitlePageFrames);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("title-page-number-of-blended-frames",AnimatorSetNumberOfTitlePageBlendedFrames);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("logo-opacity",AnimatorSetLogoOpacity);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<int> >("logo-position",AnimatorSetLogoPosition);

	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("surface",SurfaceShow);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("cross-section",CrossSectionShow);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("volume",VolumeShow);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("particles",ParticlesShow);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("vector-field",VectorFieldShow);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<bool> >("tensor-field",TensorFieldShow);

	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<float> >("surface-level",SurfaceIsoSurfaceLevel);
	this->AddPrototype< iScriptKit::FunctionCallAssignmentStatement<double> >("cross-section-position",CrossSectionLocation);

	//
	// This is deprecated: use embed control-script instead
	//
	this->AddPrototype< iScriptKit::DeprecatedStatement >("load uniform-scalars-file","Use instead: 'embed control-script exec DataReader:LoadData/Data-UniformScalars/<file-name>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("load basic-particles-file","Use instead: 'embed control-script exec DataReader:LoadData/Data-BasicParticles/<file-name>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("load uniform-vectors-file","Use instead: 'embed control-script exec DataReader:LoadData/Data-UniformVectors/<file-name>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("load uniform-tensors-file","Use instead: 'embed control-script exec DataReader:LoadData/Data-UniformTensors/<file-name>'.");

	this->AddPrototype< iScriptKit::DeprecatedStatement >("uniform-scalars-lower-limit","Use instead: 'embed control-script exec Data-UniformScalars:LowerLimit[<component>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("uniform-scalars-upper-limit","Use instead: 'embed control-script exec Data-UniformScalars:UpperLimit[<component>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("basic-particles-lower-limit","Use instead: 'embed control-script exec Data-UniformScalars:LowerLimit[<component>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("basic-particles-upper-limit","Use instead: 'embed control-script exec Data-UniformScalars:UpperLimit[<component>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("uniform-vectors-upper-limit","Use instead: 'embed control-script exec Data-UniformScalars:UpperLimit[<component>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("uniform-tensors-upper-limit","Use instead: 'embed control-script exec Data-UniformScalars:UpperLimit[<component>]/<value>'.");

	this->AddPrototype< iScriptKit::DeprecatedStatement >("surface-var","Use instead: 'embed control-script exec Surface:IsoSurfaceVar[<instance>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("surface-opacity","Use instead: 'embed control-script exec Surface:Opacity[<instance>]/<value>'.");

	this->AddPrototype< iScriptKit::DeprecatedStatement >("cross-section-var","Use instead: 'embed control-script exec CrossSection:Var[<instance>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("cross-section-dir","Use instead: 'embed control-script exec CrossSection:Dir[<instance>]/<value>'.");

	this->AddPrototype< iScriptKit::DeprecatedStatement >("volume-var","Use instead: 'embed control-script exec Volume:Var[<instance>]/<value>'.");

	this->AddPrototype< iScriptKit::DeprecatedStatement >("particles-opacity","Use instead: 'embed control-script exec Particles:Opacity[<instance>]/<value>'.");

	this->AddPrototype< iScriptKit::DeprecatedStatement >("marker-position","Use instead: 'embed control-script exec Marker:Position[<instance>]/<value>/<value>/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("marker-size","Use instead: 'embed control-script exec Marker:Size[<instance>]/<value>'.");
	this->AddPrototype< iScriptKit::DeprecatedStatement >("marker-opacity","Use instead: 'embed control-script exec Marker:Opacity[<instance>]/<value>'.");
}


iAnimatorScript::~iAnimatorScript()
{
	mEmbeddedControlScript->Delete();
}


bool iAnimatorScript::IsCommandWordLetter(char c) const
{
	//
	//  Also allow a dash in between
	//
	return (c=='-' || this->iBasicScript::IsCommandWordLetter(c));
}


//
//  Reset Animator settings
//
bool iAnimatorScript::Reset(iScript *self)
{
	curAnimator->ResetState();
	curAnimator->Reset();
	curVisual->GetRenderTool()->GetCamera()->Reset();
	return true;
}


//
//  Render scene
//
bool iAnimatorScript::Render(iScript *self, short, int n)
{
	int i;

	curScript->SetInRender(true);

	if(n == -1) n = iMath::_IntMax;
	self->GetErrorStatus()->Monitor(curAnimator->GetErrorStatus(),true);
	for(i=0; i<n && !self->CheckAbort(i,n,1) && self->GetErrorStatus()->NoError(); i++) 
	{
		if(!curAnimator->RenderImages()) self->GetErrorStatus()->Set("Animation finished.",-1);
	}

	curScript->SetInRender(false);

	return self->GetErrorStatus()->NoError();
}


bool iAnimatorScript::RenderAll(iScript *self, short, int n)
{
	int i, k;

	curScript->SetInRender(true);

	for(i=0; i<n && !self->CheckAbort(i,n,1) && self->GetErrorStatus()->NoError(); i++) 
	{
		for(k=0; !self->CheckAbort(k,curControl->GetNumberOfViewModules(),2) && self->GetErrorStatus()->NoError() && k<curControl->GetNumberOfViewModules(); k++) if(k != curControl->GetCurrentViewModuleIndex())
		{
			self->GetErrorStatus()->Monitor(curControl->GetViewModule(k)->GetAnimator()->GetErrorStatus(),true);
			//
			//  Force other animators to continue, even if they are done
			//
			curControl->GetViewModule(k)->GetAnimator()->RenderImages(false);
		}
		if(self->GetErrorStatus()->NoError())
		{
			self->GetErrorStatus()->Monitor(curAnimator->GetErrorStatus(),true);
			if(!curAnimator->RenderImages()) self->GetErrorStatus()->Set("Animation finished.",-1);
		}
	}

	curScript->SetInRender(false);

	return self->GetErrorStatus()->NoError();
}


bool iAnimatorScript::RenderSet(iScript *self, short, int num, const int *v, int index)
{
	VECTOR_HELPER_ASSERT(curControl->GetNumberOfViewModules());

	if(index != -1)
	{
		self->GetErrorStatus()->Set("This statement does not accept a single-component assignment.");
		return false;
	}

	curScript->SetInRender(true);

	int i;
	for(i=0; self->GetErrorStatus()->NoError() && !self->CheckAbort(i,num,1) && i<num; i++) if(v[i]>0 && v[i]<=curControl->GetNumberOfViewModules() && v[i]!=1+curControl->GetCurrentViewModuleIndex())
	{
		self->GetErrorStatus()->Monitor(curControl->GetViewModule(v[i]-1)->GetAnimator()->GetErrorStatus(),true);
		curControl->GetViewModule(v[i]-1)->GetAnimator()->RenderImages(false);
	}
	for(i=0; self->GetErrorStatus()->NoError() && !self->CheckAbort(i,num,1) && i<num; i++) if(v[i] == 1+curControl->GetCurrentViewModuleIndex())
	{
		self->GetErrorStatus()->Monitor(curAnimator->GetErrorStatus(),true);
		if(!curAnimator->RenderImages()) self->GetErrorStatus()->Set("Animation finished.",-1);
		break;
	}

	curScript->SetInRender(false);

	return self->GetErrorStatus()->NoError();
}


//
// Embed control script
//
bool iAnimatorScript::ExecuteControlScript(iScript *self, const iString& filename)
{
	if(filename.IsEmpty())
	{
		self->GetErrorStatus()->Set("A name of a file with the Control Script must be specified.");
		return false;
	}

	iString fname(filename);
	iDirectory::ExpandFileName(fname,curControl->GetShell()->GetEnvironment(Environment::Base));

	iFile f(fname);
	if(!f.Open(iFile::_Read,iFile::_Text))
	{
		self->GetErrorStatus()->Set("Control script file is not found.");
		return false;
	}

	iString st, tmp;
	while(f.ReadLine(tmp)) st += tmp + "\n";
	if(st.IsEmpty())
	{
		self->GetErrorStatus()->Set("Control script file is empty or unreadable.");
		return false;
	}

	f.Close();

	self->GetErrorStatus()->Monitor(curScript->mEmbeddedControlScript->GetErrorStatus(),true,"Control script: ");
	curScript->mEmbeddedControlScript->SetText(st);
	curScript->mEmbeddedControlScript->Execute(true); // do not reset the script!!!

	return self->GetErrorStatus()->NoError();
}


bool iAnimatorScript::EmbedControlScript(iScript *self, const iString &value)
{
	if(curScript->mEmbeddedControlScript == 0)
	{
		self->GetErrorStatus()->Set("This script does not support embedding.");
		return false;
	}

	//
	//  Replace %% with \n
	//
	iString text(value);
	text.Replace("%%","\n");

	self->GetErrorStatus()->Monitor(curScript->mEmbeddedControlScript->GetErrorStatus(),true,"Control script: ");
	curScript->mEmbeddedControlScript->SetText(text);
	curScript->mEmbeddedControlScript->Execute(true); // do not reset the script!!!

	return self->GetErrorStatus()->NoError();
}



//
//  Load Sets
//
bool iAnimatorScript::Load(iScript *self, short, int value)
{
	self->GetErrorStatus()->Monitor(curReader->GetErrorStatus(),true,"Control script: ");
	curReader->LoadRecord(value,0,curAnimator->GetDebugFlag()>1);
	if(self->GetErrorStatus()->NoError()) curAnimator->ResetCurrentFile();
	return self->GetErrorStatus()->NoError();
}

