/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iobjectfactory.h"


//
//  This file is documented only partially.
//
#include "iabstractextension.h"
#include "ibasicdatasubjects.h"
#include "icrosssectionviewsubject.h"
#include "idatareader.h"
#include "iedition.h"
#include "iparticlesviewsubject.h"
#include "ipicker.h"
#include "ipointer.h"
#include "isurfaceviewsubject.h"
#include "itensorfieldviewsubject.h"
#include "ivectorfieldviewsubject.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "ivolumeviewsubject.h"


//
//  Templates
//
#include "iarraytemplate.h"


//
//  Helper class
//
class iAbstractExtensionPointer : public iPointer::Ordered<iAbstractExtension>
{

public:

	iAbstractExtensionPointer(iAbstractExtension *ptr = 0) : iPointer::Ordered<iAbstractExtension>(ptr){}
};


//
//  Main class
//
iArray<iAbstractExtensionPointer>& iObjectFactory::Extensions()
{
	static iOrderedArray<iAbstractExtensionPointer> ext;
	return ext;
}


void iObjectFactory::AttachExtension(iAbstractExtension *ext)
{
	if(ext != 0) Extensions().Add(ext);
}


//
// **************************************************************************
//
//  IMPORTANT: Factory methods do not need to be checked for null pointers
//             since they are only called from within New() functions
//
// **************************************************************************
//
iViewObject* iObjectFactory::CreateViewObject(iViewObjectParent *parent)
{
	iViewModule *vm(parent->GetViewModule());
	const iObjectType &type(parent->GetObjectType());
	
	//
	//  Particles are special - they are not single-type
	//
	iViewObject *obj = new iViewObject(parent,false,!type.IsMatch(iParticlesViewSubject::Type()));
	if(obj == 0) return 0;

	int i, j;
	iViewSubject *sub;
	iArray<iAbstractExtensionPointer> &ext(Extensions());

	//
	//  View objects
	//
	if(type.IsMatch(iCrossSectionViewSubject::Type()))
	{
		obj->AddSubject(new iCrossSectionViewSubject(vm,iUniformScalarsDataSubject::DataType(),"Cross Section"));
		for(i=0; i<ext.Size(); i++)
		{
			j = 0;
			while((sub = ext[i]->CreateCrossSectionSubject(vm,j++)) != 0)
			{
				obj->AddSubject(sub);
			}
		}
	}

	if(type.IsMatch(iParticlesViewSubject::Type()))
	{
		obj->AddSubject(new iParticlesViewSubject(vm,iBasicParticlesDataSubject::DataType(),"Particles"));
		for(i=0; i<ext.Size(); i++)
		{
			j = 0;
			while((sub = ext[i]->CreateParticlesSubject(vm,j++)) != 0)
			{
				obj->AddSubject(sub);
			}
		}
	}

	if(type.IsMatch(iSurfaceViewSubject::Type()))
	{
		obj->AddSubject(new iSurfaceViewSubject(vm,iUniformScalarsDataSubject::DataType(),"Surface"));
		for(i=0; i<ext.Size(); i++)
		{
			j = 0;
			while((sub = ext[i]->CreateSurfaceSubject(vm,j++)) != 0)
			{
				obj->AddSubject(sub);
			}
		}
	}

	if(type.IsMatch(iTensorFieldViewSubject::Type()))
	{
		obj->AddSubject(new iTensorFieldViewSubject(vm,iUniformTensorsDataSubject::DataType(),iUniformScalarsDataSubject::DataType(),"Tensor Field"));
		for(i=0; i<ext.Size(); i++)
		{
			j = 0;
			while((sub = ext[i]->CreateTensorFieldSubject(vm,j++)) != 0)
			{
				obj->AddSubject(sub);
			}
		}
	}

	if(type.IsMatch(iVectorFieldViewSubject::Type()))
	{
		obj->AddSubject(new iVectorFieldViewSubject(vm,iUniformVectorsDataSubject::DataType(),iUniformScalarsDataSubject::DataType(),"Vector Field"));
		for(i=0; i<ext.Size(); i++)
		{
			j = 0;
			while((sub = ext[i]->CreateVectorFieldSubject(vm,j++)) != 0)
			{
				obj->AddSubject(sub);
			}
		}
	}

	if(type.IsMatch(iVolumeViewSubject::Type()))
	{
		obj->AddSubject(new iVolumeViewSubject(vm,iUniformScalarsDataSubject::DataType(),"Volume"));
		for(i=0; i<ext.Size(); i++)
		{
			j = 0;
			while((sub = ext[i]->CreateVolumeSubject(vm,j++)) != 0)
			{
				obj->AddSubject(sub);
			}
		}
	}

	for(i=0; i<ext.Size(); i++)
	{
		j = 0;
		while((sub = ext[i]->CreateSpecialSubject(vm,j++,type)) != 0)
		{
			obj->AddSubject(sub);
		}
	}

	obj->Verify();

	iEdition::ApplySettings(obj,type);

	return obj;
}


//
//  These functions install extensions on various extendable classes.
//
void iObjectFactory::InstallExtensions(iDataReader *reader)
{
	int i;
	iArray<iAbstractExtensionPointer> &ext(Extensions());

	iDataReaderExtension *s;
	for(i=0; i<ext.Size(); i++)
	{
		s = ext[i]->CreateDataReaderExtension(reader);
		if(s != 0) reader->InstallExtension(s);
	}
}


void iObjectFactory::InstallExtensions(iPicker *picker)
{
	int i;
	iArray<iAbstractExtensionPointer> &ext(Extensions());

	iPickerExtension *s;
	for(i=0; i<ext.Size(); i++)
	{
		s = ext[i]->CreatePickerExtension(picker);
		if(s != 0) picker->InstallExtension(s);
	}
}


void iObjectFactory::InstallExtensions(iViewModule *vm)
{
	int i;
	iArray<iAbstractExtensionPointer> &ext(Extensions());

	iViewModuleExtension *s;
	for(i=0; i<ext.Size(); i++)
	{
		s = ext[i]->CreateViewModuleExtension(vm);
		if(s != 0) vm->InstallExtension(s);
	}
}

