/*
 * i8kctl.c -- Utility to access SMM BIOS on Dell Inspiron laptops
 *
 * Copyright (C) 2001  Massimo Dal Zotto <dz@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include "i8k.h"

#define PROG_VERSION "v1.5 10/11/2001"

static int i8k_fd;

char *
i8k_get_bios_version()
{
    char bios_version[4];
    int args[1];
    int rc;

    if ((rc=ioctl(i8k_fd, I8K_BIOS_VERSION, &args)) < 0) {
	return NULL;
    }

    bios_version[0] = (args[0] >> 16) & 0xff;
    bios_version[1] = (args[0] >>  8) & 0xff;
    bios_version[2] = (args[0]      ) & 0xff;
    bios_version[3] = '\0';

    return strdup(bios_version);
}

char *
i8k_get_machine_id()
{
    char args[16];
    int rc;

    if ((rc=ioctl(i8k_fd, I8K_MACHINE_ID, &args)) < 0) {
	return NULL;
    }

    return strdup(args);
}

int
i8k_set_fan(int fan, int speed)
{
    int args[2];
    int rc;

    args[0] = fan;
    args[1] = speed;
    if ((rc=ioctl(i8k_fd, I8K_SET_FAN, &args)) < 0) {
	return rc;
    }

    return args[0];
}

int
i8k_get_fan_status(int fan)
{
    int args[1];
    int rc;

    args[0] = fan;
    if ((rc=ioctl(i8k_fd, I8K_GET_FAN, &args)) < 0) {
	return rc;
    }

    return args[0];
}

int
i8k_get_fan_speed(int fan)
{
    int args[1];
    int rc;

    args[0] = fan;
    if ((rc=ioctl(i8k_fd, I8K_GET_SPEED, &args)) < 0) {
	return rc;
    }

    return args[0];
}

int
i8k_get_cpu_temp()
{
    int args[1];
    int rc;

    if ((rc=ioctl(i8k_fd, I8K_GET_TEMP, &args)) < 0) {
	return rc;
    }

    return args[0];
}

int
i8k_get_power_status()
{
    int args[1];
    int rc;

    if ((rc=ioctl(i8k_fd, I8K_POWER_STATUS, &args)) < 0) {
	return rc;
    }

    return args[0];
}

int
i8k_get_fn_status()
{
    int args[1];
    int rc;

    if ((rc=ioctl(i8k_fd, I8K_FN_STATUS, &args)) < 0) {
	return rc;
    }

    return args[0];
}

int
fan(int argc, char **argv)
{
    int left, right;

    if ((argc > 1) && isdigit(argv[1][0])) {
	left = i8k_set_fan(I8K_FAN_LEFT, atoi(argv[1]));
    } else {
	left = i8k_get_fan_status(I8K_FAN_LEFT);
    }

    if ((argc > 2) && isdigit(argv[2][0])) {
	right = i8k_set_fan(I8K_FAN_RIGHT, atoi(argv[2]));
    } else {
	right = i8k_get_fan_status(I8K_FAN_RIGHT);
    }

    printf("%d %d\n", left, right);
    return 0;
}

int
fan_speed(int argc, char **argv)
{
    int left, right;

    left = i8k_get_fan_speed(I8K_FAN_LEFT);
    right = i8k_get_fan_speed(I8K_FAN_RIGHT);

    printf("%d %d\n", left, right);
    return 0;
}

int
bios_version()
{
    char *version;

    if ((version=i8k_get_bios_version()) == NULL) {
	version = "?";
    }

    printf("%s\n", version);
    return 0;
}

int
machine_id()
{
    char *machine_id;

    if ((machine_id=i8k_get_machine_id()) == NULL) {
	machine_id = "?";
    }

    printf("%s\n", machine_id);
    return 0;
}

int
cpu_temperature()
{
    printf("%d\n", i8k_get_cpu_temp());
    return 0;
}

int
ac_status()
{
    printf("%d\n", i8k_get_power_status());
    return 0;
}

int
fn_key()
{
    printf("%d\n", i8k_get_fn_status());
    return 0;
}

int
status()
{
    int fn_key, cpu_temp, ac_power;
    int left_fan, right_fan, left_speed, right_speed;
    char *bios_version, *bios_machine_id;

    bios_version    = i8k_get_bios_version();
    bios_machine_id = i8k_get_machine_id();
    cpu_temp        = i8k_get_cpu_temp();
    left_fan        = i8k_get_fan_status(I8K_FAN_LEFT);
    right_fan       = i8k_get_fan_status(I8K_FAN_RIGHT);
    left_speed      = i8k_get_fan_speed(I8K_FAN_LEFT);
    right_speed     = i8k_get_fan_speed(I8K_FAN_RIGHT);
    ac_power        = i8k_get_power_status();
    fn_key          = i8k_get_fn_status();

    /*
     * Info:
     *
     * 1)  Proc format version (this will change if format changes)
     * 2)  BIOS version
     * 3)  BIOS machine ID
     * 4)  Cpu temperature
     * 5)  Left fan status
     * 6)  Right fan status
     * 7)  Left fan speed
     * 8)  Right fan speed
     * 9)  AC power
     * 10) Fn Key status
     */
    printf("%s %s %s %d %d %d %d %d %d %d\n",
	   I8K_PROC_FMT,
	   bios_version,
	   bios_machine_id,
	   cpu_temp,
	   left_fan,
	   right_fan,
	   left_speed,
	   right_speed,
	   ac_power,
	   fn_key);

    return 0;
}

void
usage(char *progname)
{
    printf("Usage: %s "
	   "[fan [<l> <r>] | speed | version | bios | id | temp | ac | fn]\n",
	   progname);
}

int
main(int argc, char **argv)
{
    char *progname;

    if ((progname=strrchr(argv[0],'/'))) {
	progname++;
    } else {
	progname = argv[0];
    }

    if (argc >= 2) {
	if ((strcmp(argv[1],"-v")==0) || (strcmp(argv[1],"--version")==0)) {
	    argc--; argv++;
	    printf("i8kctl %s\n", PROG_VERSION);
	}
    }
    if (argc >= 2) {
	if ((strcmp(argv[1],"-h")==0) || (strcmp(argv[1],"--help")==0)) {
	    usage(progname);
	    exit(0);
	}
    }

    i8k_fd = open(I8K_PROC, O_RDONLY);
    if (i8k_fd < 0) {
	perror("can't open " I8K_PROC);
	exit(1);
    }

    /* If called as i8kfan emulate the old program */
    if (strcmp(progname,"i8kfan")==0) {
	return fan(argc,argv);
    }

    /* No args, print the i8k status like /proc/i8k */
    if (argc < 2) {
	return status();
    }

    if (strcmp(argv[1],"fan")==0) {
	argc--; argv++;
	return fan(argc,argv);
    }
    if (strcmp(argv[1],"version")==0) {
	printf("%s\n", I8K_PROC_FMT);
	return 0;
    }
    if (strcmp(argv[1],"speed")==0) {
	return fan_speed(argc,argv);
    }
    if (strcmp(argv[1],"bios")==0) {
	return bios_version();
    }
    if (strcmp(argv[1],"id")==0) {
	return machine_id();
    }
    if (strcmp(argv[1],"temp")==0) {
	return cpu_temperature();
    }
    if (strcmp(argv[1],"ac")==0) {
	return ac_status();
    }
    if (strcmp(argv[1],"fn")==0) {
	return fn_key();
    }

    fprintf(stderr,"invalid command: %s\n", argv[1]);
    return 1;
}

/* end of file */
