/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// use-proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "use-proc.h"

#include "defs.h"


uint UseProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *name;
    const char *value;
    STREAM incfile;
    BOOL noexpand;
    int result;
    uint type;
    VARSTORE varstore;
    VARSTORE *topVarstore;
    uint ctr;
    uint flag;

    /* must declare at least 1 attribute, the macro name */
    if (htmlMarkup->attribCount == 0)
    {
        HtpMsg(MSG_ERROR, task->infile, "macro declaration not complete");
        return MARKUP_ERROR;
    }

    /* a variable reference should NOT include a new declaration */
    if (htmlMarkup->attrib[0].value != NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "improper USE syntax");
        return MARKUP_ERROR;
    }

    /* check if the noexpand flag is present */
    noexpand = FALSE;
    if(htmlMarkup->attribCount == 2
       && stricmp(htmlMarkup->attrib[1].name, "NOEXPAND") == 0
       && htmlMarkup->attrib[1].value == NULL)
    {
        noexpand = TRUE;
    }

    name = htmlMarkup->attrib[0].name;

    /* verify the macro exists */
    if(VariableExists(task->varstore, name) == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile, "macro %s has not been declared", name);
        return MARKUP_ERROR;
    }

    /* get the type of macro */
    type = GetVariableType(task->varstore, name);

    if(type == VAR_TYPE_INTERNAL)
    {
        /* oof ... the user picked a variable name we use internally */
        /* !! a fix is to use both type and name as a key to get variable */
        /* out of hash, and therefore the name is not the only identifier */
        /* this will have to wait for later */
        HtpMsg(MSG_ERROR, task->infile,
            "reserved variable name \"%s\" used ... please use different name in HTP file",
            name);
        return MARKUP_ERROR;
    }

    if(type == VAR_TYPE_DEF_MACRO)
    {
        /* nope */
        HtpMsg(MSG_ERROR, task->infile,
            "illegal to dereference a DEF macro with USE");
        return MARKUP_ERROR;
    }

    assert((type == VAR_TYPE_SET_MACRO) || (type == VAR_TYPE_BLOCK_MACRO));

    /* if more than one parameter is on the USE tag, then assume they are */
    /* local variables for the macro */
    if(!noexpand && htmlMarkup->attribCount > 1)
    {
        if(type == VAR_TYPE_SET_MACRO)
        {
            /* nope, not yet */
            HtpMsg(MSG_ERROR, task->infile, "macro parameters can only be used for BLOCK macros");
            return MARKUP_ERROR;
        }

        /* create a local variable store */
        if(InitializeVariableStore(&varstore) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to initialize local context for macro");
            return MARKUP_ERROR;
        }

        /* add each additional parameter to the local varstore */
        for(ctr = 1; ctr < htmlMarkup->attribCount; ctr++)
        {
            flag = (htmlMarkup->attrib[ctr].quoted == TRUE) ? VAR_FLAG_QUOTED
                : VAR_FLAG_NONE;
            /* No need to duplicate values, as varstore is
             * destroyed before this function returns.  
             */
            if(StoreVariable(&varstore, htmlMarkup->attrib[ctr].name,
                htmlMarkup->attrib[ctr].value, VAR_TYPE_SET_MACRO, flag,
                NULL, NULL) == FALSE)
            {
                HtpMsg(MSG_ERROR, task->infile, "unable to add variable to block's local context");
                DestroyVariableStore(&varstore);
                return MARKUP_ERROR;
            }
        }

        /* make this variable store the topmost context */
        PushVariableStoreContext(task->varstore, &varstore);
        topVarstore = &varstore;
    }
    else
    {
        topVarstore = task->varstore;
    }

    if(type == VAR_TYPE_SET_MACRO)
    {
        /* get the value of the macro */
        value = GetVariableValue(task->varstore, name);

        /* if NULL, then the macro was declared with no value, this is okay, */
        /* just don't do anything */
        if(value == NULL)
        {
            return DISCARD_MARKUP;
        }

        HtpMsg(MSG_INFO, task->infile, "dereferencing macro \"%s\"", name);

        PutStreamString(task->outfile, value);
        return DISCARD_MARKUP;
    }
    else /* type == VAR_TYPE_BLOCK_MACRO */
    {
        /* !! magic number */
        TASK newTask;
        STREAM *blockFile;

        /* get the blockFile of the macro */
        blockFile = (STREAM*) GetVariableValue(task->varstore, name);

        /* if NULL, big-time error */
        assert (blockFile != NULL);

        HtpMsg(MSG_INFO, task->infile, "dereferencing block macro \"%s\"%s", name, noexpand ? " with noexpand" : "");
        if (noexpand)
        {
            PutStreamString(task->outfile, blockFile->u.buffer.buffer);
            return DISCARD_MARKUP;
        }

        result = CreateBufferReader(&incfile, blockFile);
        assert(result);

        /* build a new task structure */
        newTask.infile = &incfile;
        newTask.outfile = task->outfile;
        newTask.sourceFilename = task->sourceFilename;

        /* re-use current variable store if no local variable store was */
        /* allocated, otherwise use the new one */
        newTask.varstore = topVarstore;

        /* process the new input file */
        result = ProcessTask(&newTask);
        if (!result) {
            /* Error message was already spitted out.  However, we
             * should give a hint where the meta-tag was called.  
             */
            HtpMsg(MSG_ERROR, task->infile, "... in block macro \"%s\"", name);
        }

        /* remove the new context (and make sure it is, in fact, the block's */
        /* context) */
        if(topVarstore == &varstore)
        {
            assert(PeekVariableStoreContext(topVarstore) == topVarstore);
            PopVariableStoreContext(topVarstore);
            DestroyVariableStore(&varstore);
        }

        CloseStream(&incfile);

        /* if the new file did not process, return an error, otherwise discard */
        /* the markup */
        return (result == TRUE) ? DISCARD_MARKUP : MARKUP_ERROR;
    }
}   


