/*
Gwenview - A simple image viewer for KDE
Copyright (C) 2000-2002 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Qt includes
#include <qpopupmenu.h>

// KDE includes
#include <kaction.h>
#include <kdebug.h>
#include <klocale.h>
#include <kstdaction.h>

// Our includes
#include "filednddetailview.h"
#include "fileoperation.h"
#include "filethumbnailview.h"
#include "imagefilereader.h"

#include "fileview.moc"

static const char* CONFIG_START_WITH_THUMBNAILS="start with thumbnails";


FileView::FileView(QWidget* parent)
: QWidgetStack(parent), mPopupMenu(0L)
{
// Actions
	mSelectFirst=new KAction(i18n("&First"),"start",Key_Home,
		this,SLOT(slotSelectFirst()), this);

	mSelectLast=new KAction(i18n("&Last"),"finish",Key_End,
		this,SLOT(slotSelectLast()), this);

	mSelectPrevious=KStdAction::back(this, SLOT(slotSelectPrevious()),this );
	mSelectPrevious->setAccel(Key_Backspace);

	mSelectNext=KStdAction::forward(this, SLOT(slotSelectNext()),this );
	mSelectNext->setAccel(Key_Space);

	mNoThumbnails=new KRadioAction(i18n("Details"),"view_detailed",0,this,SLOT(updateThumbnailSize()),this);
	mNoThumbnails->setExclusiveGroup("thumbnails");
	mSmallThumbnails=new KRadioAction(i18n("Small thumbnails"),"smallthumbnails",0,this,SLOT(updateThumbnailSize()),this);
	mSmallThumbnails->setExclusiveGroup("thumbnails");
	mMedThumbnails=new KRadioAction(i18n("Medium thumbnails"),"medthumbnails",0,this,SLOT(updateThumbnailSize()),this);
	mMedThumbnails->setExclusiveGroup("thumbnails");
	mLargeThumbnails=new KRadioAction(i18n("Large thumbnails"),"largethumbnails",0,this,SLOT(updateThumbnailSize()),this);
	mLargeThumbnails->setExclusiveGroup("thumbnails");

// File reader
	mFileReader=new ImageFileReader();

	connect(mFileReader,SIGNAL(newItems(const KFileItemList&)),
		this,SLOT(newItems(const KFileItemList&)) );
	connect(mFileReader,SIGNAL(clear()),
		this,SLOT(clear()) );
	connect(mFileReader,SIGNAL(deleteItem(KFileItem*)),
		this,SLOT(removeItem(KFileItem*)) );
	connect(mFileReader,SIGNAL(started(const QString&)),
		this,SLOT(fileReaderStarted()) );
	connect(mFileReader,SIGNAL(completed()),
		this,SLOT(fileReaderCompleted()) );
	connect(mFileReader,SIGNAL(canceled()),
		this,SLOT(fileReaderCanceled()) );

// Propagate completed and canceled signals
	connect(mFileReader,SIGNAL(completed()),
		this,SIGNAL(completed()) );
	connect(mFileReader,SIGNAL(canceled()),
		this,SIGNAL(canceled()) );

// File detail widget
	mFileDetailView=new FileDnDDetailView(this,"filedetailview");
	addWidget(mFileDetailView,0);

	connect(mFileDetailView,SIGNAL(clicked(QListViewItem*)),
		this,SLOT(detailChanged(QListViewItem*)) );
	connect(mFileDetailView,SIGNAL(returnPressed(QListViewItem*)),
		this,SLOT(detailChanged(QListViewItem*)) );
	connect(mFileDetailView,SIGNAL(rightButtonClicked(QListViewItem*,const QPoint&,int)),
		this,SLOT(detailRightButtonClicked(QListViewItem*,const QPoint&,int)) );

// Thumbnail widget
	mFileThumbnailView=new FileThumbnailView(this);
	addWidget(mFileThumbnailView,1);

	connect(mFileThumbnailView,SIGNAL(clicked(QIconViewItem*)),
		this,SLOT(thumbnailChanged(QIconViewItem*)) );
	connect(mFileThumbnailView,SIGNAL(returnPressed(QIconViewItem*)),
		this,SLOT(thumbnailChanged(QIconViewItem*)) );
	connect(mFileThumbnailView,SIGNAL(rightButtonClicked(QIconViewItem*,const QPoint&)),
		this,SLOT(thumbnailRightButtonClicked(QIconViewItem*,const QPoint&)) );

// Propagate signals
	connect(mFileThumbnailView,SIGNAL(updateStarted(int)),
		this,SIGNAL(updateStarted(int)) );
	connect(mFileThumbnailView,SIGNAL(updateEnded()),
		this,SIGNAL(updateEnded()) );
	connect(mFileThumbnailView,SIGNAL(updatedOneThumbnail()),
		this,SIGNAL(updatedOneThumbnail()) );

}


FileView::~FileView() {
	delete mFileReader;
}


void FileView::plugActionsToAccel(KAccel* accel) {
	mSelectFirst->plugAccel(accel);
	mSelectLast->plugAccel(accel);
	mSelectPrevious->plugAccel(accel);
	mSelectNext->plugAccel(accel);
	mNoThumbnails->plugAccel(accel);
	mSmallThumbnails->plugAccel(accel);
	mMedThumbnails->plugAccel(accel);
	mLargeThumbnails->plugAccel(accel);
}


void FileView::installRBPopup(QPopupMenu* menu) {
	mPopupMenu=menu;
}


//-Public slots----------------------------------------------------------
void FileView::setFolderAndFile(QString folder,QString filename) {
	mFilenameToSelect=filename;
	mFolder=folder;
	mFileThumbnailView->stopThumbnailUpdate();

	mFileReader->openURL(KURL(mFolder),true);
	updateActions();
}


void FileView::cancel() {
	if (!mFileReader->isFinished()) {
		mFileReader->stop();
		return;
	}
	if (mMode==Thumbnail) {
		mFileThumbnailView->stopThumbnailUpdate();
	}
}


void FileView::selectFilename(QString filename) {
	KFileViewItem* item;
	currentFileView()->setCurrentItem(filename);
	item=currentFileView()->selectedItems()->getFirst();
	if (item) {
		currentFileView()->ensureItemVisible(item);
	}
	updateActions();
	emitFilenameChanged();
}


void FileView::slotSelectFirst() {
	currentBrowsableFileView()->selectFirst();
	emitFilenameChanged();
}


void FileView::slotSelectLast() {
	currentBrowsableFileView()->selectLast();
	emitFilenameChanged();
}


void FileView::slotSelectPrevious() {
	currentBrowsableFileView()->selectPrevious();
	emitFilenameChanged();
}


void FileView::slotSelectNext() {
	currentBrowsableFileView()->selectNext();
	emitFilenameChanged();
}


//-Private slots---------------------------------------------------------
void FileView::detailChanged(QListViewItem* item) {
	if (!item) return;
	emitFilenameChanged();
}


void FileView::thumbnailChanged(QIconViewItem* item) {
	if (!item) return;
	emitFilenameChanged();
}


void FileView::detailRightButtonClicked(QListViewItem* item,const QPoint& pos,int) {
	if (!item) return;
	emitFilenameChanged();
	if (mPopupMenu)
		mPopupMenu->popup(pos);
}


void FileView::thumbnailRightButtonClicked(QIconViewItem* item,const QPoint& pos) {
	if (!item) return;
	emitFilenameChanged();
	mPopupMenu->popup(pos);
}


void FileView::updateThumbnailSize() {
	if (mNoThumbnails->isChecked()) {
		setMode(FileView::FileList);
		return;
	} else {
		if (mSmallThumbnails->isChecked()) {
			mFileThumbnailView->setThumbnailSize(ThumbnailSize::Small);
		} else if (mMedThumbnails->isChecked()) {
			mFileThumbnailView->setThumbnailSize(ThumbnailSize::Med);
		} else {
			mFileThumbnailView->setThumbnailSize(ThumbnailSize::Large);
		}
		setMode(FileView::Thumbnail);
	}
}


//-Properties------------------------------------------------------------
QString FileView::filename() const {
	return currentBrowsableFileView()->currentItemText();
}


KFileView* FileView::currentFileView() const {
	if (mMode==FileList) {
		return mFileDetailView;
	} else {
		return mFileThumbnailView;
	}
}


/**
 * This is the same as currentFileView() except that it does not return
 * the same interface, thus avoid type casting all over the place
 */
BrowsableFileView* FileView::currentBrowsableFileView() const {
	if (mMode==FileList) {
		return mFileDetailView;
	} else {
		return mFileThumbnailView;
	}
}


bool FileView::currentIsFirst() const {
	return currentBrowsableFileView()->currentIsFirst();
}


bool FileView::currentIsLast() const {
	return currentBrowsableFileView()->currentIsLast();
}


uint FileView::fileCount() const {
	return currentFileView()->items()->count();
}

void FileView::setMode(FileView::Mode mode) {
	mFilenameToSelect=filename();
	mMode=mode;

	if (mMode==FileList) {
		mFileThumbnailView->stopThumbnailUpdate();
		raiseWidget(mFileDetailView);
	} else {
		raiseWidget(mFileThumbnailView);
	}

	mFileReader->listDirectory();
}



//-File reader slots-----------------------------------------------------
void FileView::removeItem(KFileItem* item) {
	currentFileView()->removeItem((KFileViewItem*)item);
}


void FileView::newItems(const KFileItemList& items) {
	mThumbnailsNeedUpdate=true;
	currentFileView()->addItemList(items);
}


void FileView::clear() {
	currentFileView()->clear();
}


void FileView::fileReaderStarted() {
	mThumbnailsNeedUpdate=false;
	if (mFilenameToSelect.isEmpty()) {
		mFilenameToSelect=filename();
	}
}


void FileView::fileReaderCompleted() {
	if (mFilenameToSelect.isEmpty()) {
		slotSelectFirst();
	} else {
		selectFilename(mFilenameToSelect);
	}

	if (mMode==Thumbnail && mThumbnailsNeedUpdate) {
		mFileThumbnailView->startThumbnailUpdate();
	}
}


void FileView::fileReaderCanceled() {
	if (mMode==Thumbnail) {
		mFileThumbnailView->stopThumbnailUpdate();
	}
	if (mFilenameToSelect.isEmpty()) {
		slotSelectFirst();
	} else {
		selectFilename(mFilenameToSelect);
	}
}


void FileView::stopFileReader() {
	mFileReader->stop();
}


//-Private---------------------------------------------------------------
void FileView::updateActions() {
	if (fileCount()==0) {
		mSelectFirst->setEnabled(false);
		mSelectPrevious->setEnabled(false);	
		mSelectNext->setEnabled(false);
		mSelectLast->setEnabled(false);	
		mNoThumbnails->setEnabled(false);	
		mSmallThumbnails->setEnabled(false);	
		mMedThumbnails->setEnabled(false);	
		mLargeThumbnails->setEnabled(false);	
		return;
	}

	mNoThumbnails->setEnabled(true);	
	mSmallThumbnails->setEnabled(true);	
	mMedThumbnails->setEnabled(true);	
	mLargeThumbnails->setEnabled(true);	

	bool isFirst=currentIsFirst();
	bool isLast=currentIsLast();
	
	mSelectFirst->setEnabled(!isFirst);
	mSelectPrevious->setEnabled(!isFirst);	
	mSelectNext->setEnabled(!isLast);
	mSelectLast->setEnabled(!isLast);	
}


void FileView::emitFilenameChanged() {
	emit filenameChanged(filename());
	updateActions();
}


//-File operations-------------------------------------------------------
void FileView::copyFile() {
    FileOperation::copyTo(mFolder + '/' + filename(),this);
}


void FileView::moveFile() {
	KURL url;
	url.setPath(mFolder + '/' + filename());

// Get the next item text or the previous one if there's no next item
	mNewFilenameToSelect=currentBrowsableFileView()->nextItemText();
	if (mNewFilenameToSelect.isEmpty()) {
		mNewFilenameToSelect=currentBrowsableFileView()->previousItemText();
	}

// move the file, slotSelectNewFilename will be called on success
	FileOperation::moveTo(url,this,this,SLOT(slotSelectNewFilename()) );
}


void FileView::deleteFile() {
	KURL url;
	url.setPath(mFolder + '/' + filename());

// Get the next item text or the previous one if there's no next item
	mNewFilenameToSelect=currentBrowsableFileView()->nextItemText();
	if (mNewFilenameToSelect.isEmpty()) {
		mNewFilenameToSelect=currentBrowsableFileView()->previousItemText();
	}

// Delete the file, slotSelectNewFilename will be called on success
	FileOperation::del(url,this,this,SLOT(slotSelectNewFilename()) );
}


void FileView::slotSelectNewFilename() {
	mFilenameToSelect=mNewFilenameToSelect;
}


void FileView::renameFile() {
	FileOperation::rename(mFolder + '/' + filename(),this,this,SLOT(slotRenamed(const QString&)) );
}


void FileView::slotRenamed(const QString& newFilename) {
	mFilenameToSelect=newFilename;

// Refresh the view and select the renamed item
	mFileReader->setURLDirty(true);
	mFileReader->listDirectory();
}


//-Configurable implementation----------------------------------------------
void FileView::readConfig(KConfig* config,const QString& group) {
	mFileThumbnailView->readConfig(config,group);

	config->setGroup(group);
	bool startWithThumbnails=config->readBoolEntry(CONFIG_START_WITH_THUMBNAILS,false);
	setMode(startWithThumbnails?Thumbnail:FileList);

	if (startWithThumbnails) {
		switch (mFileThumbnailView->thumbnailSize()) {
		case ThumbnailSize::Small:
			mSmallThumbnails->setChecked(true);
			break;
		case ThumbnailSize::Med:
			mMedThumbnails->setChecked(true);
			break;
		case ThumbnailSize::Large:
			mLargeThumbnails->setChecked(true);
			break;
		}
	} else {
		mNoThumbnails->setChecked(true);
	}
}


void FileView::writeConfig(KConfig* config,const QString& group) const {
	mFileThumbnailView->writeConfig(config,group);

	config->setGroup(group);

	config->writeEntry(CONFIG_START_WITH_THUMBNAILS,!mNoThumbnails->isChecked());
}
