// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The central queues for pipelined data processing
// ****************************************************************************

#include "QSemaphore"
#include "QMutex"

#include "util.h"
#include "fifo.h"

class t_FifoStdLocal
{
   public:
      QQueue<t_pFifoBlock>  Queue;
      int                   MaxBlocks;
      QSemaphore          *pSemEmptyBlocks;
      QSemaphore          *pSemUsedBlocks;
      QMutex              *pMutexQueue;
};


static QMutex  MutexGlobal;
static quint64 Allocs   =0;
static quint64 Frees    =0;
static qint64  Allocated=0;  // let's use signed, so we can detect underruns


// We use ANSI C malloc/free for the FIFO blocks, as we want to be sure that it's fast


APIRET t_Fifo::Create (t_pFifoBlock &pBlock, unsigned int BufferSize)
{
   int TotalSize;

   TotalSize = BufferSize + sizeof(t_FifoBlock);
   pBlock = (t_pFifoBlock) UTIL_MEM_ALLOC (TotalSize);
   if (pBlock == NULL)
      CHK_CONST (ERROR_FIFO_MALLOC_FAILED)
//   memset (pBlock, 0xAA, sizeof(t_FifoBlock) + BufferSize);
   pBlock->MagicStart         = FIFO_MAGIC_START;
   pBlock->BufferSize         = BufferSize;
   pBlock->DataSize           = 0;
   pBlock->LastBlock          = false;

   pBlock->EwfPreprocessed    = false;
   pBlock->EwfDataSize        = 0;
   pBlock->EwfCompressionUsed = 0;
   pBlock->EwfChunkCRC        = 0;
   pBlock->EwfWriteCRC        = 0;

   pBlock->pAaffPreprocess    = NULL;

   pBlock->Nr                 = FIFO_NR_NOTSET;

   //lint -save e826 suspicious ptr to ptr conversion
   FIFO_SET_MAGIC_END (pBlock)
   //lint -restore

   MutexGlobal.lock();
   Allocs++;
   Allocated += TotalSize;
   MutexGlobal.unlock();

   return NO_ERROR;
}

unsigned int t_Fifo::GetCompressionOptimisedBufferSize (int CompressionBlockSize)
{
   return CompressionBlockSize + 4096 - sizeof(t_FifoBlock);
}

APIRET t_Fifo::CreateCompressionOptimised (t_pFifoBlock &pBlock, int CompressionBlockSize)
{
   CHK (Create (pBlock, GetCompressionOptimisedBufferSize (CompressionBlockSize)))

   return NO_ERROR;
}

//lint -save -e661 -e826 -esym(613,pBlock) Possible access of out-of-bounds pointer, suspicious ptr to ptr conversion, possible use of NULL pointer
APIRET t_Fifo::Destroy (t_pFifoBlock &pBlock)
{
   int TotalSize;

   if (pBlock == NULL)
      CHK (ERROR_FIFO_DOUBLE_FREE)

   if (pBlock->MagicStart != FIFO_MAGIC_START)
   {
      LOG_ERROR ("Broken start magic: %08X", pBlock->MagicStart);
      CHK (ERROR_FIFO_START_CORRUPTED)
   }

   if (FIFO_GET_MAGIC_END(pBlock) != FIFO_MAGIC_END)
   {
      LOG_ERROR ("Broken end magic: %08X", FIFO_GET_MAGIC_END(pBlock));
      CHK (ERROR_FIFO_END_CORRUPTED)
   }
   TotalSize = pBlock->BufferSize + sizeof(t_FifoBlock);

//   memset (pBlock, 0xBB, TotalSize);
   UTIL_MEM_FREE (pBlock);
   pBlock = NULL;

   MutexGlobal.lock();
   Frees++;
   Allocated -= TotalSize;
   if (Allocated < 0)
   {
      LOG_ERROR ("Memory underrun");
      MutexGlobal.unlock();
      CHK (ERROR_FIFO_END_CORRUPTED)
   }
   MutexGlobal.unlock();

   return NO_ERROR;
}
//lint -restore

t_FifoStd::t_FifoStd (int MaxBlocks)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_FIFO_MALLOC_FAILED  ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_FIFO_DOUBLE_FREE    ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_FIFO_EMPTY          ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_FIFO_END_CORRUPTED  ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_FIFO_START_CORRUPTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_FIFO_MEMORY_UNDERUN ))
      Initialised = true;
   }
   pOwn = new t_FifoStdLocal;
   pOwn->MaxBlocks = MaxBlocks;

   pOwn->pSemEmptyBlocks = new QSemaphore (MaxBlocks);
   pOwn->pSemUsedBlocks  = new QSemaphore (0);
   pOwn->pMutexQueue     = new QMutex     ();
}

t_FifoStd::~t_FifoStd ()
{
   t_pFifoBlock pBlock;

   while (!pOwn->Queue.isEmpty())
   {
      CHK_EXIT (t_FifoStd::Get (pBlock))
      if (pBlock)
         CHK_EXIT (Destroy (pBlock))
   }

   delete pOwn->pSemEmptyBlocks;
   delete pOwn->pSemUsedBlocks;
   delete pOwn->pMutexQueue;
   delete pOwn;
}

APIRET t_FifoStd::Insert (t_pFifoBlock pBlock)
{
   pOwn->pSemEmptyBlocks->acquire();
   pOwn->pMutexQueue->lock();
   pOwn->Queue.enqueue (pBlock);
   pOwn->pMutexQueue->unlock();
   pOwn->pSemUsedBlocks->release();

   return NO_ERROR;
}

APIRET t_FifoStd::InsertDummy (void)
{
   CHK (Insert (NULL));
   return NO_ERROR;
}

APIRET t_FifoStd::Get (t_pFifoBlock &pBlock)
{
   pOwn->pSemUsedBlocks->acquire();
   if (pOwn->Queue.isEmpty())
      CHK (ERROR_FIFO_EMPTY)
   pOwn->pMutexQueue->lock();
   pBlock = pOwn->Queue.dequeue();
   pOwn->pMutexQueue->unlock();
   pOwn->pSemEmptyBlocks->release();

   return NO_ERROR;
}

APIRET t_FifoStd::Count (int &Cnt)
{
   pOwn->pMutexQueue->lock();
   Cnt = pOwn->Queue.count();
   pOwn->pMutexQueue->unlock();
   return NO_ERROR;
}

APIRET t_FifoStd::Usage (int &Percent)
{
   int Cnt;

   Count(Cnt);
   Percent = (100LL * Cnt) / pOwn->MaxBlocks;
   return NO_ERROR;
}


// Qt provides no possibility for waking threads waiting for semaphores. This fn wakes a potential
// thread hanging in t_FifoStd::Get by inserting a NULL element into the Fifo. Threads hanging in
// t_FifoStd::Insert are woken by removing a block from the Fifo. Of course, this method is not really
// clean, as it not only wakes threads but manipulates the data in the Fifo... the threads accessing
// the Fifo must be able to cope with this.
// By the way: A non-blocking semaphore could be implemented manually by putting QSemaphore::tryAcquire
// together with a msleep(0) in to loop. But this solution turned out to be non-optimal in performance.

APIRET t_FifoStd::WakeWaitingThreads (void)
{
   t_pFifoBlock pBlock;
   bool          WouldBlock;

   WouldBlock = !pOwn->pSemUsedBlocks->tryAcquire();
   if (WouldBlock)                         // If this access would block there might be a thread waiting in "Get"
        CHK (Insert (NULL))
   else pOwn->pSemUsedBlocks->release();

   WouldBlock = !pOwn->pSemEmptyBlocks->tryAcquire();
   if (WouldBlock)                         // If this access would block there might be a thread waiting in "Insert"
   {
      CHK (Get (pBlock))
      if (pBlock)
         CHK (t_Fifo::Destroy (pBlock))
   }
   else
   {
      pOwn->pSemEmptyBlocks->release();
   }

   return NO_ERROR;
}



// ----------------------------------------------------------------------------------------
//                                       FIFO Compress In
// ----------------------------------------------------------------------------------------

class t_FifoCompressLocal   // used as well for t_FifoCompressOut
{
   public:
      t_pFifoStd *ppFifoArr;
      int           FifoArrLen;
      int           TotalMaxBlocks;    // Sum of MaxBlocks of all FIFOs
      QMutex      *pMutexQueue;
      int           NextSubFifoNr;
};

static APIRET FifoCompressInit (t_FifoCompressLocal **ppOwn, int SubFifos, int MaxBlocks)
{
   *ppOwn = new t_FifoCompressLocal;
   (*ppOwn)->TotalMaxBlocks = SubFifos * MaxBlocks;
   (*ppOwn)->FifoArrLen     = SubFifos;
   (*ppOwn)->ppFifoArr      = (t_pFifoStd *) UTIL_MEM_ALLOC ((size_t)SubFifos * sizeof(t_pFifoStd));

   for (int i=0; i<SubFifos; i++)
      (*ppOwn)->ppFifoArr[i] = new t_FifoStd (MaxBlocks);

   (*ppOwn)->NextSubFifoNr = 0;
   (*ppOwn)->pMutexQueue   = new QMutex();

   return NO_ERROR;
}

static APIRET FifoCompressDeInit (t_FifoCompressLocal *pOwn)
{
   for (int i=0; i<pOwn->FifoArrLen; i++)
      delete (pOwn->ppFifoArr[i]);

   UTIL_MEM_FREE (pOwn->ppFifoArr);
   delete pOwn->pMutexQueue;
   delete pOwn;

   return NO_ERROR;
}

t_FifoCompressIn::t_FifoCompressIn (void)
{
   pOwn=NULL;   //lint -esym(613,t_FifoCompressIn::pOwn)  Prevent lint from telling us about possible null pointers in the following code
   CHK_EXIT (ERROR_FIFO_CONSTRUCTOR_NOT_SUPPORTED)
}

t_FifoCompressIn::t_FifoCompressIn (int SubFifos, int MaxBlocks)
{
   CHK_EXIT (FifoCompressInit (&pOwn, SubFifos, MaxBlocks))
}

t_FifoCompressIn::~t_FifoCompressIn ()
{
   CHK_EXIT (FifoCompressDeInit (pOwn))
}

APIRET t_FifoCompressIn::GetSubFifo (int SubFifoNr, t_pFifoStd &pFifo)
{
   pFifo = pOwn->ppFifoArr[SubFifoNr];

   return NO_ERROR;
}

APIRET t_FifoCompressIn::Insert (t_pFifoBlock pBlock)
{
   t_pFifoStd pFifo;

   pOwn->pMutexQueue->lock();
      CHK (GetSubFifo (pOwn->NextSubFifoNr++, pFifo))
      if (pOwn->NextSubFifoNr == pOwn->FifoArrLen)
         pOwn->NextSubFifoNr = 0;
   pOwn->pMutexQueue->unlock();

   CHK (pFifo->Insert (pBlock))

   return NO_ERROR;
}

APIRET t_FifoCompressIn::InsertDummy (void)
{
   for (int i=0; i< pOwn->FifoArrLen; i++)
      CHK (Insert (NULL))

   return NO_ERROR;
}

APIRET t_FifoCompressIn::Get (t_pFifoBlock &/*pBlock*/)
{
   return ERROR_FIFO_FUNCTION_NOT_IMPLEMENTED;
}

APIRET t_FifoCompressIn::Count (int &Cnt)
{
   t_pFifoStd pFifo;
   int         SubCnt;

   Cnt = 0;
//   pOwn->pMutexQueue->lock();                           // Do not use semaphores here because of the deadlock risk.
      for (int i=0; i<pOwn->FifoArrLen; i++)              // As a consequence, the count might be slightly wrong if
      {                                                   // other threads are inserting and deleting in parallel,
         CHK (GetSubFifo (i, pFifo))                      // but that does no harm.
         CHK (pFifo->Count (SubCnt))
         Cnt += SubCnt;
      }
//   pOwn->pMutexQueue->unlock();

   return NO_ERROR;
}

APIRET t_FifoCompressIn::Usage (int &Percent)
{
   int Cnt;

   Count(Cnt);
   Percent = (100LL * Cnt) / pOwn->TotalMaxBlocks;
   Percent = std::min (Percent, 100); // In some cases, the usage could be above 100% (see remarks for t_FifoCompressIn::Count); so we limit to 100
   return NO_ERROR;
}

APIRET t_FifoCompressIn::WakeWaitingThreads (void)
{
   t_pFifoStd pFifo;

   for (int i=0; i< pOwn->FifoArrLen; i++)
   {
      CHK (GetSubFifo (i, pFifo))
      CHK (pFifo->WakeWaitingThreads())
   }

   return NO_ERROR;
}


// ----------------------------------------------------------------------------------------
//                                       FIFO Compress Out
// ----------------------------------------------------------------------------------------

t_FifoCompressOut::t_FifoCompressOut (void)
{
   pOwn = NULL;   //lint -esym(613,t_FifoCompressOut::pOwn)  Prevent lint from telling us about possible null pointers in the following code
   CHK_EXIT (ERROR_FIFO_CONSTRUCTOR_NOT_SUPPORTED)
}

t_FifoCompressOut::t_FifoCompressOut (int SubFifos, int MaxBlocks)
{
   CHK_EXIT (FifoCompressInit (&pOwn, SubFifos, MaxBlocks))
}

t_FifoCompressOut::~t_FifoCompressOut ()
{
   CHK_EXIT (FifoCompressDeInit (pOwn))
}

APIRET t_FifoCompressOut::GetSubFifo (int SubFifoNr, t_pFifoStd &pFifo)
{
   pFifo = pOwn->ppFifoArr[SubFifoNr];

   return NO_ERROR;
}

APIRET t_FifoCompressOut::Insert (t_pFifoBlock /*pBlock*/)
{
   return ERROR_FIFO_FUNCTION_NOT_IMPLEMENTED;
}

APIRET t_FifoCompressOut::InsertDummy (void)
{
   return ERROR_FIFO_FUNCTION_NOT_IMPLEMENTED;
}

APIRET t_FifoCompressOut::InsertDummy (int SubFifoNr)
{
   t_pFifoStd pFifo;

   CHK (GetSubFifo (SubFifoNr, pFifo))
   CHK (pFifo->Insert (NULL))

   return NO_ERROR;
}

APIRET t_FifoCompressOut::Get (t_pFifoBlock &pBlock)
{
   t_pFifoStd pFifo;

   pOwn->pMutexQueue->lock();
      CHK (GetSubFifo (pOwn->NextSubFifoNr++, pFifo))
      if (pOwn->NextSubFifoNr == pOwn->FifoArrLen)
         pOwn->NextSubFifoNr = 0;
   pOwn->pMutexQueue->unlock();

   CHK (pFifo->Get (pBlock));

   return NO_ERROR;
}

APIRET t_FifoCompressOut::Count (int &Cnt)
{
   t_pFifoStd pFifo;
   int         SubCnt;

   Cnt = 0;
//   pOwn->pMutexQueue->lock();
      for (int i=0; i<pOwn->FifoArrLen; i++)
      {
         CHK (GetSubFifo (i, pFifo))
         CHK (pFifo->Count (SubCnt))
         Cnt += SubCnt;
      }
//   pOwn->pMutexQueue->unlock();

   return NO_ERROR;
}

APIRET t_FifoCompressOut::Usage (int &Percent)
{
   int Cnt;

   Count(Cnt);
   Percent = (100LL * Cnt) / pOwn->TotalMaxBlocks;
   Percent = std::min (Percent, 100); // In some cases, the usage could be above 100% (see remarks for t_FifoCompressIn::Count); so we limit to 100
   return NO_ERROR;
}

APIRET t_FifoCompressOut::WakeWaitingThreads (int SubFifoNr)
{
   t_pFifoStd pFifo;

   CHK (GetSubFifo (SubFifoNr, pFifo))
   CHK (pFifo->WakeWaitingThreads())

   return NO_ERROR;
}

APIRET t_FifoCompressOut::WakeWaitingThreads (void)
{
   for (int i=0; i<pOwn->FifoArrLen; i++)
      CHK (WakeWaitingThreads (i))

   return NO_ERROR;
}

APIRET FifoGetStatistics (quint64 &AllocCalls, quint64 &FreeCalls, qint64 &AllocatedMemory)
{
   MutexGlobal.lock();
   AllocCalls      = Allocs;
   FreeCalls       = Frees;
   AllocatedMemory = Allocated;
   MutexGlobal.unlock();

   return NO_ERROR;
}

