/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <fnmatch.h>
#include <string.h>
#include <time.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-dateedit.h>
#include <libgnomeui/gnome-file-entry.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs.h>
#include "catalog.h"
#include "comments.h"
#include "dlg-search.h"
#include "gthumb-init.h"
#include "search.h"
#include "window.h"


static void dlg_search_ui (GThumbWindow *window, 
			   gchar *catalog_path, 
			   gboolean start_search);


void 
dlg_search (GtkWidget *widget, gpointer data)
{
	GThumbWindow *window = data;
	dlg_search_ui (window, NULL, FALSE);
}


void 
dlg_catalog_edit_search (GtkWidget *widget, gpointer data)
{
	GThumbWindow *window = data;
	gchar *catalog_path;

	catalog_path = catalog_list_path_from_row (window->catalog_list, GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data));

	dlg_search_ui (window, catalog_path, FALSE);
}


void 
dlg_catalog_search (GtkWidget *widget, gpointer data)
{
	GThumbWindow *window = data;
	gchar *catalog_path;

	if (GTK_CLIST (window->catalog_list->clist)->selection == NULL)
		return;

	catalog_path = catalog_list_path_from_row (window->catalog_list, GPOINTER_TO_INT (GTK_CLIST (window->catalog_list->clist)->selection->data));

	dlg_search_ui (window, catalog_path, TRUE);
}


/* -- implementation -- */


#define SEARCH_GLADE_FILE      "gthumb_search.glade"
#define SEARCH_EDIT_GLADE_FILE "gthumb_search_edit.glade"

typedef struct {
	GThumbWindow *  window;
	GladeXML *      gui;

	GtkWidget *     dialog;
	GtkWidget *     search_progress_dialog;

	GtkWidget *     s_start_from_fileentry;
	GtkWidget *     s_start_from_entry;
	GtkWidget *     s_include_subfold_checkbutton;
	GtkWidget *     s_filename_entry;
	GtkWidget *     s_comment_entry;
	GtkWidget *     s_place_entry;
	GtkWidget *     s_keywords_entry; 
	GtkWidget *     s_date_optionmenu;
	GtkWidget *     s_date_dateedit;

	GtkWidget *     p_progress_clist;
	GtkWidget *     p_current_dir_entry;
	GtkWidget *     p_notebook;
	GtkWidget *     p_view_button;
	GtkWidget *     p_search_button;
	GtkWidget *     p_cancel_button;
	GtkWidget *     p_close_button;
	GtkWidget *     p_searching_in_hbox;

	/* -- search data -- */

	SearchData * search_data;
	gchar **     file_patterns;
	gchar **     comment_patterns;
	gchar **     place_patterns;
	gchar **     keywords_patterns; 

	GnomeVFSAsyncHandle *handle;
	GnomeVFSURI *uri;
	GList *files;
	GList *dirs;

	gchar *catalog_path;
} DialogData;


static void
free_search_criteria_data (DialogData *data)
{
	if (data->file_patterns) {
		g_strfreev (data->file_patterns);
		data->file_patterns = NULL;
	}

	if (data->comment_patterns) {
		g_strfreev (data->comment_patterns);
		data->comment_patterns = NULL;
	}

	if (data->place_patterns) {
		g_strfreev (data->place_patterns);
		data->place_patterns = NULL;
	}

	if (data->keywords_patterns) {
		g_strfreev (data->keywords_patterns);
		data->keywords_patterns = NULL;
	} 
}


static void
free_search_results_data (DialogData *data)
{
	if (data->files) {
		path_list_free (data->files);
		data->files = NULL;
	}

	if (data->dirs) {
		path_list_free (data->dirs);
		data->dirs = NULL;
	}
}


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget *widget, 
	    DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	free_search_criteria_data (data);
	free_search_results_data (data);
	search_data_free (data->search_data);
	if (data->uri != NULL)
		gnome_vfs_uri_unref (data->uri);
	if (data->catalog_path != NULL)
		g_free (data->catalog_path);
	g_free (data);
}


static void search_images_async (DialogData *data);


static guint
opt_menu_get_active_idx (GtkWidget *opt_menu)
{
        GtkWidget *item;
        guint idx;
        GList *scan;
        GtkWidget *menu;

        menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opt_menu));
        item = gtk_menu_get_active (GTK_MENU (menu));

        idx = 0;
        scan = GTK_MENU_SHELL (menu)->children;
        while (scan && (item != scan->data)) {
                idx++;
                scan = scan->next;
        }

        return idx;
}


/* called when the "search" button is pressed. */
static void
search_clicked_cb (GtkWidget *widget, 
		   DialogData *data)
{
	gchar *entry;

	/* collect search data. */

	free_search_criteria_data (data);
	search_data_free (data->search_data);

	data->search_data = search_data_new ();

	/* * start from */

	entry = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (data->s_start_from_fileentry), FALSE);
	search_data_set_start_from (data->search_data, entry);
	g_free (entry);

	/* * recursive */

	search_data_set_recursive (data->search_data, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->s_include_subfold_checkbutton)));

	/* * file pattern */

	entry = gtk_entry_get_text (GTK_ENTRY (data->s_filename_entry));
	search_data_set_file_pattern (data->search_data, entry);
	if (entry != NULL) 
		data->file_patterns = search_util_get_patterns (entry);

	/* * comment pattern */

	entry = gtk_entry_get_text (GTK_ENTRY (data->s_comment_entry));
	search_data_set_comment_pattern (data->search_data, entry);
	if (entry != NULL) 
		data->comment_patterns = search_util_get_patterns (entry);

	/* * place pattern */

	entry = gtk_entry_get_text (GTK_ENTRY (data->s_place_entry));
	search_data_set_place_pattern (data->search_data, entry);
	if (entry != NULL) 
		data->place_patterns = search_util_get_patterns (entry);

	/* * keywords pattern */

	entry = gtk_entry_get_text (GTK_ENTRY (data->s_keywords_entry));
	search_data_set_keywords_pattern (data->search_data, entry);
	if (entry != NULL) 
		data->keywords_patterns = search_util_get_patterns (entry);

	/* * date scope pattern */

	search_data_set_date_scope (data->search_data, opt_menu_get_active_idx (data->s_date_optionmenu));

	/* * date */

	search_data_set_date (data->search_data, gnome_date_edit_get_date (GNOME_DATE_EDIT (data->s_date_dateedit)));

	/* pop up the progress dialog. */

	gtk_widget_hide (data->dialog);
	gtk_notebook_set_page (GTK_NOTEBOOK (data->p_notebook), 0);
	gtk_widget_set_sensitive (data->p_searching_in_hbox, TRUE);
	gtk_widget_set_sensitive (data->p_view_button, FALSE);
	gtk_widget_set_sensitive (data->p_search_button, FALSE);
	gtk_widget_show (data->search_progress_dialog);

	search_images_async (data);
}


static void cancel_progress_dlg_cb (GtkWidget *widget, DialogData *data);


/* called when the progress dialog is closed. */
static void
destroy_progress_cb (GtkWidget *widget, 
		     DialogData *data)
{
	cancel_progress_dlg_cb (widget, data);
	gtk_widget_destroy (data->dialog);
}


/* called when the "new search" button in the progress dialog is pressed. */
static void
new_search_clicked_cb (GtkWidget *widget, 
		       DialogData *data)
{
	cancel_progress_dlg_cb (widget, data);
	gtk_widget_hide (data->search_progress_dialog);
	gtk_widget_show (data->dialog);
}


/* called when the "view" button in the progress dialog is pressed. */
static void
view_result_cb (GtkWidget *widget, 
		DialogData *data)
{
	Catalog *catalog;
	gchar *catalog_name, *catalog_path;
	GList *scan;

	if (data->files == NULL)
		return;

	catalog = catalog_new ();

	catalog_name = g_strconcat (_("Search Result"),
				    CATALOG_EXT,
				    NULL);
	catalog_path = get_catalog_full_path (catalog_name);
	g_free (catalog_name);

	catalog_set_path (catalog, catalog_path);
	catalog_set_search_data (catalog, data->search_data);

	for (scan = data->files; scan; scan = scan->next)
		catalog_add_item (catalog, (gchar*) scan->data);

	catalog_write_to_disk (catalog);
	catalog_free (catalog);

	window_go_to_catalog (data->window, catalog_path);

	g_free (catalog_path);

	gtk_widget_destroy (data->search_progress_dialog);
}


/* called when the "ok" button is pressed. */
static void
save_result_cb (GtkWidget *widget, DialogData *data)
{
	Catalog *catalog;
	GList *scan;

	/* save search data. */

	catalog = catalog_new ();
	catalog_set_path (catalog, data->catalog_path);
	catalog_set_search_data (catalog, data->search_data);	
	for (scan = data->files; scan; scan = scan->next)
		catalog_add_item (catalog, (gchar*) scan->data);

	catalog_write_to_disk (catalog);
	catalog_free (catalog);

	window_set_sidebar_content (data->window, CATALOG_LIST);
	window_go_to_catalog (data->window, data->catalog_path);

	gtk_widget_destroy (data->search_progress_dialog);
}


static void
view_or_save_cb (GtkWidget *widget, 
		 DialogData *data)
{
	if (data->catalog_path == NULL)
		view_result_cb (widget, data);
	else
		save_result_cb (widget, data);
}


static void
search_finished (DialogData *data)
{
	gtk_entry_set_text (GTK_ENTRY (data->p_current_dir_entry), " ");

	if (data->files == NULL)
		gtk_notebook_set_page (GTK_NOTEBOOK ((data)->p_notebook), 1);

	gtk_widget_set_sensitive (data->p_searching_in_hbox, FALSE);
	gtk_widget_set_sensitive (data->p_view_button, data->files != NULL);
	gtk_widget_set_sensitive (data->p_search_button, TRUE);
}


/* called when the "cancel" button in the progress dialog is pressed. */
static void
cancel_progress_dlg_cb (GtkWidget *widget, 
			DialogData *data)
{
	if (data->handle == NULL)
		return;
	gnome_vfs_async_cancel (data->handle);
	data->handle = NULL;
	search_finished (data);
}


static void
dlg_search_ui (GThumbWindow *window, 
	       gchar *catalog_path, 
	       gboolean start_search)
{
	DialogData *data;
	GtkWidget *btn_search;
	GtkWidget *btn_close;

	data = g_new (DialogData, 1);

	data->file_patterns = NULL;
	data->comment_patterns = NULL;
	data->place_patterns = NULL;
	data->keywords_patterns = NULL; 
	data->dirs = NULL;
	data->files = NULL;
	data->window = window;
	data->handle = NULL;
	data->search_data = NULL;
	data->uri = NULL;
	data->catalog_path = catalog_path;

	if (catalog_path == NULL)
		data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" SEARCH_GLADE_FILE , NULL);
	else
		data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" SEARCH_EDIT_GLADE_FILE , NULL);

        if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "search_dialog");

	data->s_start_from_fileentry = glade_xml_get_widget (data->gui, "s_start_from_fileentry");
	data->s_start_from_entry = glade_xml_get_widget (data->gui, "s_start_from_entry");
	data->s_include_subfold_checkbutton = glade_xml_get_widget (data->gui, "s_include_subfold_checkbutton");
	data->s_filename_entry = glade_xml_get_widget (data->gui, "s_filename_entry");
	data->s_comment_entry = glade_xml_get_widget (data->gui, "s_comment_entry");
	data->s_place_entry = glade_xml_get_widget (data->gui, "s_place_entry");
	data->s_keywords_entry = glade_xml_get_widget (data->gui, "s_keywords_entry");
	data->s_date_optionmenu = glade_xml_get_widget (data->gui, "s_date_optionmenu");
	data->s_date_dateedit = glade_xml_get_widget (data->gui, "s_date_dateedit");

	data->search_progress_dialog = glade_xml_get_widget (data->gui, "search_progress_dialog");
	data->p_progress_clist = glade_xml_get_widget (data->gui, "p_progress_clist");
	data->p_current_dir_entry = glade_xml_get_widget (data->gui, "p_current_dir_entry");
	data->p_notebook = glade_xml_get_widget (data->gui, "p_notebook");
	data->p_view_button = glade_xml_get_widget (data->gui, "p_view_button");
	data->p_search_button = glade_xml_get_widget (data->gui, "p_search_button");
	data->p_cancel_button = glade_xml_get_widget (data->gui, "p_cancel_button");
	data->p_close_button = glade_xml_get_widget (data->gui, "p_close_button");
	data->p_searching_in_hbox = glade_xml_get_widget (data->gui, "p_searching_in_hbox");

        btn_search = glade_xml_get_widget (data->gui, "s_search_button");
        btn_close = glade_xml_get_widget (data->gui, "s_close_button");

	/* Set widgets data. */

	if (catalog_path == NULL) {
		if (data->window->dir_list->path != NULL)
			gtk_entry_set_text (GTK_ENTRY (data->s_start_from_entry), data->window->dir_list->path);
		else
			gtk_entry_set_text (GTK_ENTRY (data->s_start_from_entry), g_get_home_dir ());
	} else {
		Catalog *catalog;
		SearchData *search_data;

		catalog = catalog_new ();
		catalog_load_from_disk (catalog, data->catalog_path);

		search_data = catalog->search_data;
		gtk_entry_set_text (GTK_ENTRY (data->s_start_from_entry), search_data->start_from);
	
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->s_include_subfold_checkbutton), search_data->recursive);
		
		gtk_entry_set_text (GTK_ENTRY (data->s_filename_entry),
				    search_data->file_pattern);
		gtk_entry_set_text (GTK_ENTRY (data->s_comment_entry),
				    search_data->comment_pattern);
		gtk_entry_set_text (GTK_ENTRY (data->s_place_entry),
				    search_data->place_pattern);
		gtk_entry_set_text (GTK_ENTRY (data->s_keywords_entry),
				    search_data->keywords_pattern);	
		gtk_option_menu_set_history (GTK_OPTION_MENU (data->s_date_optionmenu),
					     search_data->date_scope);
		gnome_date_edit_set_time (GNOME_DATE_EDIT (data->s_date_dateedit),
					  search_data->date);
		catalog_free (catalog);
	}

	/* Set the signals handlers. */
	
	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) destroy_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (btn_search), "clicked",
			    GTK_SIGNAL_FUNC (search_clicked_cb),
			    data);
	gtk_signal_connect_object (GTK_OBJECT (btn_close), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));

	gtk_signal_connect (GTK_OBJECT (data->search_progress_dialog), "destroy",
			    (GtkSignalFunc) destroy_progress_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (data->p_search_button), "clicked",
			    GTK_SIGNAL_FUNC (new_search_clicked_cb),
			    data);
	gtk_signal_connect_object (GTK_OBJECT (data->p_close_button), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->search_progress_dialog));
	gtk_signal_connect (GTK_OBJECT (data->p_cancel_button), "clicked",
			    GTK_SIGNAL_FUNC (cancel_progress_dlg_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->p_view_button), "clicked",
			    GTK_SIGNAL_FUNC (view_or_save_cb),
			    data);

	/* Run dialog. */

	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (data->search_progress_dialog), GTK_WINDOW (window->app));

	if (! start_search) 
		gtk_widget_show (data->dialog);
	else 
		search_clicked_cb (NULL, data);
}


/* --------------------------- */


#define CASE_INSENSITIVE (1 << 4)

static gboolean
match_patterns (gchar **patterns, gchar *string)
{
	gint i;
	gint result;

	if (patterns[0] == NULL)
		return TRUE;

	if (string == NULL)
		return FALSE;

	result = FNM_NOMATCH;
	i = 0;
	while ((result != 0) && (patterns[i] != NULL)) {
		result = fnmatch (patterns[i], string, CASE_INSENSITIVE);
		i++;
	}

	return (result == 0);
}


static gboolean
file_respects_search_criteria (DialogData *data, gchar *filename)
{
	CommentData *comment_data;
	gboolean result;
	gboolean match_keywords;
	gboolean match_date;
	gint i;
	gchar *comment;
	gchar *place;
	gint   keywords_n;
	time_t time;

	if (! file_is_image (filename, preferences.fast_file_type))
		return FALSE;

	comment_data = comments_load_comment (filename);

	if (comment_data == NULL) {
		comment = NULL;
		place = NULL;
		keywords_n = 0;
		time = 0;
	} else {
		comment = comment_data->comment;
		place = comment_data->place;
		keywords_n = comment_data->keywords_n;
		time = comment_data->time;
	}

	match_keywords = data->keywords_patterns[0] == NULL;
	for (i = 0; i < keywords_n; i++)
		if (match_patterns (data->keywords_patterns, 
				    comment_data->keywords[i])) {
			match_keywords = TRUE;
			break;
		}

	match_date = FALSE;
	if (data->search_data->date_scope == DATE_ANY)
		match_date = TRUE;
	else if ((data->search_data->date_scope == DATE_BEFORE) 
		 && (time != 0)
		 && (time < data->search_data->date))
		match_date = TRUE;
	else if ((data->search_data->date_scope == DATE_EQUAL_TO) 
		 && (time != 0)
		 && (time == data->search_data->date))
		match_date = TRUE;
	else if ((data->search_data->date_scope == DATE_AFTER) 
		 && (time != 0)
		 && (time > data->search_data->date))
		match_date = TRUE;

	result = (match_patterns (data->file_patterns, filename)
		  && match_patterns (data->comment_patterns, comment)
		  && match_patterns (data->place_patterns, place)
		  && match_keywords
		  && match_date);

	comment_data_free (comment_data);

	return result;
}


static void
add_file_list (DialogData *data, GList *file_list)
{
	GList *scan;

	gtk_clist_freeze (GTK_CLIST (data->p_progress_clist));
	for (scan = file_list; scan; scan = scan->next) {
		gchar *filename = (gchar*) scan->data;
		gchar *text[3];

		text[0] = g_strdup (file_name_from_path (filename));
		text[1] = remove_level_from_path (filename);
		text[2] = NULL;
		
		gtk_clist_append (GTK_CLIST (data->p_progress_clist), text);

		g_free (text[0]);
		g_free (text[1]);
	}
	gtk_clist_columns_autosize (GTK_CLIST (data->p_progress_clist));
	gtk_clist_thaw (GTK_CLIST (data->p_progress_clist));

	data->files = g_list_concat (data->files, file_list);
}


static void search_dir_async (DialogData *data, gchar *dir);


static void
directory_load_cb (GnomeVFSAsyncHandle *handle,
		   GnomeVFSResult result,
		   GList *list,
		   guint entries_read,
		   gpointer callback_data)
{
	DialogData *data = callback_data;
	GnomeVFSFileInfo *info;
	GList *node, *files = NULL;

	for (node = list; node != NULL; node = node->next) {
		GnomeVFSURI *full_uri = NULL;
		gchar *str_uri, *unesc_uri;

		info = node->data;

		switch (info->type) {
		case GNOME_VFS_FILE_TYPE_REGULAR:
			full_uri = gnome_vfs_uri_append_file_name (data->uri, info->name);
			str_uri = gnome_vfs_uri_to_string (full_uri, GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
			unesc_uri = gnome_vfs_unescape_string (str_uri, NULL);

			if (file_respects_search_criteria (data, unesc_uri)) 
				files = g_list_prepend (files, unesc_uri);
			else
				g_free (unesc_uri);

			g_free (str_uri);
			break;

		case GNOME_VFS_FILE_TYPE_DIRECTORY:
			full_uri = gnome_vfs_uri_append_path (data->uri, info->name);
			str_uri = gnome_vfs_uri_to_string (full_uri, GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
			unesc_uri = gnome_vfs_unescape_string (str_uri, NULL);

			data->dirs = g_list_prepend (data->dirs,  unesc_uri);
			g_free (str_uri);
			break;

		default:
		}

		if (full_uri)
			gnome_vfs_uri_unref (full_uri);
	}

	if (files != NULL)
		add_file_list (data, files);

	if (result == GNOME_VFS_ERROR_EOF) {
		gboolean good_dir_to_search_into = TRUE;

		if (! data->search_data->recursive) {
			search_finished (data);
			return;
		}
		
		do {
			GList *first_dir;
			gchar *dir;

			if (data->dirs == NULL) {
				search_finished (data);
				return;
			}

			first_dir = data->dirs;
			data->dirs = g_list_remove_link (data->dirs, first_dir);
			dir = (gchar*) first_dir->data; 
			g_list_free (first_dir);

			good_dir_to_search_into = ! file_is_hidden (file_name_from_path (dir));
			if (good_dir_to_search_into)
				search_dir_async (data, dir);
			g_free (dir);
		} while (! good_dir_to_search_into);

	} else if (result != GNOME_VFS_OK) {
		gchar *path;

		path = gnome_vfs_uri_to_string (data->uri, 
						GNOME_VFS_URI_HIDE_NONE);
		g_warning ("Cannot load directory %s : %s\n", path,
			   gnome_vfs_result_to_string (result));
		g_free (path);

		search_finished (data);
	}
}


static void 
search_dir_async (DialogData *data, gchar *dir)
{
	gchar *escaped;

	gtk_entry_set_text (GTK_ENTRY (data->p_current_dir_entry), dir);

	if (data->uri != NULL)
		gnome_vfs_uri_unref (data->uri);
	escaped = gnome_vfs_escape_path_string (dir);
	data->uri = gnome_vfs_uri_new (escaped);
	g_free (escaped);

	gnome_vfs_async_load_directory_uri (
		& (data->handle),
		data->uri,
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS,
		GNOME_VFS_DIRECTORY_FILTER_NONE,
		(GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR 
		 | GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR), 
		NULL,
		32 /* items_per_notification FIXME */,
		directory_load_cb,
		data);
}


static void 
search_images_async (DialogData *data)
{
	free_search_results_data (data);
	gtk_clist_clear (GTK_CLIST (data->p_progress_clist));
	search_dir_async (data, data->search_data->start_from);
}
