/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gtk/gtk.h>
#include <gnome.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include "file-utils.h"
#include "image-list.h"
#include "image-list-utils.h"
#include "typedefs.h"
#include "main.h"
#include "window.h"


typedef struct {
	GThumbWindow    *window;
	GladeXML  *gui;

	GtkWidget *dialog;
	GtkWidget *app_list_clist;
	GtkWidget *cancel_btn;
	GList     *app_list;
	GList     *file_list;
} DialogData;


/* called when the main dialog is closed. */
static void
open_with__destroy_cb (GtkWidget *widget, 
		       DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	if (data->file_list)
		path_list_free (data->file_list);
	if (data->app_list)
		gnome_vfs_mime_application_list_free (data->app_list);
	g_free (data);
}


/* counts how many characters of type @c are in @str. */
static gint
count (const gchar *str, char c, gint str_len)
{
	const gchar *s;
	gint n;

	for (n = 0, s = str; s != str + str_len; s++)
		if (*s == c) n++;

	return n;
}


/* escape with backslash the file name. */
static gchar*
shell_escape (const gchar *filename)
{
	static gchar bad_char[] = {'$', '\'', '`', '"', '\\', '!', '?', ' ', 
				   '(', ')', '[', ']'};
	static const gint bad_chars = sizeof (bad_char) / sizeof (gchar);
	gchar *escaped;
	gint i, l, new_l;
	const gchar *s;
	gchar *t;

	if (filename == NULL) return NULL;

	new_l = l = strlen (filename);
	for (i = 0; i < bad_chars; i++)
		new_l += count (filename, bad_char[i], l);
	escaped = g_malloc (new_l + 1);

	s = filename;
	t = escaped;
	while (*s) {
		gboolean is_bad;
	
		is_bad = FALSE;
		for (i = 0; (i < bad_chars) && !is_bad; i++)
			is_bad = (*s == bad_char[i]);

		if (is_bad)
			*t++ = '\\';
		*t++ = *s++;
	}
	*t = 0;

	return escaped;
}


static void
exec_command (gchar *application, GList *file_list)
{
	GList *scan;
	gchar *files;
	gint files_length;
	gchar *command;

	/* compute the file list length. */
	files_length = 0;
	for (scan = file_list; scan; scan = scan->next)
		files_length += strlen (scan->data) + 1;

	files = g_malloc (files_length + 1);
	*files = 0;
	for (scan = file_list; scan; scan = scan->next) {
		gchar *name_escaped;

		if (scan != file_list)
			files = strcat (files, " ");

		name_escaped = shell_escape (scan->data);
		files = strncat (files, name_escaped, strlen (name_escaped));
		g_free (name_escaped);
	}

	command = g_strdup_printf ("%s %s", application, files);
	g_free (files);

        gnome_execute_shell (g_get_home_dir (), command);
	g_free (command);
}


static void 
app_selected_cb (GtkWidget *widget, 
		 gint row, 
		 gint col,
		 GdkEvent *event,
		 gpointer callback_data)
{
	DialogData *data = callback_data;
	GnomeVFSMimeApplication *app;
	
	app = gtk_clist_get_row_data (GTK_CLIST (widget), row);
	exec_command (app->command, data->file_list);
	gtk_widget_destroy (data->dialog);
}


/* create the "open with" dialog. */
void
open_with_cb (GtkWidget *widget, 
	      void *callback_data)
{
	GThumbWindow *window = callback_data;
	DialogData *data;
	ImageList *ilist;
	GList *scan;
	gchar *col_data[2];
	GnomeVFSMimeApplication *app;
	gint row;
	GList *app_names = NULL;

	data = g_new (DialogData, 1);

	ilist = IMAGE_LIST (window->file_list->ilist);
	data->file_list = ilist_utils_get_file_list_selection (ilist);
	if (data->file_list == NULL) {
		/* no selection. */
		
		if ((window->image_path == NULL) 
		    || !path_is_file (window->image_path)) {
			/* no image in the viewer. */
			g_free (data);
			return;
		}

		data->file_list = g_list_prepend (NULL, window->image_path);
	}
	
	data->window = window;
	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_FILE , NULL);
        if (! data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "open_with_dialog");
	data->app_list_clist = glade_xml_get_widget (data->gui, "app_list_clist");
	data->cancel_btn = glade_xml_get_widget (data->gui, "open_with_cancel_btn");

	/* Set the signals handlers. */

	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) open_with__destroy_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (data->app_list_clist), 
			    "select_row",
			    (GtkSignalFunc) app_selected_cb, 
			    data);
	gtk_signal_connect_object (GTK_OBJECT (data->cancel_btn), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));
	/* Set data. */

	data->app_list = NULL;
	for (scan = data->file_list; scan; scan = scan->next) {
		const char *result;
		const char *name = scan->data;

		if (preferences.fast_file_type)
			result = gnome_vfs_mime_type_from_name_or_default (name, NULL);
		else 
			result = gnome_mime_type_from_magic (name);

		data->app_list = g_list_concat (data->app_list, gnome_vfs_mime_get_all_applications (result));
	}

	gtk_clist_column_titles_passive (GTK_CLIST (data->app_list_clist));
	gtk_clist_set_sort_column (GTK_CLIST (data->app_list_clist), 1);

	col_data[1] = NULL;
	for (scan = data->app_list; scan; scan = scan->next) {
		gboolean found;

		app = scan->data;

		found = FALSE;
		if (app_names != NULL) {
			GList *p;
			for (p = app_names; p; p = p->next)
				if (strcmp ((char*)p->data, app->command) == 0)
					found = TRUE;
		}

		if (found)
			continue;

		app_names = g_list_prepend (app_names, app->command);

		col_data[0] = app->name;
		row = gtk_clist_append (GTK_CLIST (data->app_list_clist), 
					col_data);
		gtk_clist_set_row_data (GTK_CLIST (data->app_list_clist), row,
					app);
	}
	if (app_names)
		g_list_free (app_names);

	gtk_clist_sort (GTK_CLIST (data->app_list_clist));


	/* Run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), 
				      GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}
