/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <libgnome/libgnome.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-async-ops.h>

#include "typedefs.h"
#include "dir-list.h"
#include "file-list.h"
#include "file-data.h"
#include "file-utils.h"
#include "main.h"
#include "pixbuf-utils.h"

#include "icons/dir.xpm" 
#include "icons/up_dir.xpm"

DirList *
dir_list_new ()
{
	DirList *dir_list = NULL;
	GtkWidget *clist;
	GtkWidget *scrolled;
	gint height;

	dir_list = g_new (DirList, 1);

	/* Set default values. */
	dir_list->path = NULL;
	dir_list->try_path = NULL;
	dir_list->list = NULL;
	dir_list->file_list = NULL;
	dir_list->show_dot_files = preferences.show_dot_files;
	dir_list->old_dir = NULL;
	dir_list->dir_load_handle = NULL;
	dir_list->result = GNOME_VFS_OK;

	/* Create the widgets. */
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_ALWAYS);
	clist = gtk_clist_new (1);
	gtk_clist_column_titles_passive (GTK_CLIST (clist)); 
	gtk_widget_set_usize (clist, CLIST_WIDTH_DEFAULT, CLIST_HEIGHT_DEFAULT);
	height = (GTK_WIDGET (clist)->style->font->ascent +
		  GTK_WIDGET (clist)->style->font->descent + CLIST_ROW_PAD);
	gtk_clist_set_row_height (GTK_CLIST (clist), height);	
	gtk_container_add (GTK_CONTAINER (scrolled), clist);
	dir_list->clist = clist;
	
	dir_list->root_widget = scrolled;

	return dir_list;
}


void
dir_list_free (DirList *dir_list)
{
	g_return_if_fail (dir_list != NULL);

	if (dir_list->path != NULL)
		g_free (dir_list->path);

	if (dir_list->try_path != NULL)
		g_free (dir_list->try_path);

	if (dir_list->old_dir != NULL)
		g_free (dir_list->old_dir);

	if (dir_list->file_list != NULL) {
		g_list_foreach (dir_list->file_list, (GFunc) g_free, NULL);
		g_list_free (dir_list->file_list);
	}

	if (dir_list->list != NULL) {
		g_list_foreach (dir_list->list, (GFunc) g_free, NULL);
		g_list_free (dir_list->list);
	}

	if (dir_list->dir_load_handle != NULL)
		g_free (dir_list->dir_load_handle);

	g_free (dir_list);
}


static void
dir_list_refresh_continue (PathListData *pld, 
			   gpointer data)
{
	GList *scan;
	GList *new_dir_list = NULL;
	GList *new_file_list = NULL;
	GList *filtered;
	gint width;
	GtkWidget *clist;
	GdkPixmap *dir_pixmap, *up_dir_pixmap;
	GdkBitmap *dir_bitmap, *up_dir_bitmap;
	DirList *dir_list;

	dir_list = data;
	dir_list->result = pld->result;

	if (dir_list->dir_load_handle != NULL) {
		g_free (dir_list->dir_load_handle);
		dir_list->dir_load_handle = NULL;
	}

	if (pld->result != GNOME_VFS_ERROR_EOF) {
		path_list_data_free (pld);
		if (dir_list->done_func) 
			dir_list->done_func (dir_list, dir_list->done_data);
		return;
	}

	/* Update path data. */

	if (dir_list->old_dir != NULL) {
		g_free (dir_list->old_dir);
		dir_list->old_dir = NULL;
	}

	if (dir_list->path == NULL)
		dir_list->old_dir = NULL;
	else {
		gchar *previous_dir = remove_level_from_path (dir_list->path);

		if (strcmp (previous_dir, dir_list->try_path) == 0)
			dir_list->old_dir = g_strdup (file_name_from_path (dir_list->path));
		else
			dir_list->old_dir = NULL;
		g_free (previous_dir);
	}

	if (dir_list->path != NULL) 
		g_free (dir_list->path);
	dir_list->path = dir_list->try_path;
	dir_list->try_path = NULL;

	/**/

	new_file_list = pld->files;
	new_dir_list = pld->dirs;

	pld->files = NULL;
	pld->dirs = NULL;
	path_list_data_free (pld);

	/* Delete the old dir list. */

	if (dir_list->list != NULL) {
		g_list_foreach (dir_list->list, (GFunc) g_free, NULL);
		g_list_free (dir_list->list);
		dir_list->list = NULL;
	}

	/* Set the new file list. */

	if (dir_list->file_list != NULL) {
		g_list_foreach (dir_list->file_list, (GFunc) g_free, NULL);
		g_list_free (dir_list->file_list);
	}
	dir_list->file_list = new_file_list;

	/* Set the new dir list */

	filtered = dir_list_filter_and_sort (new_dir_list, TRUE, 
					     preferences.show_dot_files);

	/* * Add the ".." entry if the current path is not "/". 
	 * path_list_new does not include the "." and ".." elements. */

	if (strcmp (dir_list->path, "/") != 0)
		filtered = g_list_prepend (filtered, g_strdup (".."));
	dir_list->list = filtered;

	g_list_foreach (new_dir_list, (GFunc) g_free, NULL);
	g_list_free (new_dir_list);

	/* Update the clist. */

	clist = dir_list->clist;
	gtk_clist_freeze (GTK_CLIST (clist));
	gtk_clist_clear (GTK_CLIST (clist));

	pixmap_from_xpm ((const char **) dir_xpm, &dir_pixmap, &dir_bitmap);
	pixmap_from_xpm ((const char **) up_dir_xpm, &up_dir_pixmap, &up_dir_bitmap);

	width = 0;
	scan = dir_list->list;
	while (scan) {
		gchar *col_data[2];
		gint   row;

		col_data[0] = scan->data;
		col_data[1] = NULL;

		row = gtk_clist_append (GTK_CLIST (clist), col_data);

		if (strcmp (scan->data, "..") != 0)
			gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
					       scan->data, 
					       10, dir_pixmap, dir_bitmap);
		else
			gtk_clist_set_pixtext (GTK_CLIST (clist), row, 0, 
					       scan->data, 
					       10, 
					       up_dir_pixmap, up_dir_bitmap);

		gtk_clist_set_row_data (GTK_CLIST (clist), row, scan->data);
		width = MAX (width, gdk_string_width (clist->style->font, col_data[0]));

		scan = scan->next;
	}

	gdk_pixmap_unref (dir_pixmap);
	gdk_bitmap_unref (dir_bitmap);
	gdk_pixmap_unref (up_dir_pixmap);
	gdk_bitmap_unref (up_dir_bitmap);

	gtk_clist_set_column_width (GTK_CLIST (dir_list->clist), 0, width);
	gtk_clist_thaw (GTK_CLIST (dir_list->clist));

	/* Make past dir visible in the list. */

	if (dir_list->old_dir) {
		GList *scan;
		gint   row;
		gint   found = FALSE;

		row = -1;
		scan = dir_list->list;
		while (scan && !found) {
			if (strcmp (dir_list->old_dir, scan->data) == 0) 
				found = TRUE;
			scan = scan->next;
			row++;
		}

		if (found) gtk_clist_moveto (GTK_CLIST (dir_list->clist), row, 0, 0.5, 0.0);
		g_free (dir_list->old_dir);
		dir_list->old_dir = NULL;
	}

	if (dir_list->done_func) 
		dir_list->done_func (dir_list, dir_list->done_data);
}


void
dir_list_change_to (DirList *dir_list,
		    const gchar *path,
		    DirListDoneFunc func,
		    gpointer data)
{
	g_return_if_fail (dir_list != NULL);

	dir_list->done_func = func;
	dir_list->done_data = data;

	if ((path != dir_list->try_path) && (dir_list->try_path != NULL)) 
		g_free (dir_list->try_path);
	dir_list->try_path = g_strdup (path);

	if (dir_list->dir_load_handle != NULL)
		path_list_handle_free (dir_list->dir_load_handle);

	dir_list->dir_load_handle = path_list_async_new (dir_list->try_path, dir_list_refresh_continue, dir_list);
}


void
dir_list_interrupt_change_to (DirList *dir_list)
{
	g_return_if_fail (dir_list != NULL);

	if (dir_list->dir_load_handle != NULL) {
		path_list_async_interrupt (dir_list->dir_load_handle);
		dir_list->dir_load_handle = NULL;
	}
}


gchar *
dir_list_path_from_row (DirList *dir_list,
			gint row)
{
	gchar *name;
	gchar *new_path;

	g_return_val_if_fail (dir_list != NULL, NULL);

	name = gtk_clist_get_row_data (GTK_CLIST (dir_list->clist), row);

	if (!name) return NULL;

	if (strcmp (name, ".") == 0)
		new_path = g_strdup (dir_list->path);
	else if (strcmp (name, "..") == 0)
		new_path = remove_level_from_path (dir_list->path);
	else {
		if (strcmp (dir_list->path, "/") == 0)
			new_path = g_strconcat (dir_list->path, 
						name, 
						NULL);
		else
			new_path = g_strconcat (dir_list->path, 
						"/", 
						name, 
						NULL);
	}
	
	return new_path;
}


GList *
dir_list_get_file_list (DirList *dir_list)
{
	g_return_val_if_fail (dir_list != NULL, NULL);
	return g_list_copy (dir_list->file_list);
}
