/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <libgnomevfs/gnome-vfs-async-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <sys/stat.h>
#include "image-loader.h"
#include "my-marshallers.h"

/* The number of bytes to read per call. */
#define CHUNK_SIZE 65536 


typedef struct {
	GdkPixbuf            *pixbuf;
	GdkPixbufAnimation   *animation;

	gboolean              as_animation; /* Whether to load the image in a
					     * GdkPixbufAnimation structure. */

	GnomeVFSURI          *uri;
	
	gchar                *buffer;
	guint                 buffer_size;

	GnomeVFSAsyncHandle  *info_handle;
	GnomeVFSAsyncHandle  *read_handle;

	GnomeVFSFileSize      bytes_read;
	GnomeVFSFileSize      bytes_total;

	gboolean              done;
	gboolean              error;

	GdkPixbufLoader      *loader;

	GTimer               *timer;
} ImageLoaderPrivateData;


enum {
	ERROR,
	DONE,
	PROGRESS,
	LAST_SIGNAL
};


static GtkObjectClass *parent_class = NULL;
static guint image_loader_signals[LAST_SIGNAL] = { 0 };


static void 
image_loader_destroy (GtkObject *object)
{
	ImageLoader *il;
	ImageLoaderPrivateData *priv;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_IMAGE_LOADER (object));
  
        il = IMAGE_LOADER (object);
	priv = il->priv;

	image_loader_stop (il);

	if (priv->pixbuf)
		gdk_pixbuf_unref (priv->pixbuf);
	
	if (priv->animation)
		gdk_pixbuf_animation_unref (priv->animation);

	if (priv->uri)
		gnome_vfs_uri_unref (priv->uri);
	
	if (priv->buffer)
		g_free (priv->buffer);

	g_timer_destroy (priv->timer);

	g_free (priv);
	il->priv = NULL;

	/* Chain up */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
image_loader_class_init (ImageLoaderClass *class)
{
	GtkObjectClass *object_class;

	parent_class = gtk_type_class (gtk_object_get_type ());
	object_class = (GtkObjectClass*) class;

	image_loader_signals[ERROR] =
                gtk_signal_new ("error",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageLoaderClass, error),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	image_loader_signals[DONE] =
                gtk_signal_new ("done",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageLoaderClass, done),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	image_loader_signals[PROGRESS] =
                gtk_signal_new ("progress",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ImageLoaderClass, progress),
                                gtk_marshal_NONE__FLOAT,
                                GTK_TYPE_NONE, 1,
				GTK_TYPE_FLOAT);

	gtk_object_class_add_signals (object_class, image_loader_signals, 
                                      LAST_SIGNAL);

	object_class->destroy = image_loader_destroy;

	class->error = NULL;
	class->done = NULL;
	class->progress = NULL;
}


static void
image_loader_init (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	il->priv = g_new (ImageLoaderPrivateData, 1);
	priv = il->priv;

	priv->pixbuf = NULL;
	priv->animation = NULL;
	priv->uri = NULL;

	priv->bytes_read = 0;
	priv->bytes_total = 0;

	priv->buffer_size = CHUNK_SIZE;
	priv->buffer = g_malloc (CHUNK_SIZE);

	priv->info_handle = NULL;
	priv->read_handle = NULL;

	priv->done = FALSE;
	priv->error = FALSE;

	priv->loader = NULL;

	priv->timer = g_timer_new ();
}


GtkType
image_loader_get_type ()
{
	static guint image_loader_type = 0;

	if (! image_loader_type) {
		GtkTypeInfo image_loader_info = {
			"ImageLoader",
			sizeof (ImageLoader),
			sizeof (ImageLoaderClass),
			(GtkClassInitFunc) image_loader_class_init,
			(GtkObjectInitFunc) image_loader_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		image_loader_type = gtk_type_unique (gtk_object_get_type (), 
						     &image_loader_info);
	}
	
	return image_loader_type;
}


GtkObject*     
image_loader_new (const gchar *path,
		  gboolean as_animation)
{
	ImageLoaderPrivateData *priv;
	ImageLoader *il;

	il = IMAGE_LOADER (gtk_type_new (image_loader_get_type ()));
	priv = (ImageLoaderPrivateData*) il->priv;

	priv->as_animation = as_animation;
	if (path != NULL) {
		gchar *escaped = gnome_vfs_escape_path_string (path);
		priv->uri = gnome_vfs_uri_new (escaped);
		g_free (escaped);
	}

	return GTK_OBJECT (il);
}


void
image_loader_set_path (ImageLoader *il,
		       const gchar *path)
{
	ImageLoaderPrivateData *priv;
	gchar *escaped_path;

	g_return_if_fail (il != NULL);
	g_return_if_fail (path != NULL);
	
	priv = il->priv;

	if (priv->uri != NULL)
		gnome_vfs_uri_unref (priv->uri);
	escaped_path = gnome_vfs_escape_path_string (path);
	priv->uri = gnome_vfs_uri_new (escaped_path);
	g_free (escaped_path);
}


void
image_loader_set_uri (ImageLoader *il,
		      const GnomeVFSURI *uri)
{
	ImageLoaderPrivateData *priv;

	g_return_if_fail (il != NULL);
	g_return_if_fail (uri != NULL);
	
	priv = il->priv;

	if (priv->uri != NULL)
		gnome_vfs_uri_unref (priv->uri);
	priv->uri = gnome_vfs_uri_dup (uri);
}


void
image_loader_set_pixbuf (ImageLoader *il,
			 GdkPixbuf *pixbuf)
{
	ImageLoaderPrivateData *priv;

	g_return_if_fail (il != NULL);
	g_return_if_fail (pixbuf != NULL);

	priv = il->priv;

	if (priv->pixbuf != NULL)
		gdk_pixbuf_unref (priv->pixbuf);

	gdk_pixbuf_ref (pixbuf);
	priv->pixbuf = pixbuf;
}


static void 
image_loader_sync_pixbuf (ImageLoader *il,
			  GdkPixbufLoader *pb_loader)
{
	GdkPixbuf *pixbuf;
	ImageLoaderPrivateData *priv;
  
	g_return_if_fail (il != NULL);
     
	priv = il->priv;

	if (priv->as_animation) {
		if (priv->animation != NULL)
			gdk_pixbuf_animation_unref (priv->animation);
		priv->animation = gdk_pixbuf_loader_get_animation (pb_loader);
		
		if ((priv->animation != NULL) 
		    && (gdk_pixbuf_animation_get_num_frames (priv->animation) > 1)) {
			gdk_pixbuf_animation_ref (priv->animation);
			return;
		} else
			priv->animation = NULL;
	}

	pixbuf = gdk_pixbuf_loader_get_pixbuf (pb_loader);

	if (priv->pixbuf == pixbuf) return;
		
	if (pixbuf) 
		gdk_pixbuf_ref (pixbuf);
	if (priv->pixbuf) 
		gdk_pixbuf_unref (priv->pixbuf);
     	priv->pixbuf = pixbuf;
}


static void 
image_loader_done (ImageLoader *il)
{
	ImageLoaderPrivateData *priv = il->priv;

	priv->error = FALSE;
	image_loader_stop (il);

	gtk_signal_emit (GTK_OBJECT (il), image_loader_signals[DONE]);
}


static void 
image_loader_error (ImageLoader *il)
{
	ImageLoaderPrivateData *priv = il->priv;

	priv->info_handle = NULL;
	priv->read_handle = NULL;

	priv->error = TRUE;
	image_loader_stop (il);

	gtk_signal_emit (GTK_OBJECT (il), image_loader_signals[ERROR]);
}


static void
close_successfully_cb (GnomeVFSAsyncHandle *handle,
		       GnomeVFSResult result,
		       gpointer data)
{
	ImageLoader *il = data;
	ImageLoaderPrivateData *priv = il->priv;

	priv->read_handle = NULL;
	image_loader_done (il);
}


static void
read_cb (GnomeVFSAsyncHandle *handle,
	 GnomeVFSResult result,
	 gpointer buffer,
	 GnomeVFSFileSize bytes_requested,
	 GnomeVFSFileSize bytes_read,
	 gpointer data)
{
	ImageLoader *il = data;
	ImageLoaderPrivateData *priv = il->priv;

	if (result == GNOME_VFS_ERROR_EOF) {
		gnome_vfs_async_close (handle,
				       close_successfully_cb,
				       il);
		return;
	}

	if (result != GNOME_VFS_OK) {
		gnome_vfs_async_cancel (handle);
		image_loader_error (il);
		return;
	}

	if (bytes_read == 0) {
		gnome_vfs_async_close (handle,
				       close_successfully_cb,
				       il);
		return;
	}

	if (! gdk_pixbuf_loader_write (priv->loader, 
				       priv->buffer, 
				       bytes_read)) {
		gnome_vfs_async_cancel (handle);
		image_loader_error (il);
		return;
	}

	priv->bytes_read += bytes_read;
	gtk_signal_emit (GTK_OBJECT (il), image_loader_signals[PROGRESS],
			 (gfloat) priv->bytes_read / priv->bytes_total);

	gnome_vfs_async_read (handle,
			      priv->buffer,
			      priv->buffer_size,
			      read_cb,
			      il);
}


static void
open_cb (GnomeVFSAsyncHandle *handle,
	 GnomeVFSResult result,
	 gpointer data)
{
	ImageLoader *il = data;
	ImageLoaderPrivateData *priv = il->priv;

	if (result != GNOME_VFS_OK) {
		gnome_vfs_async_cancel (handle);
		image_loader_error (il);
		return;
	}

	gnome_vfs_async_read (priv->read_handle,
			      priv->buffer,
			      priv->buffer_size,
			      read_cb,
			      il);
}


static void
get_file_info_cb (GnomeVFSAsyncHandle *handle,
		  GList *results,
		  gpointer data)
{
	GnomeVFSGetFileInfoResult *info_result;
	ImageLoader *il = data;
	ImageLoaderPrivateData *priv = il->priv;

	info_result = results->data;
	priv->info_handle = NULL;

	if (info_result->result != GNOME_VFS_OK) {
		image_loader_error (il);
		return;
	}

	priv->bytes_total = info_result->file_info->size;
	priv->bytes_read = 0;
	priv->done = FALSE;
	
	if (priv->pixbuf != NULL) {
		gdk_pixbuf_unref (priv->pixbuf);
		priv->pixbuf = NULL;
	}

	if (priv->animation != NULL) {
		gdk_pixbuf_animation_unref (priv->animation);
		priv->animation = NULL;
	}

	priv->loader = gdk_pixbuf_loader_new ();

	gnome_vfs_async_open_uri (& (priv->read_handle),
				  priv->uri,
				  GNOME_VFS_OPEN_READ,
				  open_cb,
				  il);
}


void
image_loader_start (ImageLoader *il)
{
	GList *uri_list;
	ImageLoaderPrivateData *priv;

	g_return_if_fail (il != NULL);

	priv = il->priv;

	g_return_if_fail (priv->uri != NULL);

	g_timer_reset (priv->timer);
	g_timer_start (priv->timer);

	uri_list = g_list_prepend (NULL, priv->uri);
	gnome_vfs_async_get_file_info (& (priv->info_handle),
				       uri_list,
				       (GNOME_VFS_FILE_INFO_DEFAULT
					| GNOME_VFS_FILE_INFO_FOLLOW_LINKS),
				       get_file_info_cb,
				       il);
	g_list_free (uri_list);
}


void 
image_loader_stop (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_if_fail (il != NULL);

	priv = il->priv;

	g_timer_stop (priv->timer);

	if (priv->info_handle != NULL) {
		gnome_vfs_async_cancel (priv->info_handle);
		priv->info_handle = NULL;
	}

	if (priv->read_handle != NULL) {
		gnome_vfs_async_cancel (priv->read_handle);
		priv->read_handle = NULL;
	}

	if (priv->loader) {
		gdk_pixbuf_loader_close (priv->loader);
		if (! priv->error) 
			image_loader_sync_pixbuf (il, priv->loader);
		gtk_object_unref (GTK_OBJECT (priv->loader));
		priv->loader = NULL;
	}

	priv->done = TRUE;
	priv->error = FALSE;
}


void 
image_loader_stop_with_error (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_if_fail (il != NULL);

	priv = il->priv;
	priv->error = TRUE;
	image_loader_stop (il);
}


GdkPixbuf *
image_loader_get_pixbuf (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_val_if_fail (il != NULL, NULL);
	priv = il->priv;

	return priv->pixbuf;
}


GdkPixbufAnimation *
image_loader_get_animation (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_val_if_fail (il != NULL, NULL);
	priv = il->priv;

	return priv->animation;
}


gfloat 
image_loader_get_percent (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_val_if_fail (il != NULL, 0.0);
	priv = il->priv;

	if (priv->bytes_total == 0) 
		return 0.0;
	else
		return (gfloat) priv->bytes_read / priv->bytes_total;
}


gint 
image_loader_get_is_done (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_val_if_fail (il != NULL, 0);
	priv = il->priv;

	return priv->done;
}


gchar *
image_loader_get_path (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;
	gchar *path;
        gchar *esc_path;

	g_return_val_if_fail (il != NULL, 0);
	priv = il->priv;

        if (priv->uri == NULL) 
                return NULL;

        path = gnome_vfs_uri_to_string (priv->uri, 
                                        GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
        esc_path = gnome_vfs_unescape_string (path, NULL);
        g_free (path);

        return esc_path;
}


GnomeVFSURI *
image_loader_get_uri (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_val_if_fail (il != NULL, 0);
	priv = il->priv;

	return priv->uri;
}


GTimer *
image_loader_get_timer (ImageLoader *il)
{
	ImageLoaderPrivateData *priv;

	g_return_val_if_fail (il != NULL, 0);
	priv = il->priv;

	return priv->timer;
}


void
image_loader_load_from_pixbuf_loader (ImageLoader *il,
				      GdkPixbufLoader *pixbuf_loader)
{
	g_return_if_fail (il != NULL);

	image_loader_sync_pixbuf (il, pixbuf_loader);
	gtk_signal_emit (GTK_OBJECT (il), image_loader_signals[DONE]);
}
