/* gtl_extend.c - Routines that handle control extensions
 *
 * Copyright (C) 1998  Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *   Common routines for handling the extension data for control messages.
 *
 * $Log: gtl_extend.c,v $
 * Revision 1.1  1998/05/01 16:37:58  zappo
 * Initial revision
 *
 */

#include "gtalklib.h"
#include "otalk.h"
#include "talk.h"
#include "gtalk.h"
#include "gtl_union.h"


/*
 * Function: EXT_clear_extension_data
 *
 *   Clear out any data we may have in the current extension data.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   4/30/98    Created
 */
void EXT_clear_extension_data()
{
  ControlExtension.element_num = 0;
}  


/*
 * Function: EXT_extension_length
 *
 *   Return the total length of the currently accumulated extension.
 *
 * Returns:     int * - The length
 * Parameters:  None
 *
 * History:
 * zappo   4/30/98    Created
 */
int EXT_extension_length()
{
  int i;
  int length = 4;
  CTL_MSG_EXTENSION_ELEMENT *search = 
    (CTL_MSG_EXTENSION_ELEMENT *)ControlExtension.data;

  if(ControlExtension.element_num == 0) return 0;

  for(i = 0; ((i < ControlExtension.element_num) &&
	      (length < sizeof(ControlExtension)));
      i++)
    {
      length += search->length + 2;
    }

  return length;
}

/*
 * Function: EXT_find_element_offset
 *
 *   Locally defined function which finds a ctl message extension
 * element structure inside ControlExtension of the given type. 
 *
 * Returns:     static CTL_MSG_EXTENSION_ELEMENT * - Element found or NULL
 * Parameters:  type   - Type of element to return a structure to.
 *              new    - If True, then return space for a new element.
 *              extend - The extension data to search in.
 * History:
 * zappo   4/30/98    Created
 */
static CTL_MSG_EXTENSION_ELEMENT *EXT_find_element_offset(type, new, extend)
     u_char type;
     u_char new;
     CTL_MSG_EXTENSION *extend;
{
  int i;
  CTL_MSG_EXTENSION_ELEMENT *search = 
    (CTL_MSG_EXTENSION_ELEMENT *)extend->data;
  CTL_MSG_EXTENSION_ELEMENT *found = NULL;

  /* Loop across each element */
  for(i = 0; ((i < ControlExtension.element_num) &&
	      !found &&
	      (((char *)search-(char *)extend) < sizeof(ControlExtension)));
      i++)
    {
      if(search->type == type)
	{
	  found = search;
	}
      /* Advance over this element into the next element */
      search = (CTL_MSG_EXTENSION_ELEMENT *)
	((u_char *)search + 2 + search->length);
    }
  /* If we found nothing, and new is TRUE, then we really did find
   * something to return the the user */
  if(!found && new)
    found = search;

  return found;
}
/*
 * Function: EXT_add_extension_data
 *
 *   Add DATA into the the control extension currently being built.
 * We depend on TYPE to figure out what kind of data it really is.
 *
 * Returns:     Nothing
 * Parameters:  type - Type of the data.
 *              data - Opaque pointer to data.
 * History:
 * zappo   4/30/98    Created
 */
void EXT_add_extension_data(type, data)
     u_char type;
     void  *data;
{
  CTL_MSG_EXTENSION_ELEMENT *new = 
    EXT_find_element_offset(type, 1, &ControlExtension);

  new->type = type;
  ControlExtension.element_num++;

  switch(type)
    {
      /* These are both strings, so we can double up the implementation */
    case EXTENSION_APPLICATION_NAME:
    case EXTENSION_PERSONAL_NAME:
      new->length = strlen((char *)data) + 1;
      strcpy((char *)&new->data, (char *)data);
      break;
    default:
      /* Silenetly fail: This should never happen */
      ControlExtension.element_num--;
      printf("Cannot add an extension: Unknown type!\n");
    }
}


/*
 * Function: EXT_fetch_extension_data
 *
 *   Return opaque data of TYPE from the current control message.
 * NULL means that it doesn't exist in the current message.
 *
 * Returns:     Nothing
 * Parameters:  type   - Type of
 *              extend - Extension data to look in.
 *              length - Length of the fetched data.
 * History:
 * zappo   4/30/98    Created
 */
void *EXT_fetch_extension_data(type, extend, length)
     u_char type;
     void *extend;
     int  *length;
{
  CTL_MSG_EXTENSION_ELEMENT *new = 
    EXT_find_element_offset(type, 0, (CTL_MSG_EXTENSION *)extend);  

  if(new)
    {
      *length = new->length;
      switch(type)
	{
	  /* These are both strings, so we can double up the implementation */
	case EXTENSION_APPLICATION_NAME:
	  return (void *)new->data.appname.application_name;
	case EXTENSION_PERSONAL_NAME:
	  return (void *)new->data.personalname.personal_name;
	default:
	  return NULL;
	}
    }
  else
    {
      return NULL;
    }
}
