/*
 * Copyright (C) 1994, 95, 96, 97, 98, 99 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * etalk.c
 *
 * Purpose:
 *  This file contains the startup code for initializing the udp and
 * connection sockets.  It then starts up main_loop which waits on all
 * registered file descriptors.
 *
 * $Log: gtalk.c,v $
 * Revision 1.39  1999/11/29 17:06:07  zappo
 * Converted old crypt code into generic filter code.
 *
 * Revision 1.38  1998/09/22 13:28:10  zappo
 * Acced a --defaultroute parameter for setting all outbound packets to
 * use a different IP address.
 *
 * Revision 1.37  1998/04/26 15:06:05  zappo
 * Moved the signal handlers after the X init.  This lets me override the
 * handlers for fiesty widget sets.
 *
 * Revision 1.36  1998/01/09 22:33:42  zappo
 * Pass in the non-flag parameter to the test command.
 *
 * Revision 1.35  1997/12/15 01:08:49  zappo
 * Added auto-answer command line parameter
 *
 * Revision 1.34  1997/12/14 19:17:52  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.33  1997/12/02 22:51:53  zappo
 * Added check to print smart error message if I can't find the local host.
 *
 * Revision 1.32  1997/11/05 03:43:53  zappo
 * Added call to a system rc file
 *
 * Revision 1.31  1997/10/25 01:48:17  zappo
 * Fixed edit char determination when there is an stty failure
 *
 * Revision 1.30  1997/10/17 02:28:50  zappo
 * Initialize default_encrypt in talkcontext.
 *
 * Revision 1.29  1997/10/15 02:09:09  zappo
 * Made sure all io names are malloced space.
 *
 * Revision 1.28  1997/10/07 00:26:45  zappo
 * Fixed up initial output to also show how to get the copyright.
 *
 * Revision 1.27  1997/10/03 23:22:23  zappo
 * Fixed determination of edit characters.  If they come out as NULL
 * use sane defaults.  (Useful when run in the debugger.)
 *
 * Revision 1.26  1997/07/22 13:22:28  zappo
 * Added param to enable signal handling
 *
 * Revision 1.25  1997/03/12 01:32:43  zappo
 * Changed call to ETC test so return value translates to shell ret value.
 *
 * Revision 1.24  1997/03/12 00:22:15  zappo
 * Added sighandlers for fatal errors which dump core, but still disables
 * the UI and cleans up talk packets.
 *
 * Revision 1.23  1997/02/23 03:24:03  zappo
 * Added initialization of cleanup status.  Changed shutdown to do
 * PROTOCOL_delete_all after ringer delete and child killing because it
 * is prone to failure sometimes, and the other aspects are much more important.
 *
 * Revision 1.22  1997/02/20 02:30:54  zappo
 * On Red Hat Linux the personal name comes through with commas for the
 * office, etc.  Seek and destroy the commas to get the name sans office.
 *
 * Revision 1.21  1997/02/01 14:33:04  zappo
 * Added new parameter --no-init-file to prevent reading of ~/.etalkrc
 *
 * Revision 1.20  1996/02/26  00:24:47  zappo
 * Added a cast to bzero a sun compiler didn't like
 *
 * Revision 1.19  1996/02/01  02:34:07  zappo
 * Changes reflect mods in subprocess management
 *
 * Revision 1.18  1996/01/06  02:30:00  zappo
 * Updated calls to ringer library to include username
 *
 * Revision 1.17  1995/12/10  00:23:01  zappo
 * Removed syslogdebug variable, updated finding of login name, and added
 * find for personal name.  Now uses many methods including passwd file,
 * and environment variables.
 *
 * Revision 1.16  1995/11/26  23:59:51  zappo
 * Initialize Ctxt.querynewtcp. emacs now displays init messages
 *
 * Revision 1.15  1995/11/21  03:47:02  zappo
 * Added inits for our init-script, download-path, and updated shutdown
 * sequence to check for running processes we need to kill
 *
 * Revision 1.14  1995/09/29  08:42:46  zappo
 * Moved connecting to Ctxt.tty into blocks related to specific
 * interfaces, and now init remote_connect to DATA_link_accept
 *
 * Revision 1.13  1995/09/22  13:48:40  zappo
 * Added codes needed to optionally have the X toolkit, or curses, or
 * both interfaces removed from the binary.
 *
 * Revision 1.12  1995/09/20  23:21:44  zappo
 * Added -c [--curses] flag, for forcing curses mode, updated for new
 * structures, updated gtalk_shutdown to use perror when necessary.
 *
 * Revision 1.11  1995/09/14  11:32:10  zappo
 * Added real edit keys read from environment when in curses/X mode.
 * Added reply flag, and subprocess flag.  Emacs must now supply a "-s"
 * when starting etalk to put it into subprocess mode.  It can now grap
 * parameters for username@place tty, and updated --help output, and the
 * name is now auto-created from password file.  Also added display init
 * (when not in subprocess mode).  Lastly, shutdown now takes a parameter
 * which is printed AFTER the display is shutdown for error messages.
 *
 * Revision 1.10  1995/07/16  15:42:25  zappo
 * verbosity now increments inself which each occurance of "-v"
 *
 * Revision 1.9  1995/05/09  23:48:13  zappo
 * reduced comment size, and added --copyright parameter
 *
 * Revision 1.8  1995/03/30  02:34:11  zappo
 * Signal handler now takes a dynamic type from autoconf
 *
 * Revision 1.7  1995/03/25  03:22:50  zappo
 * Updated copyright notice to 1995.
 *
 * Revision 1.6  1995/03/23  02:50:56  zappo
 * Help text wasn't perfectly lined up.
 *
 * Revision 1.5  1995/03/04  14:45:12  zappo
 * Added syslogdebug variable needed to link against etalklib
 *
 * Revision 1.4  1995/02/11  17:47:55  zappo
 * Added signal handling to prevent ringer files from being left around
 * when someone ungracefully terminates etalk.
 *
 * Revision 1.3  1995/01/28  16:12:32  zappo
 * Removed some command line options, and setup new ringer socket.
 *
 * Revision 1.2  1994/10/11  00:32:00  zappo
 * Initial GETOPT version.
 *
 * Revision 1.1  1994/08/29  23:36:53  zappo
 * Initial revision
 *
 * History:
 * eml 8/17/94
 * Added call to check for sizes of net messages.  Will serve as a
 * preliminary check to make everything is kool.
 *
 * ::Header:: gtalkc.h 
 */
#include "getopt.h"
#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"
#include "gtl_union.h"

#if HAVE_PWD_H == 1
#include <pwd.h>
#endif

#if HAVE_SIGNAL_H == 1
#include <signal.h>
#endif

#if HAVE_TERMIOS_H == 1
#include <termios.h>
#endif

int verbose     = FALSE;	/* verbosity variable            */
struct TalkContext Ctxt;	/* context of this talk instance */

static RETSIGTYPE gtalk_sighandle(); /* signal handler function  */
static RETSIGTYPE gtalk_faulthandle(); /* signal handler function  */
static int caught_signal = FALSE; /* flag set by signal handler. */

int main(argc, argv)
     int   argc;
     char *argv[];
{
  int local_test_flag = FALSE;	/* testing flag                        */
  int reply_now = FALSE;	/* replying flag                       */
  int read_init = TRUE;		/* read the init file ~/.etalkrc       */
  int auto_answer = FALSE;	/* should we start in auto-answer mode */
  int stuff_to_do = TRUE;	/* looping flag                        */
  int do_sighandlers = TRUE;	/* install signal handlers             */
  char *cmd_user = NULL, *cmd_tty = NULL; /* optional cmd line args    */
  char *defaultroute = NULL;	/* default route cmd line parameter    */
  struct passwd *pw;		/* password info for user name         */
  struct termios mode;		/* terminal info for edit chars        */

  printf("%s\n", GTALK_);

  if(DMN_check_compile() == Fail)
    /* Return error status on this check. */
    return 1;

  /* Default value of runstate. */
#ifdef X_TOOLKIT
  Ctxt.runstate = X;
#else
#ifdef USE_CURSES
  Ctxt.runstate = Curses;
#else
  Ctxt.runstate = Socket;
#endif /* USE_CURSES */
#endif /* X_TOOLKIT */
  /* Here is a chunk from the GNU getopt manual page with the correct
   * extensions for etalk.
   */
  while(stuff_to_do)
    {
      int option_index = 0;	/* options returned index */
      int c;			/* return value           */
#ifdef USE_CURSES
      static char *short_options = "?aCcdhqrstvn";
#else
      static char *short_options = "?aCdhqrstvn";
#endif
      static struct option long_options[] =
	{
	  { "auto-answer",  FALSE, NULL, 'a' },
	  { "copyrite",     FALSE, NULL, 'C' },
#ifdef USE_CURSES			 
	  { "curses",       FALSE, NULL, 'c' },
#endif					 
	  { "defaultroute", 1,     NULL, 'd' },
	  { "help",         FALSE, NULL, 'h' },
	  { "no-init-file", FALSE, NULL, 'q' },
	  { "no-sighandle", FALSE, NULL, 'n' }, /* undocumented */
#ifndef NO_IFACE			 
	  { "reply",        FALSE, NULL, 'r' },
#endif					 
	  { "subprocess",   FALSE, NULL, 's' },
	  { "test",         FALSE, NULL, 't' },
	  { "verbose",      FALSE, NULL, 'v' },
	  { "version",      FALSE, NULL,  0 },
	  { NULL,           FALSE, NULL,  0 }
	};

      c = getopt_long(argc, argv, short_options,
		      long_options, &option_index);

      switch(c) {
      case -1:
	/* The option list is over, look for user@host [tty] to so
	 * long as we aren't using one of the other forms...
	 */
	if(optind < argc)
	  cmd_user = argv[optind++];
	if(optind < argc)
	  cmd_tty = argv[optind++];

	stuff_to_do = FALSE;

	break;
      case 0:
	/* version is the only weird one, and it has already been
	 * printed, so just exit
	 */
	exit(0);
      case 'a':
	auto_answer = TRUE;
	break;
      case 'C':
	CW_display_copywrite();
	exit(0);
      case 'h':
      case '?':
	printf(" etalk --version\n");
	printf(" etalk --copyright\n");
	printf(" etalk [--no-init-file] [--verbose]+ --test\n");
	printf(" etalk --subprocess [--no-init-file] [--verbose]+\n");
#ifdef USE_CURSES
	printf(" etalk --reply [--no-init-file] [--curses]\n");
#else
#ifdef X_TOOLKIT
	printf(" etalk --reply [--no-init-file]\n");
#endif
#endif
#ifdef USE_CURSES
	printf(" etalk [--curses] [--no-init-file] user@host [tty]\n");
#else
#ifdef X_TOOLKITn
	printf(" etalk [--no-init-file] user@host [tty]\n");
#endif
#endif
	printf("Etalk command line options:\n");
	printf(" --auto-answer\t-a\tStart etalk in auto-answer mode.\n");
	printf(" --copyright\t-C\tDisplay copyright information.\n");
#ifdef USE_CURSES
	printf(" --curses\t-c\tDefault to Curses mode.\n");
#endif
	printf(" --defaultroute\t-d addr\tSet the default route.\n");
	printf(" --help\t\t-h,-?\tPrint this help.\n");
	printf(" --no-init-file\t-q\tDo not read ~/.etalkrc file at startup.\n");
	printf(" --test\t\t-t\tRun system verification test.\n");
#ifndef NO_IFACE
	printf(" --reply\t-r\tReply to last talk request.\n");
#endif
	printf(" --subprocess\t-s\tRun as a subprocess (under emacs).\n");
	printf(" --verbose\t-v\tPrint debugging information as we go.\n");
	printf("\t\t\tMultiple -v parameters increase debug information.\n");
	printf(" --version\t\tPrint version and exit.\n\n");
	exit(0);
      case 'q':
	read_init = FALSE;
	break;
#ifdef USE_CURSES
      case 'c':
	Ctxt.runstate = Curses;
	break;
#endif
      case 'd':
	ROUTE_update(&Ctxt, "0.0.0.0", argv[optind++]);
	break;
      case 'n':
	do_sighandlers = FALSE;
	break;
#ifndef NO_IFACE
      case 'r':
	reply_now = TRUE;
	break;
#endif
      case 's':
	Ctxt.runstate = Socket;
	break;
      case 't':
	/* When testing ALWAYS assume that the runstate is socket, otherwise
	 * the output will be messy!
	 */
	local_test_flag = TRUE;
	Ctxt.runstate = Socket;
	break;
      case 'v':
	verbose++;
	printf("Verbosity Set to %d.\n", verbose);
	break;
      default:
	printf("Illegal option %c\n", c);
	exit(0);
      }
    }

  if(Ctxt.runstate == Socket) {
    printf("Type \"show copyright\" for copyright information.\n");
  } else {
    printf("Type \"ESC x show copyright\" for copyright information.\n");
  }

#ifndef NO_IFACE
  /*
   * Error check combinations of command line parameters
   */
  if(reply_now && (Ctxt.runstate == Socket))
    {
      fprintf(stderr, "Cannot command line reply in socket mode.\n");
      fprintf(stderr, "Type %s --help for more details.\n", argv[0]);
      exit(1);
    }
  if(reply_now && cmd_user)
    {
      fprintf(stderr, "Cannot use auto-reply mode and specify a user simultaneously.\n");
      fprintf(stderr, "Type %s --help for more details.\n", argv[0]);
      exit(1);
    }
  if(verbose && (Ctxt.runstate == Curses))
    {
      fprintf(stderr, "Verbosity not allowed in CURSES mode.\n");
      verbose = 0;
    }
#endif

  /* Find my username as best I can! */
  Ctxt.myname        = getlogin();
  if(!Ctxt.myname)
    {
      Ctxt.myname    = getenv("LOGNAME");
      if(!Ctxt.myname)
	{
	  Ctxt.myname  = getenv("USER");
	  if(!Ctxt.myname)
	    {
	      fprintf(stderr, "I can't figure out who you are!");
	      return 1;
	    }
	}
    }
  /* No matter how we get the name, strdup it for safety */
  Ctxt.myname = strdup(Ctxt.myname);
  if(!Ctxt.myname)
    {
      fprintf(stderr, "strdup failure allocating name\n");
      return 1;
    }

  /* Now try to find my preferred name, with many ways to find it */
  pw = getpwnam(Ctxt.myname);
  if(!pw)
    {
      Ctxt.myprefname = getenv("NAME");
      if(!Ctxt.myprefname)
	{
	  Ctxt.myprefname = Ctxt.myname;
	  if(!Ctxt.myprefname)
	    {
	      fprintf(stderr, "Error duplicating preferred name!\n");
	      return 1;
	    }
	  /* If no environment either, we use loginname,
	   * and capitalize it! */
	  if((Ctxt.myprefname[0] <= 'z') && (Ctxt.myprefname[0] >= 'a'))
	    Ctxt.myprefname[0] += 'A' - 'a';
	}
    }
  else
    {
      char *s;

      s = pw->PW_FULL_NAME;

      while(*s && (*s != ',')) s++; /* find a comma found on some systems */

      if(*s) {
	/* if we found a ',' we have to temporarilly remove it and put it
	 * back.  In some future revision we might need it.
	 */
	*s = '\0';
	Ctxt.myprefname = strdup(pw->PW_FULL_NAME);
	*s = ',';
      } else {
	Ctxt.myprefname = strdup(pw->PW_FULL_NAME);
      }

      if(!Ctxt.myprefname)
	{
	  fprintf(stderr, "Error duplicating preferred name!\n");
	  return 1;
	}
    }
  /* No matter how we get our preferred name, strdup it for safety */
  Ctxt.myprefname = strdup(Ctxt.myprefname);
  if(!Ctxt.myprefname)
    {
      fprintf(stderr, "Error strduping preferred name\n");
      return 1;
    }

  Ctxt.myappname       = NULL;
  Ctxt.downloadpath    = strdup("/tmp");
  Ctxt.message_delay   = 30;
  Ctxt.ringerflag      = 0;
  Ctxt.querynewtcp     = QueryConnect;
  Ctxt.cleanup         = AutoClean;
  Ctxt.default_filter  = 0;

  /* Determine what our edit keys are.  The following pieces of code
   * and terminal knowledge was taken directly from the GNU stty
   * program
   */
#ifndef DONT_USE_GNU_STTY_CODE
  /* Initialize to all zeroes so there is no risk memcmp will report a
     spurious difference in an uninitialized portion of the structure.  */
#if HAVE_TERMIOS_H == 1
  bzero ((void*)&mode, sizeof (mode));
  if (tcgetattr (0, &mode))
    {
      fprintf(stderr, "Could not load terminal information...guessing edit chars\n");
#endif
      /* Setup edit keys to defaults if termios fails, or doesn't exist */
      Ctxt.editkeys[0] = '';
      Ctxt.editkeys[1] = '';
      Ctxt.editkeys[2] = '';
#if HAVE_TERMIOS_H == 1
    }
  else
    {
       /* character erase */
      if(mode.c_cc[VERASE] == '\0')
	Ctxt.editkeys[0] = '';
      else
	Ctxt.editkeys[0] = mode.c_cc[VERASE];

      /* line erase       */
      if(mode.c_cc[VKILL] == '\0')
	Ctxt.editkeys[1] = '';
      else
	Ctxt.editkeys[1] = mode.c_cc[VKILL];

#ifdef VWERASE
      /* word erase       */
      if(mode.c_cc[VWERASE] != '\0')
	Ctxt.editkeys[2] = mode.c_cc[VWERASE];
      else
#endif
	Ctxt.editkeys[2] = '';
    }
#endif /* HAVE_TERMIOS_H */
#endif /* DONT_USE_GNU_STTY_CODE */


  /* Get this process id for the daemon. */
  Ctxt.pid            = getpid();

  /* Load in hostnames here.  Errors print to stdout */
  RC_load_all_hosts(&Ctxt);

#ifndef NO_IFACE
  if(Ctxt.runstate != Socket)
    {
      DISP_init(&Ctxt);
    }
#endif

  /* Some libraries (GTK) like to install signal handlers for us.
   * Make sure we install ours AFTER initializing the display
   * to overcome our evil foe.
   */
  if(do_sighandlers) {
    signal(SIGHUP, gtalk_sighandle);
    signal(SIGINT, gtalk_sighandle);
    signal(SIGKILL, gtalk_sighandle);
    signal(SIGTERM, gtalk_sighandle);
    /* Need to handle errors so that we don't screw up terminal settings. */
#ifdef SIGFPE
    signal(SIGFPE, gtalk_faulthandle);
#endif
    signal(SIGILL, gtalk_faulthandle);
    signal(SIGSEGV, gtalk_faulthandle);
    signal(SIGBUS, gtalk_faulthandle);
#ifdef SIGPWR
    signal(SIGPWR, gtalk_sighandle); /* power shortage signal, Yikes */
#endif
  }

  DISP_message(&Ctxt, "\03Initializing Network Interface...");

  Ctxt.me             = HOST_gen_local_host();
  if(Ctxt.me == NULL) {
    gtalk_shutdown("etalk: cannot run without an IP address.");
    /* Does not return */
  }

#ifndef NO_IFACE
  if(Ctxt.runstate == Socket)
    {
#endif
      /* Set up a TTY input object. */
      Ctxt.tty            = GT_tty();

      Ctxt.tty->readme    = GTC_parse_command;

      Ctxt.emacs_connect  = TCP_listen(); /* we must set host by hand */
      Ctxt.emacs_connect->name = strdup("emacs_connect");
      Ctxt.emacs_connect->host = Ctxt.me;
      Ctxt.emacs_connect->readme = LOCAL_new_tcp;
      Ctxt.emacs_connect->timeout = 0;

      printf("\03%c%d\n", TTY_LOCALPORT,
	     ntohs(Ctxt.emacs_connect->raddr.sin_port));
#ifndef NO_IFACE
    }
  else if (Ctxt.runstate == Curses)
    {
      /* Don't allow verbose during CURSES app execution. */
      verbose = FALSE;
    }
#endif

  Ctxt.remote_connect = TCP_listen();
  Ctxt.remote_connect->name = strdup("remote_connect");
  Ctxt.remote_connect->host = Ctxt.me;
  Ctxt.remote_connect->readme = DATA_link_accept;
  Ctxt.remote_connect->timeout = 0;

  if(Ctxt.runstate == Socket)
    {
      printf("\03%c%d\n", TTY_REMOTEPORT,
	     ntohs(Ctxt.remote_connect->raddr.sin_port));
    }

  /* Set up ringer port.  Reset UDP creators in between. */
  UDP_setup_localport();
  Ctxt.udp_ring           = UDP_host(Ctxt.me->name);
  Ctxt.udp_ring->name     = strdup("ring_port");
  UDP_reset_ports();

  UDP_setup_localport();
  /* Set up local UDP port for discussion over talk protocol */
  Ctxt.udp_port           = UDP_host(Ctxt.me->name);
  Ctxt.udp_port->name     = strdup("udp_port");
  Ctxt.local_daemon       = UDP_host(Ctxt.me->name);
  Ctxt.local_daemon->name = strdup("local_daemon");
  
  DISP_message(&Ctxt, "\03Initializing Network Interface...done");

  /* Now, once our state has reached run-time equivalence, load
   * in the init script!  A NULL means use default.
   */
  if(read_init == TRUE) {
#ifdef SYSTEM_RC
    RC_read_script(&Ctxt, SYSTEM_RC);
#endif
    RC_read_script(&Ctxt, NULL);
  }

  /* Once this is read, we might override a few options... */
  if(auto_answer) 
    {
      RING_activate(Ctxt.myname, &Ctxt.ringerflag,
		    Ctxt.udp_ring, ETR_read);
    }

  if(local_test_flag == TRUE)
    {
      /* Run the test to see what's going on! */
      return (GTC_test(&Ctxt, cmd_user) == Fail);
    }
  else
    {
#ifndef NO_IFACE
      /* Check for the user parameter to initiate a call outwards. */
      if((Ctxt.runstate != Socket) && cmd_user)
	{
	  LOCAL_new_user_tty(&Ctxt, cmd_user, cmd_tty);
	}
      else if(reply_now)
	{
	  PROTOCOL_reply(&Ctxt);
	}
      else if(Ctxt.runstate != Socket)
	{
	  DISP_message(&Ctxt, "Press ESC - h for help");
	}
#endif
      /* Select loop.  This should be the only occurance with NULL as the
       * reton parameter...
       */
      GT_select_all(&Ctxt, NULL);
    }

  /* If we get here, then there is some sort of error.
   */
  return 1;
}


/*
 * Function: gtalk_sighandle
 *
 *   Etalk shutdown handler.  If any unpleasant signals are trapped,
 * make sure things are cleaned up whenever possible.
 *
 * Parameters:  val - Number of val
 *
 * History:
 * zappo   2/11/95    Created
 */
static RETSIGTYPE gtalk_sighandle(val)
     int val;
{
  caught_signal = 1;

  gtalk_shutdown(NULL);

  exit(val);
}
static RETSIGTYPE gtalk_faulthandle(val)
     int val;
{
  caught_signal = 1;

  gtalk_shutdown(NULL);

  fprintf(stderr, "Fatal Signal: Dumping core.\n");
  abort();
}


/*
 * Function: gtalk_shutdown
 *
 *   Run all necessary shutdown procedures, then exit the program.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   2/11/95    Created
 */
void gtalk_shutdown(char *message)
{
  FORK_kill_children(&Ctxt);
#ifndef NO_IFACE
  /* If we are running X, then we must disable X or unpleasant things
   * may happen if there are any events left in the X Q */
  if(Ctxt.runstate != Socket)
    {
      DISP_close(&Ctxt);
    }
#endif
  RING_deactivate(Ctxt.myname);
  /* Do protocol deletions last because this sometimes fails.  It is more
   * important to accomplish Ringer deactivation and child slaying.
   */
  PROTOCOL_delete_all(&Ctxt, TRUE);

  GT_close_all(&Ctxt, NULL);

  if(message)
    {
      perror("Shutdown complete");
      printf(message);
    }

  if(caught_signal == 0) exit(0);
}
