/* gt_route.c - Routing for dynamic ip & multiple net devices
 *
 * Copyright (C) 1998 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.ai.mit.edu.
 * 
 * Description:
 *   Manage routes needed when preparing talk packets.  Useful for
 * dynamic IP based machines, or machines with multiple network
 * devices such as firewalls.
 * 
 * History:
 * zappo   9/8/98     Created
 *
 * Tokens: ::Header:: gtalkc.h
 * $Log$
 */
#include "gtalklib.h"
#include "gtalkc.h"
#include "alloca.h"

static MakeList(list);

/* Route List:
 * 
 * The route list is basically a list of DESTINATIONS and ADDRESSES.
 * A Destination is an IP string.  When an outbound call matches a DESTINATION,
 * then we make sure that the return address is ADDRESS.
 */
struct route_list {
  struct GenericListNode link;	      /* Linked list part                    */
  int                destination[4]; /* destination network address */
  struct HostObject *address;	/* Address used on that network.    */
};


/*
 * Function: ROUTE_update
 *
 *   Create a new route entry, or update an existing route entry.
 *
 * Returns:     int - Success/fail status.
 * Parameters:  Ctxt        - Context
 *              destination - Destination as a string.
 *              address     - Host name as a string.
 * History:
 * zappo   9/8/98     Created
 */
int ROUTE_update(Ctxt, destination, address)
     struct TalkContext *Ctxt;
     char *destination, *address;
{
  struct route_list *new;
  int dest[4];
  struct HostObject *addr;

  /* Scan in the destination.  If it's a work, like 'default', then
   * it magically becomes 0.0.0.0
   */
  sscanf(destination, "%d.%d.%d.%d", &dest[0], &dest[1], &dest[2], &dest[3]);

  addr = HOST_gen_host(address);

  /* Host failure prints error for us. */
  if(!addr)
    {
      return Fail;
    }

  new = ROUTE_find_destination(dest);

  if(!new)
    {
      new = (struct route_list *)LIST_alloc(&list, sizeof(struct route_list));

      if(!new)
	return Fail;

      new->destination[0] = dest[0];
      new->destination[1] = dest[1];
      new->destination[2] = dest[2];
      new->destination[3] = dest[3];
    }

  new->address = addr;

  return Success;
}


/*
 * Function: ROUTE_find_destination, MatchDestinationLiteral
 *
 *   Finds the route entry that matches DESTINATION.
 *
 * Returns:     struct route_list * - 
 * Parameters:  destination - The network destination.
 *
 * History:
 * zappo   9/8/98     Created
 */
static unsigned char MatchDestinationLiteral(rl, c)
     struct route_list *rl;
     int c[4];
{
  return (rl->destination == c);
}
struct route_list *ROUTE_find_destination(destination)
     int destination[4];
{
  return (struct route_list *)LIST_find(&list, MatchDestinationLiteral,
					destination);
}

/*
 * Function: ROUTE_find_destination_masked
 *
 *   Finds a route from the list which matches a destination mask.
 *
 * Returns:     struct route_list * - 
 * Parameters:  destination - The 
 *
 * History:
 * zappo   9/13/98    Created
 */
static unsigned char MatchDestinationMasked(rl, host)
     struct route_list *rl;
     struct HostObject *host;
{
  unsigned char *tuple = (unsigned char *)&rl->address->addr.sin_addr.s_addr;

  return 
    (((rl->destination[0] == 0) || (rl->destination[0] == tuple[0])) &&
     ((rl->destination[1] == 0) || (rl->destination[1] == tuple[1])) &&
     ((rl->destination[2] == 0) || (rl->destination[2] == tuple[2])) &&
     ((rl->destination[3] == 0) || (rl->destination[3] == tuple[3])));
}
struct sockaddr *ROUTE_find_destination_masked(destination)
     struct HostObject *destination;
{
  struct route_list *l = (struct route_list *)LIST_find(&list,
							MatchDestinationMasked,
							destination);
  if(l != NULL)
    return (struct sockaddr *)&l->address->addr;
  else
    return NULL;
}

/*
 * Function: ROUTE_print, ROUTE_print_node
 *
 *   Prints out nodes in the routing list.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   9/8/98     Created
 */
static void ROUTE_print_node(rl, Ctxt)
     struct route_list *rl;
     struct TalkContext *Ctxt;
{
  char buffer[200];
  unsigned char *tuple = (unsigned char *)&rl->address->addr.sin_addr.s_addr;

  if(rl->destination[0] || rl->destination[1] || rl->destination[2] ||
     rl->destination[3])
    {
      sprintf(buffer, "%d.%d.%d.%d \t%s (%d.%d.%d.%d)",
	      rl->destination[0], rl->destination[1],
	      rl->destination[2], rl->destination[3],
	      rl->address->name,
	      tuple[0], tuple[1], tuple[2], tuple[3] );
    } else {
      sprintf(buffer, "default\t\t%s  (%d.%d.%d.%d)", rl->address->name,
	      tuple[0], tuple[1], tuple[2], tuple[3] );
    }
  DISP_message(Ctxt, buffer);
}
void ROUTE_print(Ctxt)
     struct TalkContext *Ctxt;
{
  if (FirstElement(list) != NULL)
    {
      DISP_message(Ctxt, "Network Mask\tLocal address");
      LIST_map(&list, ROUTE_print_node, Ctxt);
    }
  else
    {
      DISP_message(Ctxt, "No routing filters active.");
    }
}

/*
 * Function: ROUTE_number
 *
 *   Returns the number of route structures
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   8/25/95    Created
 */
int ROUTE_number()
{
  return LIST_map(&list, NULL, NULL);
}


