/*
 * Groach 'theme' module
 * Refer to theme.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <sys/stat.h>
#include <fcntl.h>

#include <gnome.h>
#include <gmodule.h>
#include "grotype.h"
#include "gromove.h"
#include "theme.h"
#include "gropixmap.h"
#include "theme-default.h"


/* Data structure definitions */
struct _GroThemePrivate {
	gint ref_count;	/* Reference count
					   Initial value is zero. It's different from Gtk+ object.
					   'theme' is usually owned by controller.
					   The controller takes care of reference count. */
	/* Hash table of GroPixmap.
	   Key is a file name.
	   GroPixmap is expensive, so I try to avoid duplication. */
	GHashTable *gropixmap_htable;
	GModule *module;

	/** parser temporary data **/
	/* the following data are allocated during parsing 'theme' file,
	   and free'd after parsing, i.e. a very short persistence. */
	gint _fd;/* file descriptor */
	GScanner *_scanner;
	/* Hash table of filenames for frames.
	   See a description below. */
	GHashTable *_tmp_fname_htable;

	guint _ns, _nd;/* current number of gstat, direction during parsing. */
};


/* Private function declarations */
static void gro_theme_destroy(GroTheme *theme);
static gboolean load_frames(GroTheme *theme, const char *dirname, ThemeLoadMode lmode);
static gboolean load_modules(GroTheme *theme, const char *dirname);
static void hash_free_key_value(gpointer key, gpointer value, gpointer data);

/* Parse related functions */
static gboolean parse_open(GroTheme *theme, const char *theme_fname);
static void parse_close(GroTheme *theme);
static const char* parse_pixmap_filename(GroTheme *theme, gint gstat, guint dir, guint i_frame);


/**
 * theme_new:
 * Parse theme file, and initialize theme data.
 **/
GroTheme*
gro_theme_load(const char *theme_dirname, ThemeLoadMode lmode)
{
	GroTheme *theme;
	GroThemePrivate *privat;
	char *theme_filename;
	gboolean b_found;
	int ns;

	g_return_val_if_fail(theme_dirname != NULL && theme_dirname[0] != '\0', NULL);
	
	theme = g_new(GroTheme, 1);
	privat = g_new(GroThemePrivate, 1);
	theme->privat = privat;

	/* Set default values */
	theme->name = NULL;
	theme->description = NULL;
	theme->author = NULL;
	theme->module_name = NULL;
	theme->num_gstat = DEFAULT_NUM_GSTAT;
	theme->num_direct = DEFAULT_NUM_DIRECT;
	theme->pix_width = DEFAULT_PIX_WIDTH;
	theme->pix_height = DEFAULT_PIX_HEIGHT;
	theme->step_pixel = DEFAULT_STEP_PIXEL;
	theme->theme_init = NULL;
	theme->move_init = move_init;
	theme->move_compute = move_compute;
	theme->move_finalize = move_finalize;
	theme->theme_finalize = NULL;
	privat->ref_count = 0;/* initial value is zero */
	if (lmode == THEME_LOAD_ONLYINFO) {
		privat->gropixmap_htable = NULL;
	} else {
		privat->gropixmap_htable = g_hash_table_new(g_str_hash, g_str_equal);
	}
	privat->module = NULL;
	
	/* Open 'theme' file, and read it */
	theme_filename = g_concat_dir_and_file(theme_dirname, THEME_FILE_NAME);
	/* The values in the file override the default values. */
	if (parse_open(theme, theme_filename) == FALSE) {
		/* Hmm... failure, what should I do? todo: memory...*/
		goto ERROR;
	}
	g_free(theme_filename);

	if (lmode == THEME_LOAD_ONLYINFO) {
		theme->num_gstat = 0;
		theme->num_direct = 0;
		theme->frames = NULL;
	} else {
		theme->frames = g_new(GPtrArray**, theme->num_gstat);
		for (ns = 0; ns < theme->num_gstat; ns++) {
			theme->frames[ns] = g_new(GPtrArray*, theme->num_direct);
		}
		
		b_found = load_frames(theme, theme_dirname, lmode);
		if (b_found == FALSE) {
			goto ERROR;
		}
		/* Load dynamic module if available */
		if (theme->module_name) {
			load_modules(theme, theme_dirname);
		}
	}
	
	/* Parser internal data are free'd */
	parse_close(theme);

	return theme;

 ERROR:
	/* todo: memory... */
	return NULL;
}

/**
 * gro_theme_ref:
 **/
void
gro_theme_ref(GroTheme *theme)
{
	GroThemePrivate *privat;

	g_return_if_fail(theme != NULL);

	privat = theme->privat;
	privat->ref_count++;
}

/**
 * gro_theme_unref:
 * if the reference cound drops to zero, destroy it.
 **/
void
gro_theme_unref(GroTheme *theme)
{
	GroThemePrivate *privat;

	g_return_if_fail(theme != NULL);

	privat = theme->privat;

	g_return_if_fail(privat->ref_count > 0);
	
	privat->ref_count--;
	if (privat->ref_count == 0)
		gro_theme_destroy(theme);
}


/* ---The followings are private functions--- */
/**
 * gro_theme_destroy:
 **/
static void
gro_theme_destroy(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	int ns, nd;

	if (privat->module) {
		g_module_close(privat->module);
	}
	for (ns = 0; ns < theme->num_gstat; ns++) {
		for (nd = 0; nd < theme->num_direct; nd++) {
			if (theme->frames[ns][nd] == NULL)
				continue;
			/* These pointers are GroPixmap*.
			   As they are free'd as hash table's values below,
			   don't need to free them here. */
			g_ptr_array_free(theme->frames[ns][nd], TRUE);
		}
		g_free(theme->frames[ns]);
	}
	g_free(theme->frames);

	if (privat->gropixmap_htable) {
		g_hash_table_foreach(privat->gropixmap_htable,
							 hash_free_key_value, NULL);
		g_hash_table_destroy(privat->gropixmap_htable);
	}
	if (theme->module_name)
		g_free(theme->module_name);
	if (theme->author)
		g_free(theme->author);
	if (theme->description)
		g_free(theme->description);
	if (theme->name)
		g_free(theme->name);

	g_free(privat);
	g_free(theme);
}


/**
 * load_frames:
 * Load pixmaps. Not load duplicate ones.
 * Pixmap file name is determined by parse_pixmap_filename().
 * Caller should free the loaded pixmaps.
 * If @lmode is THEME_LOAD_PREVIEW, load small sized pixmaps.
 * Output:
 * Return value; TRUE if success, else FALSE.
 **/
static gboolean
load_frames(GroTheme *theme, const char *dirname, ThemeLoadMode lmode)
{
	GroThemePrivate *privat = theme->privat;
	int ns, nd;
	int i;

	for (ns = 0; ns < theme->num_gstat; ns++) {
		for (nd = 0; nd < theme->num_direct; nd++) {
			theme->frames[ns][nd] = g_ptr_array_new();
			for (i = 0; ; i++) {/* forever until b_found == FALSE */
				const char *fname;
				GroPixmap *gro_pixmap;
				
				fname = parse_pixmap_filename(theme, ns, nd, i);
				if (fname == NULL) {
					if (i == 0)/* no frame */
						goto ERROR;
					break;
				}
				gro_pixmap = g_hash_table_lookup(privat->gropixmap_htable, fname);
				if (gro_pixmap == NULL) {
					guint pix_width = theme->pix_width;
					guint pix_height = theme->pix_height;

					if (lmode == THEME_LOAD_PREVIEW) {
						pix_width = pix_width / 4 * 3;
						pix_height = pix_height / 4 * 3;
					}
					gro_pixmap = gro_pixmap_new(fname, dirname, pix_width, pix_height);
					g_hash_table_insert(privat->gropixmap_htable, g_strdup(fname), gro_pixmap);
				}
				g_ptr_array_add(theme->frames[ns][nd], gro_pixmap);
			}
		}
	}

	return TRUE;
	
 ERROR:
	return FALSE;
}

/**
 * load_modules:
 * Load a module(shared library), and initalize symbols.
 * Output:
 * Return value; TRUE if success, else FALSE.
 **/
static gboolean
load_modules(GroTheme *theme, const char *dirname)
{
	GroThemePrivate *privat = theme->privat;
	GModule *module;
	gpointer tmp_sym;
	char *module_path;

	if (g_module_supported() == FALSE) {
		g_warning(_("Dynamic module loading is not supported."));
		return FALSE;
	}
	module_path = g_concat_dir_and_file(dirname, theme->module_name);
	module = g_module_open(module_path, 0);
	if (module == NULL) {
		g_warning(_("Can't dynamic load: %s."), theme->module_name);
		g_free(module_path);
		return FALSE;
	}
	privat->module = module;
	if (g_module_symbol(module, THEME_INIT_SYM, &tmp_sym))
		theme->theme_init = tmp_sym;
	if (g_module_symbol(module, MOVE_INIT_SYM, &tmp_sym))
		theme->move_init = tmp_sym;
	if (g_module_symbol(module, MOVE_COMPUTE_SYM, &tmp_sym))
		theme->move_compute = tmp_sym;
	if (g_module_symbol(module, MOVE_FINALIZE_SYM, &tmp_sym))
		theme->move_finalize = tmp_sym;
	if (g_module_symbol(module, THEME_FINALIZE_SYM, &tmp_sym))
		theme->theme_finalize = tmp_sym;

	g_free(module_path);
	return TRUE;
}
	
/**
 * hash_free_key_value:
 * Free key and value of GHashTable *gropixmap_htable.
 **/
static void
hash_free_key_value(gpointer key, gpointer value, gpointer data)
{
	g_free(key); /* file name */
	gro_pixmap_delete(value);/* GroPixmap */
}



/* ----The followings are parser related functions---- */
/* Data structure definitions */
/* Description of GHashTable *tmp_fname_htable
 * As I write in several files e.g. gromove.h,
 * I have to manage three dimensional arrays [(gstat, dir, i_frame)].
 * Temporarily, I managed pixmap file names with a hash table such as
 * key = "n1,n2,n3"  :n1,n2,n3 are intergers
 * value = "filename.png"
 * It doesn't sound C, and very inefficient.
 * However, the parse is done once, and the hash table is destroyed immediately.
 * Actually, I don't want to manage dynamic three dimensional arrays.
 * Don't confuse this table with gropixmap_htable.
 * This table is just temporarily used.
 * All you have to remember is that the file names are retrieved via
 * parse_pixmap_filename(). */

/* parse config data */
static const GScannerConfig	gro_theme_config =
{
  (
   " \t\n"
   )			/* cset_skip_characters */,
  (
   G_CSET_a_2_z
   "_"
   G_CSET_A_2_Z
   )			/* cset_identifier_first */,
  (
   G_CSET_a_2_z
   "_-0123456789"
   G_CSET_A_2_Z
   )			/* cset_identifier_nth */,
  ( "#\n" )		/* cpair_comment_single */,
  
  FALSE			/* case_sensitive */,/* _insensitive_ */
  
  TRUE			/* skip_comment_multi */,
  TRUE			/* skip_comment_single */,
  TRUE			/* scan_comment_multi */,
  TRUE			/* scan_identifier */,
  FALSE			/* scan_identifier_1char */,
  FALSE			/* scan_identifier_NULL */,
  TRUE			/* scan_symbols */,
  FALSE			/* scan_binary */,
  FALSE			/* scan_octal */,
  FALSE			/* scan_float */,
  FALSE			/* scan_hex */,
  FALSE			/* scan_hex_dollar */,
  TRUE			/* scan_string_sq */,
  TRUE			/* scan_string_dq */,
  TRUE			/* numbers_2_int */,
  FALSE			/* int_2_float */,
  FALSE			/* identifier_2_string */,
  TRUE			/* char_2_token */,
  TRUE			/* symbol_2_token */,
  FALSE			/* scope_0_fallback */,
};

typedef enum {
	THEME_TOKEN_INVALID = G_TOKEN_LAST,
	THEME_TOKEN_NAME,
	THEME_TOKEN_DESCRIPTION,
	THEME_TOKEN_AUTHOR,
	THEME_TOKEN_MODULE,
	THEME_TOKEN_STATS,
	THEME_TOKEN_DIRECTIONS,
	THEME_TOKEN_PIXMAPSIZE,
	THEME_TOKEN_STEPPIXEL,
	THEME_TOKEN_DATA,
	THEME_TOKEN_LAST
} ThemeTokenType;

static const struct {
	gchar *name;
	ThemeTokenType token;
} symbols[] = {
	{ "name", THEME_TOKEN_NAME },
	{ "description", THEME_TOKEN_DESCRIPTION },
	{ "author", THEME_TOKEN_AUTHOR },
	{ "module", THEME_TOKEN_MODULE },
	{ "stats", THEME_TOKEN_STATS },
	{ "directions", THEME_TOKEN_DIRECTIONS },
	{ "pixmap_size", THEME_TOKEN_PIXMAPSIZE },
	{ "step_pixel", THEME_TOKEN_STEPPIXEL },
	{ "data", THEME_TOKEN_DATA },
};
static const guint n_symbols = sizeof(symbols) / sizeof(symbols[0]);

/* Private in private function declarations */
static void _parse_do(GroTheme *theme);
static guint _parse_statement(GroTheme *theme);
static guint _parse_name(GroTheme *theme);
static guint _parse_description(GroTheme *theme);
static guint _parse_author(GroTheme *theme);
static guint _parse_module(GroTheme *theme);
static guint _parse_stats(GroTheme *theme);
static guint _parse_directions(GroTheme *theme);
static guint _parse_pixmapsize(GroTheme *theme);
static guint _parse_steppixel(GroTheme *theme);
static guint _parse_data(GroTheme *theme);
static guint _parse_frames(GroTheme *theme);
#ifdef DEBUG
static void _parse_verify(const GroTheme *theme);
#endif

static void _parse_hash_frees(gpointer key, gpointer value, gpointer data);


static gboolean
parse_open(GroTheme *theme, const char *theme_fname)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner;
	int fd;
	
	fd = open(theme_fname, O_RDONLY);
	if (fd == -1)
		return FALSE;
	privat->_fd = fd;

	privat->_tmp_fname_htable = g_hash_table_new(g_str_hash, g_str_equal);

	scanner = g_scanner_new((GScannerConfig*)&gro_theme_config);/* suppress warning */
	privat->_scanner = scanner;
	g_scanner_input_file(scanner, fd);
	scanner->input_name = theme_fname;

	privat->_ns = privat->_nd = 0;
	_parse_do(theme);

	return TRUE;
}

static void
parse_close(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;

#ifdef DEBUG
	_parse_verify(theme);
#endif
	
	g_scanner_destroy(privat->_scanner);

	g_hash_table_foreach(privat->_tmp_fname_htable,
						 _parse_hash_frees, NULL);
	g_hash_table_destroy(privat->_tmp_fname_htable);

	close(privat->_fd);
}

/**
 * parse_pixmap_filename:
 * With gstat, dir, i_frame, get pixmap filename from _tmp_fname_htable.
 * I will do the best effort to find it.
 * Output:
 * Return value; static data. NULL if not found.
 **/
static const char*
parse_pixmap_filename(GroTheme *theme, gint gstat, guint dir, guint i_frame)
{
	GroThemePrivate *privat = theme->privat;
	char key[32];
	const char *ret_fname;

	
	g_snprintf(key, 32, "%d,%d,%d", gstat, dir, i_frame);
	ret_fname = g_hash_table_lookup(privat->_tmp_fname_htable, key);
	/* If it's not found in hash table, try to find a default direction. */
	if (ret_fname == NULL) {
		g_snprintf(key, 32, "%d,%d,%d", gstat, 0, i_frame);
		ret_fname = g_hash_table_lookup(privat->_tmp_fname_htable, key);
	}
	
	return ret_fname;
}


/**
 * _parse_do:
 * Follows gtkrc.c of Gtk+.
 * Ignore parse error, just show warning message.
 **/
static void
_parse_do(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	int i;
	gboolean done;
	
	g_scanner_freeze_symbol_table(scanner);
	for (i = 0; i < n_symbols; i++)
		g_scanner_add_symbol(scanner, symbols[i].name, GINT_TO_POINTER(symbols[i].token));
	g_scanner_thaw_symbol_table(scanner);

	done = FALSE;
	while (!done) {
		if (g_scanner_peek_next_token(scanner) == G_TOKEN_EOF)
			done = TRUE;
		else {
			guint expected_token;
	  
			expected_token = _parse_statement(theme);

			if (expected_token != G_TOKEN_NONE) {/* error */
				gchar *symbol_name;
				gchar *msg;
	      
				msg = NULL;
				symbol_name = NULL;
				if (scanner->scope_id == 0) {
					/* if we are in scope 0, we know the symbol names
					 * that are associated with certaintoken values.
					 * so we look them up to make the error messages
					 * more readable. */
					if (expected_token > THEME_TOKEN_INVALID &&
						expected_token < THEME_TOKEN_LAST) {
						for (i = 0; i < n_symbols; i++)
							if (symbols[i].token == expected_token)
								msg = symbols[i].name;
						if (msg)
							msg = g_strconcat("e.g. `", msg, "'", NULL);
					}
					if (scanner->token > THEME_TOKEN_INVALID &&
						scanner->token < THEME_TOKEN_LAST) {
						symbol_name = "???";
						for (i = 0; i < n_symbols; i++)
							if (symbols[i].token == scanner->token)
								symbol_name = symbols[i].name;
					}
				}
				g_scanner_unexp_token(scanner, expected_token, NULL,
									  "keyword", symbol_name, msg, TRUE);
				g_free(msg);
				done = TRUE;
			}
		}
    }
}

static guint
_parse_statement(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
	
	token = g_scanner_peek_next_token(scanner);

	switch (token) {
	case THEME_TOKEN_NAME:
		return _parse_name(theme);
	case THEME_TOKEN_DESCRIPTION:
		return _parse_description(theme);
	case THEME_TOKEN_AUTHOR:
		return _parse_author(theme);
	case THEME_TOKEN_MODULE:
		return _parse_module(theme);
	case THEME_TOKEN_STATS:
		return _parse_stats(theme);
	case THEME_TOKEN_DIRECTIONS:
		return _parse_directions(theme);
	case THEME_TOKEN_PIXMAPSIZE:
		return _parse_pixmapsize(theme);
	case THEME_TOKEN_STEPPIXEL:
		return _parse_steppixel(theme);
	case THEME_TOKEN_DATA:
		return _parse_data(theme);
    default:
		g_scanner_get_next_token(scanner);
		return THEME_TOKEN_DATA;/* something toplevel token */
	}
}

static guint
_parse_name(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_NAME)
		return THEME_TOKEN_NAME;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_STRING)
		return G_TOKEN_STRING;
	theme->name = g_strdup(scanner->value.v_string);
  
	return G_TOKEN_NONE;
}

static guint
_parse_description(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_DESCRIPTION)
		return THEME_TOKEN_DESCRIPTION;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_STRING)
		return G_TOKEN_STRING;
	theme->description = g_strdup(scanner->value.v_string);
  
	return G_TOKEN_NONE;
}

static guint
_parse_author(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_AUTHOR)
		return THEME_TOKEN_AUTHOR;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_STRING)
		return G_TOKEN_STRING;
	theme->author = g_strdup(scanner->value.v_string);
  
	return G_TOKEN_NONE;
}

static guint
_parse_module(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_MODULE)
		return THEME_TOKEN_MODULE;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_STRING)
		return G_TOKEN_STRING;
	theme->module_name = g_strdup(scanner->value.v_string);
  
	return G_TOKEN_NONE;
}

static guint
_parse_stats(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_STATS)
		return THEME_TOKEN_STATS;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_INT)
		return G_TOKEN_INT;
	theme->num_gstat = scanner->value.v_int;

	return G_TOKEN_NONE;
}

static guint
_parse_directions(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_DIRECTIONS)
		return THEME_TOKEN_DIRECTIONS;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_INT)
		return G_TOKEN_INT;
	theme->num_direct = scanner->value.v_int;
  
	return G_TOKEN_NONE;
}

static guint
_parse_pixmapsize(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_PIXMAPSIZE)
		return THEME_TOKEN_PIXMAPSIZE;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_LEFT_CURLY)
		return G_TOKEN_LEFT_CURLY;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_INT)
		return G_TOKEN_INT;
	theme->pix_width = scanner->value.v_int;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_COMMA)
		return G_TOKEN_COMMA;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_INT)
		return G_TOKEN_INT;
	theme->pix_height = scanner->value.v_int;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_RIGHT_CURLY)
		return G_TOKEN_RIGHT_CURLY;

	return G_TOKEN_NONE;
}

static guint
_parse_steppixel(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
  
	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_STEPPIXEL)
		return THEME_TOKEN_STEPPIXEL;
  
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_INT)
		return G_TOKEN_INT;
	theme->step_pixel = scanner->value.v_int;
  
	return G_TOKEN_NONE;
}

static guint
_parse_data(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;

	token = g_scanner_get_next_token(scanner);
	if (token != THEME_TOKEN_DATA)
		return THEME_TOKEN_DATA;

	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_LEFT_CURLY)
		return G_TOKEN_LEFT_CURLY;

	privat->_nd = 0;
	token = g_scanner_peek_next_token(scanner);
	while (token != G_TOKEN_RIGHT_CURLY) {
		guint expected_token;

		switch (token) {
		case G_TOKEN_LEFT_CURLY:
			expected_token = _parse_frames(theme);
			if (expected_token != G_TOKEN_NONE)
				return expected_token;
			break;
		case G_TOKEN_RIGHT_CURLY:
			break;
		default:
			g_scanner_get_next_token(scanner);
			return G_TOKEN_LEFT_CURLY;
		}
		token = g_scanner_peek_next_token(scanner);
		privat->_nd++;
	}
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_RIGHT_CURLY)
		return G_TOKEN_RIGHT_CURLY;

	privat->_ns++;
	
	return G_TOKEN_NONE;
}

static guint
_parse_frames(GroTheme *theme)
{
	GroThemePrivate *privat = theme->privat;
	GScanner *scanner = privat->_scanner;
	guint token;
	int i_frame;
	
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_LEFT_CURLY)
		return G_TOKEN_LEFT_CURLY;

	token = g_scanner_peek_next_token(scanner);
	i_frame = 0;
	while (token != G_TOKEN_RIGHT_CURLY) {
		char *key;
		
		switch (token) {
		case G_TOKEN_STRING:
			token = g_scanner_get_next_token(scanner);
#ifdef DEBUG			
			g_print("filename[%d,%d,%d]:%s\n",
					privat->_ns, privat->_nd, i_frame, scanner->value.v_string);
#endif
			key = g_strdup_printf("%d,%d,%d",
								  privat->_ns, privat->_nd, i_frame);
			g_hash_table_insert(privat->_tmp_fname_htable, key, 
								g_strdup(scanner->value.v_string));
			break;
		case G_TOKEN_RIGHT_CURLY:
			break;
		default:
			g_scanner_get_next_token(scanner);
			return G_TOKEN_STRING;/* or right_curly... */
		}
		token = g_scanner_peek_next_token(scanner);
		i_frame++;
	}
	token = g_scanner_get_next_token(scanner);
	if (token != G_TOKEN_RIGHT_CURLY)
		return G_TOKEN_RIGHT_CURLY;

	return G_TOKEN_NONE;
}

#ifdef DEBUG
static void
_parse_verify(const GroTheme *theme)
{
	g_print("theme parse verify\n");
	g_print("theme->name = %s\n", theme->name);
	g_print("theme->description = %s\n", theme->description);
	g_print("theme->num_gstat = %d\n", theme->num_gstat);
	g_print("theme->num_direct = %d\n", theme->num_direct);
	g_print("theme->pix_width = %d\n", theme->pix_width);
	g_print("theme->pix_height = %d\n", theme->pix_height);
	g_print("theme->step_pixel = %d\n", theme->step_pixel);
}
#endif


static void
_parse_hash_frees(gpointer key, gpointer value, gpointer data)
{
	g_free(key);/* "x,y,z" */
	g_free(value);/* filename */
}

