/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <string.h>

#include "errorchain.h"
#include "common.h"
#include "util.h"
#include "debug.h"

#include "../icons/bomb.xpm"


struct errchain *chains = NULL;

int error_new_context(char *title)
{
     struct errchain *chain, *new_chain;
     int context = 0;

     new_chain = MALLOC(sizeof(struct errchain), "struct errchain");
     strncpy(new_chain->title, title, MAX_ERRTITLE_SIZE);
     new_chain->title[MAX_ERRTITLE_SIZE] = '\0';
     new_chain->msgs = NULL;
     new_chain->next = NULL;

     if(!chains)
	  chains = new_chain;
     else {
	  chain = chains;
	  while(chain->next) {
	       context = chain->context;
	       chain = chain->next;
	  }
	  chain->next = new_chain;
	  context = chain->context + 1;
     }

     new_chain->context = context;

     return(context);
}


void error_push(int context, char *msg)
{
     struct errchain *chain;
     struct errmsgs *new_msg, *cur_msg;

     new_msg = MALLOC(sizeof(struct errmsgs), "struct errmsgs");
     strncpy(new_msg->msg, msg, MAX_ERRMSG_SIZE);
     new_msg->msg[MAX_ERRMSG_SIZE] = '\0';
     new_msg->next = NULL;

     /* plug into messagechain */
     chain = error_chain_by_context(context);
     if(!chain->msgs)
	  chain->msgs = new_msg;
     else {
	  cur_msg = chain->msgs;
	  while(cur_msg->next)
	       cur_msg = cur_msg->next;
	  cur_msg->next = new_msg;
     }

}


/* returns chain for requested context */
struct errchain *error_chain_by_context(int q)
{
     struct errchain *chain;

     chain = chains;
     while(chain && chain->context != q)
	  chain = chain->next;

     if(!chain) {
	  fprintf(stderr, "Oops! errorchain lookup error. Exiting...\n");
	  exit(1);
     }

     return(chain);
}


void error_flush(int context)
{
     GdkPixmap *bomb;
     GdkBitmap *bomb_mask;
     GtkWidget *pixmap, *popupwin, *vbox1, *hbox, *vbox2, *msg_label, *okbutton;
     struct errchain *chain, *cur_chain;
     struct errmsgs *cur_msg, *old_msg;

     chain = error_chain_by_context(context);

     if(chain->msgs) {
	  popupwin = gtk_dialog_new();
	  gtk_widget_realize(popupwin);
	  gtk_container_border_width(GTK_CONTAINER(popupwin), 12);
	  gtk_window_set_title(GTK_WINDOW(popupwin), chain->title);
	  gtk_window_set_policy(GTK_WINDOW(popupwin), FALSE, FALSE, FALSE);
	  vbox1 = GTK_DIALOG(popupwin)->vbox;
	  gtk_widget_show(vbox1);
	  hbox = gtk_hbox_new(FALSE, 0);
	  gtk_widget_show(hbox);
	  gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	  bomb = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(popupwin)->window,
					      &bomb_mask,
					      &popupwin->style->white,
					      bomb_xpm);
	  pixmap = gtk_pixmap_new(bomb, bomb_mask);
	  gtk_widget_show(pixmap);
	  gtk_box_pack_start(GTK_BOX(hbox), pixmap, TRUE, TRUE, 10);

	  /* show messages, freeing them as we go */
	  cur_msg = chain->msgs;
	  while(cur_msg) {
	       msg_label = gtk_label_new(cur_msg->msg);
	       gtk_widget_show(msg_label);
	       gtk_box_pack_start(GTK_BOX(hbox), msg_label, FALSE, FALSE, 0);
	       old_msg = cur_msg;
	       cur_msg = cur_msg->next;
	       FREE(old_msg, "struct errmsgs");
	  }

	  vbox2 = GTK_DIALOG(popupwin)->action_area;
	  gtk_widget_show(vbox2);
	  okbutton = gtk_button_new_with_label("   OK   ");
	  gtk_signal_connect_object(GTK_OBJECT(okbutton), "clicked",
				    GTK_SIGNAL_FUNC(gtk_widget_destroy),
				    GTK_OBJECT(popupwin));
	  gtk_signal_connect_object(GTK_OBJECT(popupwin), "key_press_event",
				    GTK_SIGNAL_FUNC(close_on_esc),
				    (gpointer) popupwin);
	  GTK_WIDGET_SET_FLAGS(okbutton, GTK_CAN_DEFAULT);
	  gtk_box_pack_end(GTK_BOX(vbox2), okbutton, TRUE, FALSE, 0);
	  gtk_widget_grab_default(okbutton);
	  gtk_widget_show(okbutton);
	  gtk_widget_show(popupwin);
     }

     /* free chain */
     if(chain == chains)
          chains = chain->next;
     else {
	  cur_chain = chains;
	  while(cur_chain && cur_chain->next != chain)
	       cur_chain = cur_chain->next;
	  if(cur_chain)
	       cur_chain->next = cur_chain->next->next;
	  else {
	       fprintf(stderr, "Oops! errorchain free error. Exiting...\n");
	       exit(1);
	  }

     }

     FREE(chain, "struct errchain");

}


void error_popup(char *title, char *message)
{
     int context;

     context = error_new_context(title);
     error_push(context, message);
     error_flush(context);

}


