#include "problemAnalyzer.h"
#include "analysisViewer.h"

#include "osl/apply_move/applyMove.h"
#include "osl/record/csa.h"
#include "osl/record/csaString.h"
#include "osl/search/simpleHashRecord.h"
#include "osl/state/simpleState.h"
#include "osl/stl/vector.h"

#include "gpsshogi/gui/util.h"

#include <boost/scoped_ptr.hpp>

#include <qlayout.h>
#include <qpushbutton.h>
#include <qfiledialog.h>
#include <qdir.h>
#include <qlistview.h>
#include <qtextbrowser.h>
#if QT_VERSION >= 0x040000
#include <Q3PopupMenu>
#include <Q3ListBox>
#include <QDomNode>
#else
#include <qpopupmenu.h>
#include <qlistbox.h>
#define Q3PopupMenu QPopupMenu
#define Q3ListBox QListBox
#define Q3ListBoxText QListBoxText
#define Q3ListBoxItem QListBoxItem
#include <qdom.h>
#endif
#include <qlabel.h>
#include <qdialog.h>
#include <qmessagebox.h>

#include <qfile.h>

#include <iostream>
#include <fstream>

class ProblemItem : public Q3ListViewItem
{
public:
  ProblemItem(Q3ListView *parent, osl::Move m, int type, bool success);
  ProblemItem(Q3ListViewItem *parent, osl::Move m, int type, bool success);
  void paintCell(QPainter *p, const QColorGroup &cg,
		 int column, int width, int align);
private:
  osl::Move move;
  int type;
  bool success;
};


void ProblemItem::paintCell(QPainter *p, const QColorGroup &cg,
			    int column, int width, int align)
{
  if (column == 0 && !success)
  {
    QColorGroup colorGroup(cg);
    colorGroup.setColor(QColorGroup::Text, QColor("red"));
    Q3ListViewItem::paintCell(p, colorGroup, column, width, align);
  }
  else
    Q3ListViewItem::paintCell(p, cg, column, width, align);
}

class BaseMove
{
public:
  virtual ~BaseMove() {
  }
  virtual BaseMove *parse(QDomNode& node,
			  const osl::state::SimpleState& state) = 0;
  virtual osl::Move getMove() const {
    return osl::Move::INVALID();
  }
  virtual Q3ListViewItem *addItem(Q3ListView *view,
				 const osl::search::SimpleHashRecord *record,
				 bool& success) = 0;
  virtual Q3ListViewItem *addItem(Q3ListViewItem *item,
				 const osl::search::SimpleHashRecord *record,
				 bool& success) = 0;
  static BaseMove *parseMove(QDomNode& node, const osl::state::SimpleState& state);
protected:
  const osl::search::SimpleHashRecord *nextRecord(const osl::search::SimpleHashRecord *record, osl::Move move) {
    if (move.isInvalid() || ! record)
      return 0;
    // TODO HashKeyとSimpleHashTableが必要
    return 0;
  }
};

class ProblemMove : public BaseMove
{
public:
  ProblemMove() : move(osl::Move::INVALID()), type(NORMAL) {
  }
  BaseMove *parse(QDomNode &node, const osl::state::SimpleState& state);
  osl::Move getMove() const {
    return move;
  }
  Q3ListViewItem *addItem(Q3ListView *view,
			 const osl::search::SimpleHashRecord *record,
			 bool& success);
  Q3ListViewItem *addItem(Q3ListViewItem *item,
			 const osl::search::SimpleHashRecord *record,
			 bool& success);
  enum move_type
  {
    NORMAL,
    GOOD,
    BAD
  };

  move_type getType() const {
    return type;
  }
private:
  osl::Move move;
  move_type type;
};

class ProblemMoveSet : public BaseMove
{
public:
  ProblemMoveSet() : BaseMove() {
  }
  ~ProblemMoveSet() {
    for (size_t i = 0; i < moves.size(); i++)
    {
      delete moves[i];
    }
  }
  BaseMove *parse(QDomNode &node, const osl::state::SimpleState& state);
  Q3ListViewItem *addItem(Q3ListView *view,
			 const osl::search::SimpleHashRecord *record,
			 bool& success);
  Q3ListViewItem *addItem(Q3ListViewItem *item,
			 const osl::search::SimpleHashRecord *record,
			 bool& success);
private:
  osl::stl::vector<BaseMove *> moves;
};

class ProblemMoveList : public BaseMove
{
public:
  ProblemMoveList() : BaseMove() {
  }
  ~ProblemMoveList() {
    for (size_t i = 0; i < moves.size(); i++)
    {
      delete moves[i];
    }
  }
  BaseMove *parse(QDomNode &node, const osl::state::SimpleState& state);
  Q3ListViewItem *addItem(Q3ListView *view,
			 const osl::search::SimpleHashRecord *record,
			 bool& success);
  Q3ListViewItem *addItem(Q3ListViewItem *item,
			 const osl::search::SimpleHashRecord *record,
			 bool& success);
private:
  osl::stl::vector<BaseMove *> moves;
};

BaseMove *BaseMove::parseMove(QDomNode& node,
			      const osl::state::SimpleState& state)
{
  const QString& tagName = node.toElement().tagName();
  BaseMove *move;
  if (tagName == "move")
    move = new ProblemMove();
  else if (tagName == "moveSet")
    move = new ProblemMoveSet();
  else if (tagName == "moveList")
    move = new ProblemMoveList();
  else abort();

  return move->parse(node, state);
}

BaseMove *ProblemMove::parse(QDomNode& node, const osl::state::SimpleState& state)
{
  QDomElement element = node.toElement();
  assert(element.tagName() == "move");
  if (!element.attributeNode("good_move").isNull())
    type = GOOD;
  else if (!element.attributeNode("bad_move").isNull())
    type = BAD;

  move = osl::record::csa::strToMove(element.text().ascii(), state);
  return this;
}

Q3ListViewItem *ProblemMove::addItem(Q3ListView *view,
				    const osl::search::SimpleHashRecord *record,
				    bool& success)
{
  if (type == NORMAL)
    success = true;
  else if (type == GOOD)
    success = (record && record->bestMove().getMove() == move);
  else if (type == BAD)
    success = (record && record->bestMove().getMove() != move);

  return new ProblemItem(view, move, type, success);
}

Q3ListViewItem *ProblemMove::addItem(Q3ListViewItem *item,
				    const osl::search::SimpleHashRecord *record,
				    bool& success)
{
  if (type == NORMAL)
    success = true;
  else if (type == GOOD)
    success = (record && record->bestMove().getMove() == move);
  else if (type == BAD)
    success = (record && record->bestMove().getMove() != move);

  return new ProblemItem(item, move, type, success);
}

BaseMove *ProblemMoveList::parse(QDomNode &node, const osl::state::SimpleState& s)
{
  // MoveList can contain <move>s and <moveSet> at the end.
  // Otherwise, follwing doMove will fail.
  osl::state::SimpleState state(s);
  for (QDomNode n = node.firstChild(); !n.isNull(); n = n.nextSibling())
  {
    BaseMove *move = BaseMove::parseMove(n, state);
    moves.push_back(move);
    if (!move->getMove().isInvalid())
    {
      osl::apply_move::ApplyMoveOfTurn::doMove(state, move->getMove());
    }
  }
  return this;
}

BaseMove *ProblemMoveSet::parse(QDomNode &node,
				const osl::state::SimpleState& state)
{
  // moveSet can contain <move>s and <moveList>s.
  for (QDomNode n = node.firstChild(); !n.isNull(); n = n.nextSibling())
  {
    BaseMove *move = BaseMove::parseMove(n, state);
    moves.push_back(move);
  }
  return this;
}

Q3ListViewItem *ProblemMoveSet::addItem(Q3ListView *view,
				       const osl::search::SimpleHashRecord *record,
				       bool& success)
{
  success = true;
  bool bestMoveMatches = false;
  bool goodMoveExists = false;

  for (size_t i = 0; i < moves.size(); i ++)
  {
    bool result;
    Q3ListViewItem *item = moves[i]->addItem(view, record, result);
    // If one of the good moves is the best move, it is OK.
    if (item)
    {
      ProblemMove *move = (ProblemMove *)moves[i];
      if (move->getType() == ProblemMove::GOOD)
      {
	goodMoveExists = true;
	if (result)
	{
	  bestMoveMatches = true;
	}
      }
    }
    success = success && result;
  }
  if (goodMoveExists)
    success = success && bestMoveMatches;
  return 0;
}

Q3ListViewItem *ProblemMoveSet::addItem(Q3ListViewItem *viewItem,
				       const osl::search::SimpleHashRecord *record,
				       bool& success)
{
  success = true;
  bool bestMoveMatches = false;
  bool goodMoveExists = false;

  for (size_t i = 0; i < moves.size(); i ++)
  {
    bool result;
    Q3ListViewItem *item = moves[i]->addItem(viewItem, record, result);
    // If one of the good moves is the best move, it is OK.
    if (item)
    {
      ProblemMove *move = (ProblemMove *)moves[i];
      if (move->getType() == ProblemMove::GOOD)
      {
	goodMoveExists = true;
	if (result)
	{
	  bestMoveMatches = true;
	}
      }
    }
    success = success && result;
  }
  if (goodMoveExists)
    success = success && bestMoveMatches;
  return 0;
}

Q3ListViewItem *ProblemMoveList::addItem(Q3ListView *view,
					const osl::search::SimpleHashRecord *record,
					bool& success)
{
  success = true;
  if (moves.size() == 0)
    return 0;

  const osl::search::SimpleHashRecord *r = record;
  bool result;
  Q3ListViewItem *item = moves[0]->addItem(view, r, result);
  success = success && result;
  r = nextRecord(r, moves[0]->getMove());
  for (size_t i = 1; i < moves.size(); i ++)
  {
    item = moves[i]->addItem(item, r, result);
    success = success && result;
    r = nextRecord(r, moves[i]->getMove());
  }
  return 0;
}

Q3ListViewItem *
ProblemMoveList::addItem(Q3ListViewItem *i,
			 const osl::search::SimpleHashRecord *record,
			 bool& success)
{
  Q3ListViewItem *item = i;
  success = true;
  const osl::search::SimpleHashRecord *r = record;
  bool result;
  for (size_t i = 0; i < moves.size(); i ++)
  {
    item = moves[i]->addItem(item, record, result);
    success = success && result;
    r = nextRecord(r, moves[i]->getMove());
  }
  return 0;
}

class Problem
{
public:
  Problem() : state(osl::HIRATE), move(0) {
  }
  const osl::state::SimpleState& getState() {
    return state;
  }
  void parse(QDomNode &node);
  bool verifyAndAddItems(Q3ListView *view,
			 const osl::search::SimpleHashRecord *record);
  QString getComment() const {
    return comment;
  }

private:
  osl::state::SimpleState state;
  boost::scoped_ptr<BaseMove> move;
  QString comment;
};

void Problem::parse(QDomNode &n)
{
  for (QDomNode node = n; !node.isNull(); node = node.nextSibling())
  {
    if (node.isElement() && node.toElement().tagName() == "problem")
    {
      for (QDomNode child = node.firstChild();
	   !child.isNull(); child = child.nextSibling())
      {
	if (child.isElement())
	{
	  QDomElement element = child.toElement();
	  if (element.tagName() == "state" && element.text() != "")
	  {
	    state = osl::record::csa::CsaString(element.text()).getInitialState();
	  }
	  else if (element.tagName() == "moves")
	  {
	    QDomNode moveNode = child.firstChild();
	    move.reset(BaseMove::parseMove(moveNode, state));
	  }
	  if (element.tagName() == "comment")
	  {
	    comment = element.text().stripWhiteSpace();
	  }
	}
      }
      return;
    }
  }
  std::cerr << "<problem> not found" << std::endl;
}

bool Problem::verifyAndAddItems(Q3ListView *view,
				const osl::search::SimpleHashRecord *record)
{
  bool result;
  move->addItem(view, record, result);
  return result;
}

ProblemItem::ProblemItem(Q3ListView *parent, osl::Move m, int type, bool success)
  : Q3ListViewItem(parent, gpsshogi::gui::Util::moveToString(m), ""),
    move(m), type(type), success(success)
{
  if (type == ProblemMove::GOOD)
  {
    setText(1, "GOOD");
  }
  else if (type == ProblemMove::BAD)
  {
    setText(1, "BAD");
  }
}

ProblemItem::ProblemItem(Q3ListViewItem *parent, osl::Move m, int type, bool success)
  : Q3ListViewItem(parent, gpsshogi::gui::Util::moveToString(m), ""),
    move(m), type(type), success(success)
{
  if (type == ProblemMove::GOOD)
  {
    setText(1, "GOOD");
  }
  else if (type == ProblemMove::BAD)
  {
    setText(1, "BAD");
  }
}

ProblemAnalyzer::ProblemAnalyzer(QWidget *parent, const char *name)
  : TabChild(parent, name), analyzed(false)
{
  board = new gpsshogi::gui::Board(osl::state::SimpleState(osl::HIRATE), this);
  analysisViewer = new AnalysisViewer(this);

  moveTree = new Q3ListView(this);
  moveTree->addColumn("Move");
  moveTree->addColumn("Type");
  moveTree->setSorting(moveTree->columns() + 1);
  moveTree->setRootIsDecorated(true);
  button = new QPushButton(this);
  button->setText("Search");
  nextButton = new QPushButton(this);
  nextButton->setText("Next");
  prevButton = new QPushButton(this);
  prevButton->setText("Prev");
  showButton = new QPushButton(this);
  showButton->setText("Show Problem");
  commentView = new QTextBrowser(this);

  QVBoxLayout *leftLayout = new QVBoxLayout;
  leftLayout->addWidget(moveTree);
  leftLayout->addWidget(button);
  leftLayout->addWidget(nextButton);
  leftLayout->addWidget(prevButton);
  leftLayout->addWidget(showButton);
  QVBoxLayout *centerLayout = new QVBoxLayout;
  centerLayout->addWidget(board);
  centerLayout->addWidget(analysisViewer, 1);
  centerLayout->addWidget(commentView);
  QHBoxLayout *mainLayout = new QHBoxLayout(this);
  mainLayout->addLayout(leftLayout);
  mainLayout->addLayout(centerLayout, 1);
  board->hide();

  connect(button, SIGNAL(clicked()),
	  this, SLOT(analyzeOne()));
  connect(nextButton, SIGNAL(clicked()),
	  this, SLOT(next()));
  connect(prevButton, SIGNAL(clicked()),
	  this, SLOT(prev()));
  connect(showButton, SIGNAL(clicked()),
	  this, SLOT(showProblem()));
  connect(analysisViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
}

void ProblemAnalyzer::forward()
{
  analysisViewer->forward();
}

void ProblemAnalyzer::backward()
{
  analysisViewer->backward();
}

void ProblemAnalyzer::toInitialState()
{
  analysisViewer->toInitialState();
}

void ProblemAnalyzer::toLastState()
{
  analysisViewer->toLastState();
}

int ProblemAnalyzer::moveCount() const
{
  return analysisViewer->moveCount();
}

osl::Player ProblemAnalyzer::turn() const
{
  return analysisViewer->turn();
}

const osl::state::SimpleState& ProblemAnalyzer::getState()
{
  if (analysisViewer->isHidden())
    return board->getState();
  else
    return analysisViewer->getState();
}

void ProblemAnalyzer::toggleOrientation()
{
  if (analysisViewer->isHidden())
    return board->toggleOrientation();
  else
    return analysisViewer->toggleOrientation();
}

osl::state::SimpleState
ProblemAnalyzer::getStateAndMovesToCurrent(osl::stl::vector<osl::Move> &moves)
{
  if (analysisViewer->isHidden())
  {
    return board->getState();
  }
  else
    return analysisViewer->getStateAndMovesToCurrent(moves);
}

void ProblemAnalyzer::contextMenuEvent(QContextMenuEvent *e)
{
  Q3PopupMenu* contextMenu = new Q3PopupMenu(this);
  contextMenu->insertItem("Reset Search Parameters", this,
			  SLOT(resetSearchParameters()));
  contextMenu->insertItem("Choose Next Problem", this,
			  SLOT(chooseProblem()));
  contextMenu->insertItem("View the result", this,
			  SLOT(viewResult()));
  contextMenu->insertItem("Save the result", this,
			  SLOT(saveResult()));
  contextMenu->insertItem("Analyze All", this,
			  SLOT(analyzeAll()));
  contextMenu->exec(e->globalPos());
  delete contextMenu;
}

bool ProblemAnalyzer::analyzeOne()
{
  if (index >= (int)files.size())
    return false;

  QDomDocument d;
  QFile file(dirName + files[index]);
  d.setContent(&file);
  QDomNode n = d.firstChild();
  Problem p;
  p.parse(n);

  osl::stl::vector<osl::Move> moves;
  bool ret;

  if (!analyzed)
    ret = analysisViewer->analyze(p.getState(), moves);
  else
    ret = analysisViewer->analyzeWithSavedValue(p.getState(), moves);

  if (ret)
  {
    analyzed = true;
    board->hide();
    analysisViewer->show();
    if (index == (int)files.size())
      button->setDisabled(true);
    commentView->setText(p.getComment());
    moveTree->clear();
    bool result = p.verifyAndAddItems(moveTree, analysisViewer->getRecord());
    results[index] = result ? SUCCESS : FAILURE;
    return true;
  }

  return false;
}

bool ProblemAnalyzer::analyze()
{
  QString directory =
    QFileDialog::getExistingDirectory(dirName, this,
				      "Problems",
				      "Choose directory wih problems");
  if (directory.isNull())
    return false;

  dirName = directory;
  QDir dir(directory);

  files = dir.entryList("*.xml");

  if (files.size() == 0)
    return false;

  index = 0;
  results.reset(new problem_result[files.size()]);
  std::fill(&results[0], &results[files.size()], NOT_TRIED);
  showProblem();

  return true;
}

void ProblemAnalyzer::next()
{
  if (index < (int) files.size() - 1)
  {
    index++;
    showProblem();
    emit statusChanged();
  }
}

void ProblemAnalyzer::prev()
{
  if (index > 0)
  {
    index--;
    showProblem();
    emit statusChanged();
  }
}

QString ProblemAnalyzer::getFilename()
{
  if (index >= 0 && index < (int)files.size())
    return files[index];
  else
    return QString();
}

void ProblemAnalyzer::showProblem()
{
  if (index >= (int)files.size())
    return;
  QDomDocument d;
  QFile file(dirName + files[index]);
  d.setContent(&file);
  QDomNode n = d.firstChild();
  Problem p;
  p.parse(n);

  moveTree->clear();
  board->setState(p.getState());
  board->show();
  analysisViewer->hide();
  commentView->setText(p.getComment());
  // p.verifyAndAddItems(moveTree, 0);
  emit statusChanged();
}

void ProblemAnalyzer::resetSearchParameters()
{
  analyzed = false;
}

class ProblemSelectDialog : public QDialog
{
public:
  ProblemSelectDialog(const QStringList& files,
		      QWidget *parent = 0, const char *name = 0);
  int getIndex() const {
    return list->currentItem();
  }
private:
  Q3ListBox *list;
};

ProblemSelectDialog::ProblemSelectDialog(const QStringList& files,
					 QWidget *parent, const char *name)
  : QDialog(parent, name, true)
{
  list = new Q3ListBox(this);

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    list->insertItem(new Q3ListBoxText(files[i]));
  }

  QVBoxLayout *layout = new QVBoxLayout(this);
  layout->addWidget(list);
  QPushButton *button = new QPushButton(this);
  button->setText("&OK");
  QPushButton *cancelButton = new QPushButton(this);
  cancelButton->setText("&Cancel");
  QHBoxLayout *hLayout = new QHBoxLayout;
  hLayout->addWidget(button);
  hLayout->addWidget(cancelButton);
  layout->addLayout(hLayout);
  connect(button, SIGNAL(clicked()), this, SLOT(accept()));
  connect(cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
  connect(list, SIGNAL(doubleClicked(Q3ListBoxItem *)), this, SLOT(accept()));
  resize(layout->sizeHint());
}

void ProblemAnalyzer::chooseProblem()
{
  boost::scoped_ptr<ProblemSelectDialog> dialog(new ProblemSelectDialog(files,
									this));
  if (dialog->exec() == QDialog::Accepted)
  {
    if (dialog->getIndex() >= 0 && dialog->getIndex() != index)
    {
      index = dialog->getIndex();
      showProblem();
    }
  }
}

void ProblemAnalyzer::viewResult()
{
  QDialog *dialog = new QDialog(this);
  QVBoxLayout *layout = new QVBoxLayout(dialog);
  Q3ListView *view = new Q3ListView(dialog);
  view->addColumn("Problem");
  view->addColumn("Result");
  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    QString result = "";
    if (results[i] == SUCCESS)
      result = "Success";
    else if (results[i] == FAILURE)
      result = "Failure";
    new Q3ListViewItem(view, files[i], result); 
  }

  QPushButton *button = new QPushButton(dialog);
  button->setText("&OK");
  connect(button, SIGNAL(clicked()), dialog, SLOT(accept()));
  layout->addWidget(view);
  layout->addWidget(button);

  dialog->show();
  dialog->raise();
  dialog->resize(layout->sizeHint());
}

void ProblemAnalyzer::saveResult()
{
  QString s = QFileDialog::getSaveFileName(0,
					   "HTML (*.html)",
					   this,
					   "save file",
					   "Save result");
  if (s.isNull())
    return;

  if (QFile(s).exists())
  {
    int button = QMessageBox::question(this, "Overwrite?",
				       "File " + s + " exists, overwrite?",
				       QMessageBox::Yes,
				       QMessageBox::No | QMessageBox::Default);
    if (button == QMessageBox::No)
      return;
  }

  std::ofstream os(s.ascii());
  os << "<table>" << std::endl;

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    os << "<tr><td>" << files[i].ascii() << "</td><td>";
    if (results[i] == SUCCESS)
      os << "Good";
    else if (results[i] == FAILURE)
      os << "Bad";
    else
      os << "-";
    os << "</td></tr>" << std::endl;
  }

  os << "</table>" << std::endl;
}


void ProblemAnalyzer::analyzeAll()
{
  resetSearchParameters();
  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    bool problem_analyzed = analyzeOne();
    if (!problem_analyzed)
      break;
    index++;
  }
}

QWidget *ProblemAnalyzer::moveGenerateDialog()
{
  return analysisViewer->moveGenerateDialog();
}
