/* prompt.c - all things prompt-related */

/* level-1 system headers */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "sh_sys_types.h"
#include <sys/stat.h>
#include <ctype.h>
#include <pthread.h>
#include "sh_termio.h"

/* level-2 system headers */
#include "sv_readline.h"
#include "sv_regex.h"

/* local headers */
#include "cache.h"
#include "cli.h"
#include "cli_output.h"
#include "conf.h"
#include "connection.h"
#include "gnut.h"
#include "hash.h"
#include "host.h"
#include "http.h"
#include "lib.h"
#include "list.h"
#include "monitor.h"
#include "net.h"
#include "prompt.h"
#include "qry.h"
#include "share.h"
#include "threads.h"
#include "transfer.h"

Variable_Info var_list[] = {
  {"HOSTS",		prompt_host_stats},
  {"FILES",		prompt_host_stats},
  {"SIZE",		prompt_host_stats},
  {"MSGS_R",		prompt_net_stats},
  {"MSGS_S",		prompt_net_stats},
  {"BYTES_R",		prompt_net_stats},
  {"BYTES_S",		prompt_net_stats},
  {"QUERIES",		prompt_query_stats},
  {"QUERY_RESP",	prompt_query_stats},
  {"NUM_SHARED",	prompt_share_stats},
  {"SIZE_SHARED",	prompt_share_stats},
  {"RESPONSES",		prompt_results_stats},
  {0,			0},
};

char pmt_result[BUF_MAX];

/* prompt_str() - returns a prompt string, post-interpolation */
char * prompt_str(void)
{
  static char prompt[PROMPT_STR_LEN];
  char *chr, *position;

  bzero((void *)prompt, PROMPT_STR_LEN);
  chr = conf_get_str("prompt");

  do {
    position = strstr(chr, "${");

    if (position) { /* variable interpolation */
      strncat(prompt, chr, position - chr);
      strcat(prompt, interp_var(position, &chr));
    } else { /* no interpolation remaining */
      strcat(prompt, chr); /* TODO: make this not a buffer overflow */
    }
  } while(position);

  /* 0.4.28.c14 */
  for (chr=prompt; *chr && *(chr+1); chr++) {
    if ((*chr == '\\') && (*(chr+1) == 'n')) {
      *chr = ' ';
      *(chr+1) = '\n';
    }
  } 

  return prompt;
}

/* char *interp_var(char *start, char **finish)

   start is a pointer to string that goes something like
  "${foobar}check one two"
   finish ends up a pointer to string that goes something like "check one two"
   return value is a pointer to the value of the variable     */
char * interp_var(char *start, char **finish)
{
  char *value, *end, *var_name;

  /* sanity check */
  if( !start ||
      start[0] != '$' || start[1] != '{' ||
      !(end = strchr(start, '}'))
      )
    {	
      *finish = 0;
      return "<Error>";
    }

  if (end - start == 2) {	   /* I like it when ${} expands to "\n" */
    *finish = end + 1;
    return "\n";
  }

  start += 2; end--;		   /* start and finish bound the variable */
  var_name = ymaloc(end - start + 2, 303);
  bzero(var_name, end - start + 2);
  strncpy(var_name, start, (end - start) + 1);
  /* fprintf(stderr, "Var is '%s'\n", var_name); */
  if (G_ISLOWER(var_name[0])) {
    /* lowercase - config variable */
    value = conf_get_str(var_name);
  } else {
    /* uppercase - gnut info stuff */
    Variable_Info *a;

    for (a = var_list; a && strcmp(a->name, var_name); a++) {
    }

    if (a) { /* variable match */
      (a->useful_function)(var_name);
      value = pmt_result; /* TODO: i don't like this global variable crap - jhs */
    } else {
      value = 0;
    }
  }
  *finish = end + 2; /* just after the ${...} b.s. */
  fre_str(&var_name, 374);
  return (value) ? value : "<Error>";
}

/* these functions all look up stuff for prompt variables */

/* prompt_host_stats() - returns info based on var_name
 * HOSTS) total hosts on network
 * FILES) total files on network
 * SIZE)  total size of files on network
 */
void prompt_host_stats(char *var_name) /* works like info_hosts() */
{
  uint num;
  float64 bytes, files;

  int munge = conf_get_int("munge");

  host_totals(&num, &files, &bytes);

  if(strcmp(var_name, "HOSTS") == 0) {
    sprintf(pmt_result, "%d", num);
  } else if(strcmp(var_name, "FILES") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(files, 0));
    } else {
      sprintf(pmt_result, "%g", files);
    }
  }  else if(strcmp(var_name, "SIZE") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(bytes, 0));
    } else {
      sprintf(pmt_result, "%g M", bytes / 1000000.0);
    }
  }  else {
    sprintf(pmt_result, "<Error>");
  }
}

/* prompt_net_stats() - returns info based on var_name
 * MSGS_R)  messages received
 * MSGS_S)  messages sent
 * BYTES_R) bytes received
 * BYTES_S) bytes sent
 * BLACKLIST_S) messages blacklisted
 */
void prompt_net_stats(char *var_name) /* works like info_net() */
{
  int messages, sent;
  float64 n_rcv_bytes, n_snt_bytes;
  float64 n_blacklisted;

  int munge = conf_get_int("munge");

  gnut_connection_net_totals(&messages, &sent, &n_rcv_bytes, &n_snt_bytes,
			     &n_blacklisted);

  if(strcmp(var_name, "MSGS_R") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(messages, 0));
    } else {
      sprintf(pmt_result, "%d", messages);
    }
  } else if(strcmp(var_name, "MSGS_S") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(sent, 0));
    } else {
      sprintf(pmt_result, "%d", sent);
    }
  } else if(strcmp(var_name, "BYTES_R") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(n_rcv_bytes, 0));
    } else {
      sprintf(pmt_result, "%g", n_rcv_bytes);
    }
  } else if(strcmp(var_name, "BYTES_S") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(n_snt_bytes, 0));
    } else {
      sprintf(pmt_result, "%g", n_snt_bytes);
    }
  } else if(strcmp(var_name, "BLACKLIST_S") == 0) {
    if(munge) {
      sprintf(pmt_result, "%s", format_si(n_blacklisted, 0));
    } else {
      sprintf(pmt_result, "%g", n_blacklisted);
    }
  } else {
    sprintf(pmt_result, "<Error>");
  }
}

/* prompt_query_stats - returns query stats:
 * QUERIES)    messages received
 * QUERY_RESP) bytes sent */
void prompt_query_stats(char *var_name)	/* works like info_net() */
{
  if(strcmp(var_name, "QUERIES") == 0) {
    sprintf(pmt_result, "%d", con_num_queries);
  } else if(strcmp(var_name, "QUERY_RESP") == 0) {
    sprintf(pmt_result, "%d", con_num_responses);
  } else {
    sprintf(pmt_result, "<Error>");
  }
}

/* prompt_share_stats - returns query stats:
 * NUM_SHARED)  number of files shared
 * SIZE_SHARED) total bytes of files shared */
void prompt_share_stats(char *var_name) /* works like info_shares() */
{
  uint32 num;
  float64 sh_size;
  int munge;

  munge = conf_get_int("munge");

  share_totals(&num, &sh_size);

  if (strcmp(var_name, "NUM_SHARED") == 0) {
    sprintf(pmt_result, "%u", num);
  } else if (strcmp(var_name, "SIZE_SHARED") == 0) {
    if (munge) {
      sprintf(pmt_result, "%s", format_si(sh_size, 0));
    } else {
      sprintf(pmt_result, "%g", sh_size/1000.0);
    }
  } else {
    sprintf(pmt_result, "<Error>");
  }
}

/* prompt_results_stats - returns the number of find responses received */
void prompt_results_stats(char *var_name) /* works like info_shares() */
{
  sprintf(pmt_result, "%d", query_num());
}
