/*
**  IMAPMailWindowController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#import "IMAPMailWindowController.h"

#import "ExtendedCell.h"
#import "GNUMail.h"
#import "GNUMailConstants.h"

#ifndef MACOSX
#import "IMAPMailWindow.h"
#endif

#import "Utilities.h"

#import <Pantomime/Folder.h>
#import <Pantomime/IMAPFolder.h>
#import <Pantomime/IMAPStore.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/LocalStore.h>

@implementation IMAPMailWindowController

//
//
//
- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
#else
  IMAPMailWindow *imapMailWindow;
 
  

  imapMailWindow = [[IMAPMailWindow alloc] initWithContentRect: NSMakeRect(150,100,750,595)
					   styleMask: NSClosableWindowMask|NSTitledWindowMask|
					   NSMiniaturizableWindowMask|NSResizableWindowMask
					   backing: NSBackingStoreRetained
					   defer: NO];
  self = [super initWithWindow: imapMailWindow];

  [imapMailWindow layoutWindow];
  [imapMailWindow setDelegate: self];
  [imapMailWindow setFrame: [Utilities rectForMailWindow] display: NO];

  // We link our outlets
  tableScrollView = [imapMailWindow tableScrollView];
  textScrollView = [imapMailWindow textScrollView];

  splitView = [imapMailWindow splitView];
  tableView = [imapMailWindow tableView];
  textView = [imapMailWindow textView];
  
  next = [imapMailWindow next];
  previous = [imapMailWindow previous];
  delete = [imapMailWindow delete];
  mailboxes = [imapMailWindow mailboxes];
  compose = [imapMailWindow compose];
  forward = [imapMailWindow forward];
  reply = [imapMailWindow reply];
  addresses = [imapMailWindow reply];
  find = [imapMailWindow find];
  get = [imapMailWindow get];

  label = (NSTextField *)[imapMailWindow label];
  
  statusColumn = [imapMailWindow statusColumn];
  idColumn = [imapMailWindow idColumn];
  dateColumn = [imapMailWindow dateColumn];
  fromColumn = [imapMailWindow fromColumn];
  subjectColumn = [imapMailWindow subjectColumn];
  sizeColumn = [imapMailWindow sizeColumn];

  RELEASE(imapMailWindow);
#endif

  // We set our window title
  [[self window] setTitle: @""];
  
  // We must retain our columns under OS X after the outlets are linked (we release them in -dealloc)
#ifdef MACOSX
  RETAIN(statusColumn);
  RETAIN(idColumn);
  RETAIN(dateColumn);
  RETAIN(fromColumn);
  RETAIN(subjectColumn);
  RETAIN(sizeColumn);
#endif

  // We set our custom cell
  [statusColumn setDataCell: AUTORELEASE([[ExtendedCell alloc] init])];

  // We load our accessory views
  [self _loadAccessoryViews];

  // We restore our sorting order
  [self _restoreSortingOrder];

  // We restore our split view knob position
  [self _restoreSplitViewSize];

  // We load the right set of columns
  [self reloadTableColumns];
  
  return self;
}


//
//
//
- (void) dealloc
{
  IMAPStore *aStore;
  
  NSLog(@"IMAPMailWindowController: -dealloc");
  
  RELEASE(fetchingLock);
  
  TEST_RELEASE(timer);

  // FIXME - This will NOT work for multiple IMAP folders per IMAP server.
  aStore = (IMAPStore *)[[self folder] store];
  RELEASE(aStore);

  [super dealloc];
}


//
// action methods
//

//
//
//
- (IBAction) getNewMessages: (id) sender
{

  // We attempt to aquire the lock
  while (! [fetchingLock tryLock] )
    {
      [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
                                  beforeDate: [NSDate distantFuture]];
    }
  
  [(IMAPFolder *)[self folder] noop];
  
  [self tableViewShouldReloadData];
  [super updateStatusLabel];

  [fetchingLock unlock];
}


//
// delegate methods
//

- (void) windowWillClose: (NSNotification *) not
{
  [timer invalidate];

//
// FIXME. Why aren't we able to aquire the lock under OS-X?
//
#ifndef MACOSX  
  while (! [fetchingLock tryLock] )
    {
      [[NSRunLoop currentRunLoop] runMode: NSDefaultRunLoopMode
                                  beforeDate: [NSDate distantFuture]];
    }
#endif

  [fetchingLock unlock];

  [self _transferMessagesToTrashFolder];

  [super windowWillClose: not];
  
  // FIXME - That WILL NOT WORK with multiple opened folders per IMAP server
  // We close our store
  [(IMAPStore *)[[self folder] store] close];
}


//
//
//
- (void) windowDidLoad
{
  [super windowDidLoad];

  fetchingLock = [[NSLock alloc] init];
}


//
// access / mutation method
//

- (void) setFolder: (Folder *) theFolder
{
  NSDictionary *allReceivingValues, *allIMAPValues;
  int nbOfSecondsToWait;

  [super setFolder: theFolder];

  // We get our right timer delay
  allReceivingValues = [[NSUserDefaults standardUserDefaults] objectForKey: @"RECEIVING"];
  allIMAPValues = [allReceivingValues objectForKey: [(IMAPStore *)[[self folder] store] name] ];
  
  if (! [allIMAPValues objectForKey: @"RETRIEVEMINUTES"] ||
      [[allIMAPValues objectForKey:@"RETRIEVEMINUTES"] intValue] == 0)
    {
      NSLog(@"IMAP: Waiting 10 minutes before next fetch...");
      nbOfSecondsToWait = 600;
    }
  else
    {
      nbOfSecondsToWait = [[allIMAPValues objectForKey:@"RETRIEVEMINUTES"] intValue] *  60;
    }

  // We finally start our timer
  timer =  [NSTimer scheduledTimerWithTimeInterval: nbOfSecondsToWait
		    target: self
		    selector: @selector(getNewMessages:)
		    userInfo: nil
		    repeats: YES];
  RETAIN(timer);
}

@end


//
// Private interface
//
@implementation IMAPMailWindowController (Private)

- (void) _transferMessagesToTrashFolder
{
  LocalFolder *trashFolder;
  LocalStore *localStore;
  
  IMAPStore *aStore;
  NSDictionary *allValues;

  id aWindow = nil;
  BOOL aBOOL;
 
  // We get a reference to our IMAPStore and all the receiving values
  aStore = (IMAPStore *)[folder store];
  allValues = [[[NSUserDefaults standardUserDefaults] objectForKey:@"RECEIVING"] objectForKey: [aStore name]];
  
  // We verify if we need to transfer the messages to the Trash folder.
  aBOOL = ([[allValues objectForKey: @"TRANSFERTOLOCALTRASH"] intValue] == NSOnState ? YES : NO);

  if (! aBOOL )
    {
      return;
    }

  // We can now tranfer to trash, we get our local store
  localStore = [(GNUMail *)[NSApp delegate] localStore];
  
  // We get our trash folder
  trashFolder = (LocalFolder *)[localStore folderForName: [[NSUserDefaults standardUserDefaults] 
							    objectForKey: @"TRASHFOLDERNAME"]];

  // If our folder is already open, we obtain the window used by this folder
  if (! trashFolder )
    {
      aWindow = [Utilities windowForFolderName: [[NSUserDefaults standardUserDefaults] 
						  objectForKey: @"TRASHFOLDERNAME"]];
      trashFolder = (LocalFolder *)[[aWindow windowController] folder];
    }
  
  if ( trashFolder )
    {
      NSArray *allDeletedMessages;
      int i;
      
      allDeletedMessages = [[self folder] expunge: YES];
      
      for (i = 0; i < [allDeletedMessages count]; i++)
      	{
     	  [trashFolder appendMessageFromRawSource: [allDeletedMessages objectAtIndex: i]];
      	}
      
      if ( aWindow )
	{
	  [[aWindow delegate] tableViewShouldReloadData];
	}
      else
	{
	  [trashFolder close];
	}
    }
}

@end
