/*
**  TransportMethodEditorWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "TransportMethodEditorWindowController.h"

#import "GNUMailConstants.h"
#import "Utilities.h"

#ifndef MACOSX
#import "TransportMethodEditorWindow.h"
#endif

#import <Pantomime/SMTP.h>

@implementation TransportMethodEditorWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
 
#else
  TransportMethodEditorWindow *transportMethodEditorWindow;
  
  transportMethodEditorWindow =
    [[TransportMethodEditorWindow alloc] initWithContentRect: NSMakeRect(200,200,400,380)
					 styleMask: (NSTitledWindowMask |
						     NSMiniaturizableWindowMask)
					 backing: NSBackingStoreBuffered
					 defer: YES];
  
  self = [super initWithWindow: transportMethodEditorWindow];
  
  [transportMethodEditorWindow layoutWindow];
  [transportMethodEditorWindow setDelegate: self];
  
  // We link our outlets
  transportMethodPopUpButton = [transportMethodEditorWindow transportMethodPopUpButton];
  mailerField = [transportMethodEditorWindow mailerField];
  smtpHostField = [transportMethodEditorWindow smtpHostField];
  smtpUsernameField = [transportMethodEditorWindow smtpUsernameField];
  smtpPasswordSecureField = [transportMethodEditorWindow smtpPasswordSecureField];
  rememberPassword = [transportMethodEditorWindow rememberPassword];

  authenticateUsingButton = [transportMethodEditorWindow authenticateUsingButton];
  supportedMechanismsButton = [transportMethodEditorWindow supportedMechanismsButton];
  supportedMechanismsPopUp = [transportMethodEditorWindow supportedMechanismsPopUp];
  
  personalProfilePopUpButton = [transportMethodEditorWindow personalProfilePopUpButton];
  
  cancelButton = [transportMethodEditorWindow cancelButton];
  okButton = [transportMethodEditorWindow okButton];

  RELEASE(transportMethodEditorWindow);
#endif

  // We set our window title
  [[self window] setTitle: _(@"Add a transport method")];

  // We populate our personal profile popup button
  [self _loadPersonalProfilesInPopUpButton];

  [self transportMethodHasChanged: nil];

  return self;
}


- (void) dealloc
{
  NSLog(@"TransportMethodEditorWindowController: -dealloc");
 
  RELEASE(transportMethod);

  [super dealloc];
}


//
// action methods
//

- (IBAction) okClicked: (id) sender
{  
  // If we must create a new one...
  if ( ![self transportMethod] )
    {
      transportMethod = [[NSMutableDictionary alloc] init];
    }

  // We synchronize our popups
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];
  [personalProfilePopUpButton synchronizeTitleAndSelectedItem];
  [supportedMechanismsPopUp synchronizeTitleAndSelectedItem];

  [transportMethod setObject: [NSNumber numberWithInt: ([transportMethodPopUpButton indexOfSelectedItem] + 1)]
		   forKey: @"TRANSPORT_METHOD"];
  
  [transportMethod setObject: [mailerField stringValue]
		   forKey: @"MAILER_PATH"];
  
  [transportMethod setObject: [smtpHostField stringValue]
		   forKey: @"SMTP_HOST"];
  
  [transportMethod setObject: [smtpUsernameField stringValue]
		   forKey: @"SMTP_USERNAME"];
  
  if ( [rememberPassword state] == NSOnState )
    {
      [transportMethod setObject: [Utilities encryptPassword: [smtpPasswordSecureField stringValue]
					     withKey: [smtpHostField stringValue]]
		       forKey: @"SMTP_PASSWORD"];
    }
  else
    {
      [transportMethod removeObjectForKey: @"SMTP_PASSWORD"];
    }
  
  [transportMethod setObject: [NSNumber numberWithInt: [rememberPassword state]]
		   forKey: @"REMEMBERPASSWORD"];

  [transportMethod setObject: [personalProfilePopUpButton titleOfSelectedItem]
		   forKey: @"PERSONAL_PROFILE"];
  
  [transportMethod setObject: [NSNumber numberWithInt: [authenticateUsingButton state]]
		   forKey: @"SMTP_AUTH"];
  
  [transportMethod setObject: [supportedMechanismsPopUp titleOfSelectedItem]
		   forKey: @"SMTP_AUTH_MECHANISM"];
  
  [NSApp stopModal];
  [self close];
}


- (IBAction) cancelClicked: (id) sender
{
  [NSApp abortModal];
  [self close];
}


- (IBAction) transportMethodHasChanged: (id) sender
{
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];

  // If we selected Mailer
  if ( [transportMethodPopUpButton indexOfSelectedItem] == 0 )
    {
      [mailerField setEditable: YES];
      [smtpHostField setEditable: NO];
      [smtpUsernameField setEditable: NO];
      [smtpPasswordSecureField setEditable: NO];

      // We set the next key views
#ifndef MACOSX
      [transportMethodPopUpButton setNextKeyView: mailerField];
      [mailerField setNextKeyView: personalProfilePopUpButton];
      [smtpHostField setNextKeyView: nil];
      [smtpHostField setPreviousKeyView: nil];
      [smtpUsernameField setNextKeyView: nil];
      [smtpUsernameField setPreviousKeyView: nil];
      [smtpPasswordSecureField setNextKeyView: nil];
      [smtpPasswordSecureField setPreviousKeyView: nil];
      [personalProfilePopUpButton setNextKeyView: cancelButton];
      [cancelButton setNextKeyView: okButton];
      [okButton setNextKeyView: transportMethodPopUpButton];
#endif
    }
  else
    {
      [mailerField setEditable: NO];
      [smtpHostField setEditable: YES];
      [smtpUsernameField setEditable: YES];
      [smtpPasswordSecureField setEditable: YES];

      // We set the next key views
#ifndef MACOSX
      [transportMethodPopUpButton setNextKeyView: smtpHostField]; // FIXME
      [mailerField setNextKeyView: nil];
      [mailerField setPreviousKeyView: nil];
      [smtpHostField setNextKeyView: authenticateUsingButton];
      [authenticateUsingButton setNextKeyView: supportedMechanismsButton];
      [supportedMechanismsButton setNextKeyView: supportedMechanismsPopUp];
      [supportedMechanismsPopUp setNextKeyView: smtpUsernameField];
      [smtpUsernameField setNextKeyView: smtpPasswordSecureField];
      [smtpPasswordSecureField setNextKeyView: personalProfilePopUpButton];
      [personalProfilePopUpButton setNextKeyView: cancelButton];
      [cancelButton setNextKeyView: okButton];
      [okButton setNextKeyView: transportMethodPopUpButton];
#endif
    }
}


- (IBAction) supportedMechanismsButtonClicked: (id) sender
{
  NSArray *anArray;
  SMTP *aSMTP;
  int i;

  [supportedMechanismsPopUp removeAllItems];
  [supportedMechanismsPopUp addItemWithTitle: @"Most secure"];

  aSMTP = [[SMTP alloc] initWithName: [smtpHostField stringValue]];
  
  anArray = [aSMTP supportedAuthenticationMechanisms];

  for (i = 0; i < [anArray count]; i++)
    {
      [supportedMechanismsPopUp addItemWithTitle: [anArray objectAtIndex: i]];
    }

  [aSMTP close];
  RELEASE(aSMTP);
}


- (IBAction) authenticateUsingButtonClicked: (id) sender
{
  BOOL aBOOL;

  aBOOL = ([authenticateUsingButton state] == NSOnState ? YES : NO);
  
  [smtpUsernameField setEditable: aBOOL];
  [smtpPasswordSecureField setEditable: aBOOL];
  [supportedMechanismsButton setEnabled: aBOOL];
  [supportedMechanismsPopUp setEnabled: aBOOL];
  [rememberPassword setEnabled: aBOOL];
}

//
// delegate methods
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  // We do nothing
}

- (void) windowDidLoad
{

}

//
// access/mutation methods
//
- (NSMutableDictionary *) transportMethod
{
  return transportMethod;
}

- (void) setTransportMethod: (NSMutableDictionary *) theTransportMethod
{
  if ( theTransportMethod )
    {
      [[self window] setTitle: _(@"Edit a transport method")];

      RETAIN(theTransportMethod);
      RELEASE(transportMethod);
      transportMethod = theTransportMethod;

      // We fill our fields
      [transportMethodPopUpButton selectItemAtIndex: ([[transportMethod objectForKey: @"TRANSPORT_METHOD"] intValue] - 1) ];
      [mailerField setStringValue: [transportMethod objectForKey: @"MAILER_PATH"]];
      [smtpHostField setStringValue: [transportMethod objectForKey: @"SMTP_HOST"]];
      [smtpUsernameField setStringValue: [transportMethod objectForKey: @"SMTP_USERNAME"]];
      [smtpPasswordSecureField setStringValue: [Utilities decryptPassword: [transportMethod objectForKey: @"SMTP_PASSWORD"]
							  withKey: [transportMethod objectForKey: @"SMTP_HOST"]] ];
      [personalProfilePopUpButton selectItemWithTitle: [transportMethod objectForKey: @"PERSONAL_PROFILE"]];
      
      // For SMTP AUTH
      if ( [transportMethod objectForKey: @"SMTP_AUTH"] )
	{
	  [authenticateUsingButton setState: [[transportMethod objectForKey: @"SMTP_AUTH"] intValue]];
	  [supportedMechanismsPopUp addItemWithTitle: [transportMethod objectForKey: @"SMTP_AUTH_MECHANISM"]];
	  [supportedMechanismsPopUp selectItemWithTitle: [transportMethod objectForKey: @"SMTP_AUTH_MECHANISM"]];
	}

      if ( [transportMethod objectForKey: @"REMEMBERPASSWORD"] )
	{
	  [rememberPassword setState: [[transportMethod objectForKey: @"REMEMBERPASSWORD"] intValue]];
	}
      else
	{
	  [rememberPassword setState: NSOffState];
	}
      
      [self transportMethodHasChanged: nil];
      [self authenticateUsingButtonClicked: nil];
    }
  else
    {
      RELEASE(transportMethod);
      transportMethod = nil;
    }
}

//
// private methods
//
- (void) _loadPersonalProfilesInPopUpButton
{
  NSArray *allKeys;
  int i;

  [personalProfilePopUpButton removeAllItems];

  allKeys = [[[NSUserDefaults standardUserDefaults] objectForKey: @"PERSONAL"] allKeys];

  for (i = 0; i < [allKeys count]; i++)
    {
      [personalProfilePopUpButton addItemWithTitle: [allKeys objectAtIndex: i]];
    }

  [personalProfilePopUpButton selectItemAtIndex: 0];
}

@end
