/*
**  SendingViewController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "SendingViewController.h"

#import "GNUMailConstants.h"
#import "NSStringExtensions.h"
#import "TransportMethodEditorWindowController.h"

#ifndef MACOSX
#import "SendingView.h"
#endif

static SendingViewController *singleInstance = nil;

@implementation SendingViewController

- (id) initWithNibName: (NSString *) theName
{
  self = [super init];

#ifdef MACOSX
  
  if (! [NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSLog(@"Fatal error occured while loading the SendingView nib file");
      AUTORELEASE(self);
      return nil;
    }

  RETAIN(view);

  transportMethodColumn = [transportTableView tableColumnWithIdentifier: @"transport-method"];
  transportValueColumn = [transportTableView tableColumnWithIdentifier: @"transport-value"];
  headerKeyColumn = [headerTableView tableColumnWithIdentifier: @"header-key"];
  headerValueColumn = [headerTableView tableColumnWithIdentifier: @"header-value"];
  
  [transportTableView setTarget: self];
  [transportTableView setDoubleAction: @selector(editTransportMethod:)];
  
#else
  // We link our view
  view = [[SendingView alloc] initWithParent: self];
  [view layoutView];
  
  // We link our outlets
  transportTableView = [view transportTableView];
  transportMethodColumn = [view transportMethodColumn];
  transportValueColumn = [view transportValueColumn];

  headerTableView = [view headerTableView];
  headerKeyColumn = [view headerKeyColumn];
  headerValueColumn = [view headerValueColumn];
  
  headerKeyField = [view headerKeyField];
  headerValueField = [view headerValueField];

#endif

  // We first initialize our dictionary of sending values and additional headers
  allSendingValues = [[NSMutableArray alloc] init];
  allAdditionalHeaders = [[NSMutableDictionary alloc] init];

  // We get our defaults for this panel
  [self initializeFromDefaults];

  return self;
}

- (void) dealloc
{
  singleInstance = nil;
  
  RELEASE(allSendingValues);
  RELEASE(allAdditionalHeaders);
  RELEASE(view);

  [super dealloc];
}


//
// deletage methods
//


- (id)           tableView: (NSTableView *) aTableView
 objectValueForTableColumn: (NSTableColumn *) aTableColumn
                       row:(int) rowIndex
{
  if ( aTableView == transportTableView )
    {
      NSDictionary *aDictionary;

      aDictionary = [allSendingValues objectAtIndex: rowIndex];
      
      if (aTableColumn == transportMethodColumn)
	{
	  return ([[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP ? _(@"SMTP") : _(@"Mailer"));
	}
      else
	{
	  if ( [[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP )
	    {
	      return [aDictionary objectForKey: @"SMTP_HOST"];
	    }
	  else
	    {
	      return [aDictionary objectForKey: @"MAILER_PATH"];
	    }
	}
    }
  else if ( aTableView == headerTableView )
    {
      NSArray *anArray;
      
      anArray = [[allAdditionalHeaders allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
      
      if (aTableColumn == headerKeyColumn)
	{
	  return [anArray objectAtIndex: rowIndex];
	}
      else
	{
	  return [allAdditionalHeaders objectForKey: [anArray objectAtIndex: rowIndex]];
	}
    }

  // Never reached
  return @"";
}


- (void) tableViewSelectionDidChange:(NSNotification *) aNotification
{
  if ( [aNotification object] == headerTableView &&
       [headerTableView selectedRow] >= 0 )
    {
      NSArray *anArray;
      
      anArray = [[allAdditionalHeaders allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
      
      [headerKeyField setStringValue: [anArray objectAtIndex: [headerTableView selectedRow]]];
      [headerValueField setStringValue:
			  [allAdditionalHeaders objectForKey: [anArray objectAtIndex: 
									 [headerTableView selectedRow]]]];
    }
}

- (int) numberOfRowsInTableView: (NSTableView *) aTableView
{
  if ( aTableView == transportTableView )
    {
      if ( allSendingValues )
	{
	  return [allSendingValues count];
	}
    }
  else if ( aTableView == headerTableView )
    {
      if ( allAdditionalHeaders )
	{
	  return [allAdditionalHeaders count];
	}
    }

  // Never reached
  return 0;
}

- (void) tableView: (NSTableView *) aTableView
    setObjectValue: (id) anObject
    forTableColumn: (NSTableColumn *) aTableColumn
	       row: (int) rowIndex
{
  if ( aTableView == headerTableView )
    {
      
      NSArray *anArray;
      
      anArray = [[allAdditionalHeaders allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
      
      if ( aTableColumn == headerKeyColumn )
	{
	  NSString *aKey, *aValue;

	  aKey =  [anArray objectAtIndex: rowIndex];
	  aValue = [allAdditionalHeaders objectForKey: aKey];
	  RETAIN(aValue);
	  [allAdditionalHeaders removeObjectForKey: aKey];
	  [allAdditionalHeaders setObject: aValue forKey: anObject];
	  RELEASE(aValue);

	  [headerTableView reloadData];
	  [headerTableView setNeedsDisplay:YES];
	}
      else if ( aTableColumn == headerValueColumn )
	{
	  [allAdditionalHeaders setObject: anObject forKey: [anArray objectAtIndex: rowIndex]];
	}
    }
}

//
//
//
- (IBAction) addHeader: (id) sender
{
  NSString *aKey, *aValue;

  aKey = [[headerKeyField stringValue] stringByTrimmingWhiteSpaces];
  aValue = [[headerValueField stringValue] stringByTrimmingWhiteSpaces];

  if ( [aKey length] == 0 || [aValue length] == 0 )
    {
      NSBeep();
      return;
    }
  else
    {
      [allAdditionalHeaders setObject: aValue forKey: aKey];
      
      [headerKeyField setStringValue: @""];
      [headerValueField setStringValue: @""];
      [headerTableView reloadData];
    }
}


//
//
//
- (IBAction) removeHeader: (id) sender
{
  if ([headerTableView selectedRow] >= 0)
    {
      NSArray *anArray;
      NSString *aKey;
      
      anArray = [[allAdditionalHeaders allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
      aKey = [anArray objectAtIndex: [headerTableView selectedRow]];
      
      if ( aKey )
	{
	  [allAdditionalHeaders removeObjectForKey: aKey];
	  [headerTableView reloadData];
	  [headerTableView setNeedsDisplay:YES];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) addTransportMethod: (id) sender
{
  TransportMethodEditorWindowController *aTransportMethodEditorWindowController;
  int result;

  aTransportMethodEditorWindowController = [[TransportMethodEditorWindowController alloc]
					     initWithWindowNibName: @"TransportMethodEditorWindow"];

  result = [NSApp runModalForWindow: [aTransportMethodEditorWindowController window]];
  
  // We must add...
  if (result == NSRunStoppedResponse)
    {
      [allSendingValues addObject: [aTransportMethodEditorWindowController transportMethod]];
      [transportTableView reloadData];
    }

  RELEASE(aTransportMethodEditorWindowController);
}


//
//
//
- (IBAction) editTransportMethod: (id) sender
{
  TransportMethodEditorWindowController *aTransportMethodEditorWindowController;
  int result;

  if ( [transportTableView numberOfSelectedRows] != 1 )
    {
      NSBeep();
      return;
    }
  
  aTransportMethodEditorWindowController = [[TransportMethodEditorWindowController alloc]
					     initWithWindowNibName: @"TransportMethodEditorWindow"];
  
  [aTransportMethodEditorWindowController setTransportMethod:
					    [allSendingValues objectAtIndex:
								[transportTableView selectedRow]] ];
  
  result = [NSApp runModalForWindow: [aTransportMethodEditorWindowController window]];
  
  // We must add...
  if (result == NSRunStoppedResponse)
    {
      [allSendingValues replaceObjectAtIndex: [transportTableView selectedRow]
			withObject: [aTransportMethodEditorWindowController transportMethod]];
      [transportTableView reloadData];
    }

  RELEASE(aTransportMethodEditorWindowController);
}


//
//
//
- (IBAction) removeTransportMethod: (id) sender
{
  int choice;

  if ( [transportTableView numberOfSelectedRows] != 1 )
    {
      NSBeep();
      return;
    }
  
  choice = NSRunAlertPanel(_(@"Delete..."),
			   _(@"Are you sure you want to delete this transport method?"),
			   _(@"Cancel"), // default
			   _(@"Yes"),    // alternate
			   NULL);
  
  if (choice == NSAlertAlternateReturn)
    {
      [allSendingValues removeObjectAtIndex: [transportTableView selectedRow]];
      [transportTableView reloadData];

      // We set a selection, if we can.
      if ( [allSendingValues count] )
	{
	  [transportTableView selectRow: 0
			      byExtendingSelection: NO];
	}
    }
}


//
// access methods
//

- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];
  
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"MailIcon_send" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"Sending");
}

- (NSView *) view
{
  return view;
}

- (BOOL) hasChangesPending
{
  return YES;
}


//
//
//
- (void) initializeFromDefaults
{
  // For the sending values  
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"] )
    {
      [allSendingValues addObjectsFromArray:
      			  [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"] ];
      [transportTableView reloadData];
    }

  // For the additional headers (ADDITIONALOUTGOINGHEADERS)  
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"ADDITIONALOUTGOINGHEADERS"] )
    {
      [allAdditionalHeaders addEntriesFromDictionary:[[NSUserDefaults standardUserDefaults] 
						       objectForKey: @"ADDITIONALOUTGOINGHEADERS"] ];
      [headerTableView reloadData];
    }
}


- (void) saveChanges
{
  // For the sending values
  [[NSUserDefaults standardUserDefaults] setObject: allSendingValues 
					 forKey: @"SENDING"];

  // For the additional headers
  [[NSUserDefaults standardUserDefaults] setObject: allAdditionalHeaders 
					 forKey: @"ADDITIONALOUTGOINGHEADERS"];

  // We inform that our personal profiles have changed
  [[NSNotificationCenter defaultCenter]
    postNotificationName: TransportMethodsHaveChanged
    object: nil
    userInfo: nil];
}

- (void) windowDidLoad
{

    NSLog(@"window did load");
}

//
// class methods
//
+ (id) singleInstance
{
  if (! singleInstance)
    {
      singleInstance = [[SendingViewController alloc] initWithNibName: @"SendingView"];
    }

  return singleInstance;
}

@end
