/*
**  FilteringViewController.m
**
**  Copyright (c) 2001
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "FilteringViewController.h"

#import "GNUMailConstants.h"
#import "Filter.h"
#import "FilterEditorWindowController.h"
#import "FilterManager.h"
#import "ExtendedTableView.h"
#import "ExtendedTableColumn.h"

#ifndef MACOSX
#import "FilteringView.h"
#endif

static FilteringViewController *singleInstance = nil;

@implementation FilteringViewController

- (id) initWithNibName: (NSString *) theName
{
  BOOL isDir;

  self = [super init];

  if ( [[NSFileManager defaultManager] fileExistsAtPath: PathToFilters()
				       isDirectory: &isDir] &&
       ! isDir )
    {
      filterManager = [FilterManager filtersFromDisk];
      RETAIN(filterManager);
    }
  else
    {
      // create a new address book and save it to disk
      filterManager = [[FilterManager alloc] init];
      [NSArchiver archiveRootObject: filterManager toFile: PathToFilters()];
    }

#ifdef MACOSX
  
  if (! [NSBundle loadNibNamed: theName
		  owner: self] )
    {
      NSLog(@"Fatal error occured while loading the FilteringView nib file");
      AUTORELEASE(self);
      return nil;
    }
  
  RETAIN(view);
  
  rulesColumn = [tableView tableColumnWithIdentifier: @"rules"];
  
  [tableView setTarget: self];
  [tableView setDoubleAction: @selector(edit:)];

#else
  // We link our view
  view = [[FilteringView alloc] initWithParent: self];
  [view layoutView];

  // We link our outlets
  tableView = [view tableView];
  rulesColumn = [view rulesColumn];
  activeColumn = [view activeColumn];
  add = [view add];
  delete = [view delete];
  duplicate = [view duplicate];
  edit = [view edit];
#endif

#ifdef MACOSX
  {
    NSButtonCell *cell = [[NSButtonCell alloc] init];
    ExtendedTableColumn *etc;
    [cell setButtonType: NSSwitchButton];
    [cell setImagePosition: NSImageOnly];
    [cell setControlSize: NSSmallControlSize];
    etc = [tableView tableColumnWithIdentifier: @"active"];
    [etc setDataCell: cell];
    [etc setShouldUseMouse: YES];
    [etc setShouldUseAndSetState: YES];
    RELEASE(cell);
  }
#endif
    
  // We get our defaults for this panel
  [self initializeFromDefaults];

  return self;
}


- (void) dealloc
{
  singleInstance = nil;

  RELEASE(filterManager);
  RELEASE(view);

  [super dealloc];
}

//
// Data Source methods
//

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [[filterManager filters] count];
}

- (id) tableView: (NSTableView *)aTableView
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  Filter *aFilter;
  
  aFilter = [[filterManager filters] objectAtIndex: rowIndex];

  if ( aTableColumn == rulesColumn )
    { 
      return [aFilter description];
    }
  
  return ( [aFilter isActive] ? _(@"Yes") : _(@"No") );
}

- (int) tableView: (NSTableView *)aTableView
stateForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  Filter *aFilter;
  
  aFilter = [[filterManager filters] objectAtIndex: rowIndex];

  return ( [aFilter isActive] ? NSOnState : NSOffState );
}

- (void) tableView: (NSTableView *)aTableView
	 setState: (int) aState
   forTableColumn: (NSTableColumn *)aTableColumn 
	      row: (int)rowIndex
{
  Filter *aFilter;
  
  aFilter = [[filterManager filters] objectAtIndex: rowIndex];
  if (aState == NSOnState)
    {
      [aFilter setIsActive: YES];
    }
  else if (aState == NSOffState)
    {
      [aFilter setIsActive: NO];
    }
}

//
// action methods
//

- (IBAction) add : (id) sender
{
  FilterEditorWindowController *filterEditorWindowController;
  int result;

  filterEditorWindowController = [[FilterEditorWindowController alloc] initWithWindowNibName: @"FilterEditorWindow"];
  
  [filterEditorWindowController setFilterManager: filterManager];
  [filterEditorWindowController setFilter: nil];

  result = [NSApp runModalForWindow: [filterEditorWindowController window]];
  
  if (result == NSRunStoppedResponse)
    {
      [tableView reloadData];
    }
}


//
//
//
- (IBAction) delete: (id) sender
{
  int choice;

  if ([tableView numberOfSelectedRows] == 0 || [tableView numberOfSelectedRows] > 1)
    {
      NSBeep();
      return;
    }
  
  choice = NSRunAlertPanel(_(@"Delete..."),
			   _(@"Are you sure you want to delete this filter?"),
			   _(@"Cancel"), // default
			   _(@"Yes"),    // alternate
			   nil);

  // If we delete it...
  if (choice == NSAlertAlternateReturn)
    {
      [filterManager removeFilter: [filterManager filterAtIndex:[tableView selectedRow]]];
      [tableView reloadData];
    }
}


//
//
//
- (IBAction) duplicate: (id) sender
{
  if ([tableView numberOfSelectedRows] == 0 || [tableView numberOfSelectedRows] > 1)
    {
      NSBeep();
      return;
    }
  else
    {
      Filter *aFilter;
      
      aFilter = [[filterManager filterAtIndex: [tableView selectedRow]] copy];
      [aFilter setDescription: [NSString stringWithFormat: @"%@ (copy)", [aFilter description]]];
      [filterManager addFilter: aFilter];
      [tableView reloadData];
    }
}

//
//
//
- (IBAction) edit: (id) sender

{
  FilterEditorWindowController *filterEditorWindowController;
  Filter *aFilter;
  int result;

  if ([tableView numberOfSelectedRows] == 0 || [tableView numberOfSelectedRows] > 1)
    {
      NSBeep();
      return;
    }
 
  aFilter = [filterManager filterAtIndex: [tableView selectedRow]];
			      
  filterEditorWindowController = [[FilterEditorWindowController alloc] 
				     initWithWindowNibName: @"FilterEditorWindow"];
  
  [[filterEditorWindowController window] setTitle: _(@"Edit a filter")];
  [filterEditorWindowController setFilterManager: filterManager];
  [filterEditorWindowController setFilter: aFilter];


  result = [NSApp runModalForWindow: [filterEditorWindowController window]];

  if (result == NSRunStoppedResponse)
    {
      NSLog(@"We update...");
      [tableView reloadData];
    }
}


//
//
//
- (IBAction) moveDown: (id) sender

{
  if ([tableView numberOfSelectedRows] == 0 ||
      [tableView numberOfSelectedRows] > 1 ||
      [tableView selectedRow] == ([[filterManager filters] count] - 1))
    {
      NSBeep();
      return;
    }
  else
    {
      Filter *aFilter;
      int index;
      
      index = [tableView selectedRow];
      aFilter = [filterManager filterAtIndex: index];
      RETAIN(aFilter);
      
      [filterManager removeFilter: aFilter];
      [filterManager addFilter: aFilter
		     atIndex: (index + 1)];
      [tableView reloadData];
      [tableView selectRow: (index + 1) byExtendingSelection: NO];
      
      RELEASE(aFilter);
    }
}


//
//
//
- (IBAction) moveUp: (id) sender

{
  if ([tableView numberOfSelectedRows] == 0 ||
      [tableView numberOfSelectedRows] > 1 ||
      [tableView selectedRow] == 0)
    {
      NSBeep();
      return;
    }
  else
    {
      Filter *aFilter;
      int index;
      
      index = [tableView selectedRow];
      aFilter = [filterManager filterAtIndex: index];
      RETAIN(aFilter);
      
      [filterManager removeFilter: aFilter];
      [filterManager addFilter: aFilter
		     atIndex: (index - 1)];
      [tableView reloadData];
      [tableView selectRow: (index - 1) byExtendingSelection: NO];
      
      RELEASE(aFilter);
    }
}


//
// access methods
//

- (NSImage *) image
{
  NSBundle *aBundle;
  
  aBundle = [NSBundle bundleForClass: [self class]];
  
  return AUTORELEASE([[NSImage alloc] initWithContentsOfFile:
					[aBundle pathForResource: @"Filtering" ofType: @"tiff"]]);
}

- (NSString *) name
{
  return _(@"Filtering");
}

- (NSView *) view
{
  return view;
}


//
//
//

- (BOOL) hasChangesPending
{
  return YES;
}


- (void) initializeFromDefaults
{

}


- (void) saveChanges
{
  [filterManager synchronize];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: FiltersHaveChanged
    object: nil
    userInfo: nil];
}

//
// class methods
//
+ (id) singleInstance
{
  if (! singleInstance)
    {
      singleInstance = [[FilteringViewController alloc] initWithNibName: @"FilteringView"];
    }

  return singleInstance;
}

@end
