/*
**  AddressBookController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**          Jonathan B. Leffert <jonathan@leffert.net>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "AddressBookController.h"

#import "Address.h"
#import "AddressBook.h"
#import "AddressTaker.h"

#ifndef MACOSX
#import "AddressBookWindow.h"
#endif

#import "GNUMail.h"
#import "GNUMailConstants.h"
#import "Group.h"
#import "NSStringExtensions.h"
#import "Utilities.h"

#import <Pantomime/InternetAddress.h>

static AddressBookController *singleInstance = nil;

@implementation AddressBookController

// we use the window ivar to represent the address book window, but we
// expect only a single instance of the controller app-wide.

- (id) initWithWindowNibName: (NSString *) windowNibName
{

#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
 
#else
  AddressBookWindow *theWindow;
 
  theWindow =
    [[AddressBookWindow alloc] initWithContentRect: NSMakeRect(200,200,480,345)
			       styleMask: (NSTitledWindowMask |
					   NSClosableWindowMask |
					   NSMiniaturizableWindowMask)
			       backing: NSBackingStoreBuffered
			       defer: YES];

  self = [super initWithWindow: theWindow];

  [theWindow layoutWindow];
  [theWindow setDelegate: self];
  [theWindow setMinSize: [theWindow frame].size];

  // We link our outlets
  browser = [theWindow browser];
  nameTextField = [theWindow nameTextField];
  add = [theWindow add];
  delete = [theWindow delete];
  modify = [theWindow modify];
  popup = [theWindow popup];

  RELEASE(theWindow);
#endif

  // We set some initial parameters
  [[self window] setTitle: _(@"Address Book")];

  // Initially, we disable most buttons
  [add setEnabled: YES];
  [add setAction: @selector(addGroup:)];
  [add setTitle: _(@"Add Group")];
  
  [delete setEnabled: NO];
  [delete setTitle: _(@"Delete Group")];
  
  [modify setEnabled: NO];
  
  [popup setEnabled: NO];

#ifdef MACOSX
  [browser setAction: @selector(selectionInBrowserHasChanged:)];
  [browser setDoubleAction: @selector(doubleClicked:)];
#endif

  [self _loadItemsInMoveToPopUpButton];

  return self;
}

- (void) dealloc
{
  RELEASE(addressBook);

  [super dealloc];
}


//
// This method is used to add an address to the address book
// from an 'external' object. It searches for the group
// "General" (localized) and if it can't find it, it creates a new one.
//
- (void) addToAddressBook: (Address *) theAddress
{
  if ( theAddress &&
       [theAddress isKindOfClass: [Address class]] )
    {
      NSArray *allGroups;
      Group *aGroup;
      int i;
      
      allGroups = [[self addressBook] allGroups];
      aGroup = nil;

      for (i = 0; i < [allGroups count]; i++)
	{
	  aGroup = [allGroups objectAtIndex: i];
	  
	  if ([[aGroup name] caseInsensitiveCompare: _(@"General")] == NSOrderedSame)
	    {
	      break;
	    }

	  aGroup = nil;
	}

      // We haven't found a group "General", let's create one
      if ( !aGroup )
	{
	  aGroup = [[Group alloc] initWithName: _(@"General")];
	  [[self addressBook] addGroup: aGroup];
          [[self addressBook] sort];
	}

      [aGroup addAddress: theAddress];
      [aGroup sort];

      [[self addressBook] synchronize];

      // Let's refresh our browser
      [browser loadColumnZero];
    }
  else
    {
      NSBeep();
    }
}

//
// action methods
//
- (IBAction) addAddress: (id) sender
{
  NSString *aString;
  
  aString = [[nameTextField stringValue] stringByTrimmingWhiteSpaces];

  if ( [aString length] )
    {
      InternetAddress *anInternetAddress;
      Address *anAddress;
      NSArray *allGroups;
      Group *aGroup;

      allGroups = [[self addressBook] allGroups];
      
      aGroup = [allGroups objectAtIndex: [browser selectedRowInColumn: 0]];
      
      anInternetAddress = [[InternetAddress alloc] initWithString: aString];
      anAddress = [[Address alloc] init];

      [anAddress setName: [anInternetAddress personal]];
      [anAddress setEmailAddress: [anInternetAddress address]];
      
      [aGroup addAddress: anAddress];

      RELEASE(anInternetAddress);
      RELEASE(anAddress);

      [aGroup sort];

      [[self addressBook] synchronize];
     
      [browser reloadColumn: 1];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) addGroup: (id) sender
{
  NSString *aString;
  
  aString = [[nameTextField stringValue] stringByTrimmingWhiteSpaces];

  if ( [aString length] )
    {
      NSArray *allGroups;
      Group *aGroup;
      int i;

      allGroups = [[self addressBook] allGroups];
      
      // We verify that the group isn't already there.
      for (i = 0; i < [allGroups count]; i++)
	{
	  aGroup = [allGroups objectAtIndex: i];

	  if ( [[aGroup name] caseInsensitiveCompare: aString] == NSOrderedSame )
	    {
	      NSRunInformationalAlertPanel(_(@"Error!"),
					   _(@"This group already exist! Please choose an other name."),
					   _(@"OK"),
					   NULL, 
					   NULL,
					   NULL);
	      return;
	    }
	}

      // Everything is alright, let's add our new group
      aGroup = [[Group alloc] initWithName: aString];
      
      [[self addressBook] addGroup: aGroup];
      [[self addressBook] sort];
      [[self addressBook] synchronize];
      
      RELEASE(aGroup);

      [browser loadColumnZero];
      [self _loadItemsInMoveToPopUpButton];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) deleteAddress: (id) sender
{
  if ( [browser selectedCellInColumn: 1] )
    {
      int choice;
      
      choice = NSRunAlertPanel(_(@"Delete..."),
			       _(@"Are you sure you want to delete this address?"),
			       _(@"No"),     // default
			       _(@"Yes"),    // alternate
			       NULL);        // other button
      
      
      if (choice == NSAlertAlternateReturn)
	{
	  Group *aGroup;
	  
	  aGroup = [[[self addressBook] allGroups] objectAtIndex: [browser selectedRowInColumn: 0]];
	  
	  // We verify to be sure we aren't trying to delete our fake address
	  if ( [[aGroup addresses] count] == 0)
	    {
	      NSBeep();
	    }
	  else
	    {
	      Address *anAddress;
	      
	      anAddress = [[aGroup addresses] objectAtIndex: [browser selectedRowInColumn: 1]];

	      [aGroup removeAddress: anAddress];
	      [[self addressBook] synchronize];
	      
	      [browser reloadColumn: 1];
	    }
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) deleteGroup: (id) sender
{
  
  if ( [browser selectedCellInColumn: 0] )
    {
      int choice;
      
      choice = NSRunAlertPanel(_(@"Delete..."),
			       _(@"Are you sure you want to delete this group?"),
			       _(@"No"),     // default
			       _(@"Yes"),    // alternate
			       NULL);        // other button
      
      
      if (choice == NSAlertAlternateReturn)
	{
	  Group *aGroup;
	  
	  aGroup = [[[self addressBook] allGroups] objectAtIndex: [browser selectedRowInColumn: 0]];
	  
	  [[self addressBook] removeGroup: aGroup];
	  [[self addressBook] synchronize];
	  
	  [browser loadColumnZero];
	  [self _loadItemsInMoveToPopUpButton];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) doubleClicked: (id) sender
{
  int modifier = [[[sender window] currentEvent] modifierFlags];

  if ( (modifier & NSControlKeyMask) &&
       ! (modifier & NSShiftKeyMask) )
    {
      return [self ccClicked: sender];
    }
  else if ( ! (modifier & NSControlKeyMask) &&
            (modifier & NSShiftKeyMask) )
    {
      return [self bccClicked: sender];
    }
  else
    {
      return [self toClicked: sender];
    }
}


//
//
//
- (IBAction) toClicked: (id) sender
{
  // We selected an address...
  if ([browser selectedCellInColumn: 1] &&
      [GNUMail lastAddressTakerWindowOnTop] )
    {
      unsigned int row;
      
      row = [browser selectedRowInColumn: 1];

      if ( row >= 0 )
	{
	  NSArray *allGroups, *allAddresses;
	  
	  allGroups = [[self addressBook] allGroups];
	  allAddresses = [[allGroups objectAtIndex: [browser selectedRowInColumn: 0]] addresses];

	  [(id<AddressTaker>)[GNUMail lastAddressTakerWindowOnTop]
			     takeToAddress: [allAddresses objectAtIndex: row]];
	}
      else
	{
	  NSBeep();
	}
    }
  // We selected a group
  else if ([browser selectedCellInColumn: 0] &&
	   [GNUMail lastAddressTakerWindowOnTop] )
    {
      NSArray *allAddresses;
      Group *aGroup;
      int i;

      aGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];
      allAddresses = [aGroup addresses];

      for (i = 0; i < [allAddresses count]; i++)
	{
	  [(id<AddressTaker>)[GNUMail lastAddressTakerWindowOnTop]
			     takeToAddress: [allAddresses objectAtIndex: i]];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) ccClicked: (id) sender
{
  // We selected an address...
  if ([browser selectedCellInColumn: 1] &&
      [GNUMail lastAddressTakerWindowOnTop] )
    {
      unsigned int row;
      
      row = [browser selectedRowInColumn: 1];

      if ( row >= 0 )
	{
	  NSArray *allGroups, *allAddresses;
	  
	  allGroups = [[self addressBook] allGroups];
	  allAddresses = [[allGroups objectAtIndex: [browser selectedRowInColumn: 0]] addresses];

	  [(id<AddressTaker>)[GNUMail lastAddressTakerWindowOnTop]
			     takeCcAddress: [allAddresses objectAtIndex: row]];
	}
      else
	{
	  NSBeep();
	}
    }
  // We selected a group
  else if ([browser selectedCellInColumn: 0] &&
	   [GNUMail lastAddressTakerWindowOnTop] )
    {
      NSArray *allAddresses;
      Group *aGroup;
      int i;

      aGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];
      allAddresses = [aGroup addresses];

      for (i = 0; i < [allAddresses count]; i++)
	{
	  [(id<AddressTaker>)[GNUMail lastAddressTakerWindowOnTop]
			     takeCcAddress: [allAddresses objectAtIndex: i]];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) bccClicked: (id) sender
{
  // We selected an address...
  if ([browser selectedCellInColumn: 1] &&
      [GNUMail lastAddressTakerWindowOnTop] )
    {
      unsigned int row;
      
      row = [browser selectedRowInColumn: 1];

      if ( row >= 0 )
	{
	  NSArray *allGroups, *allAddresses;
	  
	  allGroups = [[self addressBook] allGroups];
	  allAddresses = [[allGroups objectAtIndex: [browser selectedRowInColumn: 0]] addresses];

	  [(id<AddressTaker>)[GNUMail lastAddressTakerWindowOnTop]
			     takeBccAddress: [allAddresses objectAtIndex: row]];
	}
      else
	{
	  NSBeep();
	}
    }
  // We selected a group
  else if ([browser selectedCellInColumn: 0] &&
	   [GNUMail lastAddressTakerWindowOnTop] )
    {
      NSArray *allAddresses;
      Group *aGroup;
      int i;

      aGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];
      allAddresses = [aGroup addresses];

      for (i = 0; i < [allAddresses count]; i++)
	{
	  [(id<AddressTaker>)[GNUMail lastAddressTakerWindowOnTop]
			     takeBccAddress: [allAddresses objectAtIndex: i]];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) modifyAddress: (id) sender
{
  NSString *aString;
  
  aString = [[nameTextField stringValue] stringByTrimmingWhiteSpaces];

  if ( [aString length] &&
       [browser selectedCellInColumn: 1] )
    {
      InternetAddress *anInternetAddress;
      Address *anAddress;
      Group *aGroup;
      
      aGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];
      
      anAddress = [aGroup addressAtIndex: [browser selectedRowInColumn: 1]];

      anInternetAddress = [[InternetAddress alloc] initWithString: aString];
      [anAddress setName: [anInternetAddress personal]];
      [anAddress setEmailAddress: [anInternetAddress address]];
      
      RELEASE(anInternetAddress);
      
      [aGroup sort];

      [[self addressBook] synchronize];
     
      [browser reloadColumn: 1];
    }
  else
    {
      NSBeep();
    }

}


//
//
//
- (IBAction) modifyGroup: (id) sender
{
  NSString *aString;

  aString = [[nameTextField stringValue] stringByTrimmingWhiteSpaces];

  if ( [aString length] > 0 &&
       [browser selectedCellInColumn: 0] )
    {
      Group *aGroup;
      
      aGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];

      [aGroup setName: aString];
      [[self addressBook] sort];
      [[self addressBook] synchronize];

      [browser loadColumnZero];
      [self _loadItemsInMoveToPopUpButton];
    }
  else
    {
      NSBeep();
    }
}


- (IBAction) importClicked: (id) sender
{
  NSOpenPanel *oPanel;
  int result;
  
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection:NO];
  result = [oPanel runModalForDirectory: NSHomeDirectory()
		   file: nil 
		   types: nil];
  
  if (result == NSOKButton) 
    {
      NSArray *fileToOpen;
      
      fileToOpen = [oPanel filenames];
      
      if ([fileToOpen count] > 0)
	{
	  NSArray *allLines;
	  NSString *aString;
	  int i;

	  aString = [NSString stringWithContentsOfFile: [fileToOpen objectAtIndex: 0]];
	  
	  allLines = [aString componentsSeparatedByString: @"\n"];

	  for (i = 0; i < [allLines count]; i++)
	    {	      
	      NSString *aLine;
	      NSRange aRange;
 
	      aLine = [allLines objectAtIndex: i];	      
	      aRange = [aLine rangeOfString: @":"];
	      
	      // We verify if we've found a good line.
	      if ( aRange.length )
		{
		  NSArray *allAddresses;
		  NSString *aGroupName;
		  int j;
		  
		  Group *aGroup;

		  aGroupName = [aLine substringWithRange: NSMakeRange(0, aRange.location)];

		  // We got the group name, let's see if it already exists in our address book.
		  // We add it if it doesn't.
		  aGroup = [[self addressBook] groupForName: aGroupName];
		  
		  if (! aGroup )
		    {
		      aGroup = [[Group alloc] initWithName: aGroupName];
		      [[self addressBook] addGroup: aGroup];
		      AUTORELEASE(aGroup);
		    }

		  allAddresses = [[aLine substringFromIndex: aRange.location + 1]
				   componentsSeparatedByString: @","];
		  
		  for (j = 0; j < [allAddresses count]; j++)
		    {
		      InternetAddress *anInternetAddress;
		      Address *anAddress;
		      
		      anInternetAddress = [[InternetAddress alloc] initWithString: 
								     [allAddresses objectAtIndex: j]];

		      anAddress = [[Address alloc] init];
		      [anAddress setName: [anInternetAddress personal]];
		      [anAddress setEmailAddress: [anInternetAddress address]];
		      
		      [aGroup addAddress: anAddress];

		      RELEASE(anInternetAddress);
		      RELEASE(anAddress);
		    }

		  [browser loadColumnZero];
		  [[self addressBook] synchronize];
		}
	      else
		{
		  NSLog(@"Improperly formatted line. Ignored.");
		}
	    }
	}
    }
}

- (IBAction) selectionInPopUpHasChanged: (id) sender
{
  [popup synchronizeTitleAndSelectedItem];
  
  if ( [popup indexOfSelectedItem] <= 0 ||
       ![browser selectedCellInColumn: 1] )
    {
      NSBeep();
    }
  else
    {
      Group *selectedGroup, *aGroup;
      Address *anAddress;

      selectedGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];
      aGroup = [[self addressBook] groupForName: [popup titleOfSelectedItem]];

      // We verify that we aren't trying to transfer to the same group and that
      // we aren't trying to transfer our fake address
      if (aGroup == selectedGroup ||
	  [[selectedGroup addresses] count] == 0)
	{
	  NSBeep();
	  return;
	}
      
      // Everything is alright, let's move the address!
      anAddress = [selectedGroup addressAtIndex: [browser selectedRowInColumn: 1]];

      [aGroup addAddress: anAddress];
      [selectedGroup removeAddress: anAddress];
      
      [[self addressBook] synchronize];

      [browser reloadColumn: 1];
      [nameTextField setStringValue: @""];

      
      [popup selectItemAtIndex: 0];
      [popup synchronizeTitleAndSelectedItem];
    }
}

#ifdef MACOSX
- (IBAction) selectionInBrowserHasChanged: (id) sender
{
  int row, column;

  if ( [browser selectedCellInColumn: 1] )
    {
      row = [browser selectedRowInColumn: 1];
      column = 1;
    }
  else
    {
      row = [browser selectedRowInColumn: 0];
      column = 0;
    }
  
  [self browser: browser
	selectRow: row
	inColumn: column];
}
#endif


//
// delegate methods
//
- (void) windowDidLoad
{
  BOOL isDir;

  if ( [[NSFileManager defaultManager] fileExistsAtPath: PathToAddressBook()
				       isDirectory: &isDir] &&
       ! isDir )
    {
      // read the address book from disk
      addressBook = [AddressBook addressBookFromDisk];
      RETAIN(addressBook);
    }
  else
    {
      // create a new address book and save it to disk
      addressBook = [[AddressBook alloc] init];
      [NSArchiver archiveRootObject: addressBook 
		  toFile: PathToAddressBook()];
    }

#ifdef MACOSX   
  [browser loadColumnZero];
  [browser setNeedsDisplay: YES];
#endif

}

- (int)       browser: (NSBrowser *) sender
 numberOfRowsInColumn: (int) column
{
  if (column == 0)
    {
      return [[[self addressBook] allGroups] count];
    }
  else if (column == 1)
    {
      NSArray *allGroups;
      Group *aGroup;
      int count;

      allGroups = [[self addressBook] allGroups];
      aGroup = [allGroups objectAtIndex: [browser selectedRowInColumn: 0]];

      count = [[aGroup addresses] count];
 
      // We ALWAYS return at least one 'fake address'
      return (count == 0 ? 1 : count);
    }
  
  return 0;
}

- (void) browser: (NSBrowser *) sender
 willDisplayCell: (id) cell
	   atRow: (int) row
	  column: (int) column
{
  if (column == 0)
    {
      NSArray *allGroups;
      Group *aGroup;
      
      allGroups = [[self addressBook] allGroups];
      aGroup = [allGroups objectAtIndex: row];
      
      [cell setStringValue: [aGroup name]];
    }
  else if (column == 1)
    {
      NSArray *allGroups, *allAddresses;

      allGroups = [[self addressBook] allGroups];   
      allAddresses = [[allGroups objectAtIndex: [browser selectedRowInColumn: 0]] addresses];
      
      // We verify if we need to create a 'fake address'
      if ( [allAddresses count] == 0 )
	{
	  [cell setStringValue: _(@"< no address >")];
	}
      else
	{
	  [cell setStringValue: [[allAddresses objectAtIndex: row] description]];
	}
      
      [cell setLeaf: YES];
    }
}

- (BOOL) browser: (NSBrowser *) sender
       selectRow: (int) row
        inColumn: (int) column
{
  [add setEnabled: YES];
  [delete setEnabled: YES];
  [modify setEnabled: YES];

  if (column == 0)
    {
      NSArray *allGroups;
      Group *aGroup;

      allGroups = [[self addressBook] allGroups];
      
      aGroup = [allGroups objectAtIndex: row];
      [aGroup sort];

      [popup setEnabled: NO];

      [add setTitle: _(@"Add Group")];
      [add setAction: @selector(addGroup:)];

      [delete setTitle: _(@"Delete Group")];
      [delete setAction: @selector(deleteGroup:)];

      [modify setAction: @selector(modifyGroup:)];
      
      [nameTextField setStringValue: [aGroup name]];
    }
  else if (column == 1)
    {
      NSArray *allGroups, *allAddresses;

      allGroups = [[self addressBook] allGroups];
      allAddresses = [[allGroups objectAtIndex: [browser selectedRowInColumn: 0]] addresses];

      [popup setEnabled: YES];

      [add setTitle: _(@"Add Address")];
      [add setAction: @selector(addAddress:)];

      [delete setTitle: _(@"Delete Address")];
      [delete setAction: @selector(deleteAddress:)];

      [modify setAction: @selector(modifyAddress:)];

      // We verify if we selected our fake addresse
      if ( [allAddresses count] == 0 )
	{
	  [nameTextField setStringValue: _(@"Enter the contact's E-Mail address here")];
	}
      else
	{
	  [nameTextField setStringValue: [[allAddresses objectAtIndex: row] description]];
	}
    }

  return YES;
}

- (NSString *) browser: (NSBrowser *) sender
         titleOfColumn: (int) column
{
  if (column == 0)
    {
      return _(@"Groups");
    }
  else
    {
      if ( [browser selectedCellInColumn: 0] )
	{
	  Group *aGroup;
	  
	  aGroup = [[self addressBook] groupAtIndex: [browser selectedRowInColumn: 0]];
	 
	  return [aGroup name];
	}
    }

  return @"";
}

//
// access/mutation methods
//

- (AddressBook *) addressBook
{
  return addressBook;
}

- (void) setAddressBook: (AddressBook *) theBook
{
  RETAIN(theBook);
  RELEASE(addressBook);
  addressBook = theBook;
}


//
// class methods
//

+ (id) singleInstance
{
  if ( ! singleInstance )
    {
      singleInstance = [[self alloc] initWithWindowNibName: @"AddressBookWindow"];
    }
  return singleInstance;
}

@end


//
// Private methods
//
@implementation AddressBookController (Private)

- (void) _loadItemsInMoveToPopUpButton
{
  NSArray *allGroups;
  int i;

  allGroups = [[self addressBook] allGroups];

  [popup removeAllItems];
  [popup addItemWithTitle: _(@"Move to:")];

  for (i = 0; i < [allGroups count]; i++)
    {
      Group *aGroup;

      aGroup = [allGroups objectAtIndex: i];
      
      [popup addItemWithTitle: [aGroup name]];
    }
}

@end
