//  Gnomoradio - rainbow/http-server.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "http-server.h"
#include "hub-client.h"
#include "rainbow/util.h"
#include "config.h"

#include <errno.h>
#include <unistd.h>
#include <errno.h>

#include <algorithm>
#include <sstream>
#include <cstdio>
#include <cstdlib>

using namespace Glib;
using namespace std;

#define SERVER "Gnomoradio " VERSION
#define CRLF "\r\n"
#define ACCEPT_RANGES "Accept-Ranges: bytes" CRLF

// FIXME: keep alive

Rainbow::HttpServer::HttpServer (HubClient *hubclient)
	: sock(0),
	  hc(hubclient)
{
}

Rainbow::HttpServer::~HttpServer ()
{
	stop();
}

bool Rainbow::HttpServer::start (unsigned short port)
{
	// check if already started
	if (sock > 0)
		return true;

	// create socket
	struct sockaddr_in name;
	sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		logmsg << "HttpServer: Cannot create socket" << endl;
		return false;
	}
	
	// make socket non-blocking
	int fd_flags = fcntl(sock, F_GETFL, 0);
	if (fd_flags == -1) {
		logmsg << "HttpServer: fcntl failed" << endl;
		close(sock);
		sock = 0;
		return false;
	}
	fcntl(sock, F_SETFL, fd_flags | O_NONBLOCK);

	// bind
	name.sin_family = AF_INET;
	name.sin_port = htons(port);
	name.sin_addr.s_addr = htonl(INADDR_ANY);
	if (bind(sock, (struct sockaddr *) &name, sizeof(name)) < 0) {
		logmsg << "HttpServer: Cannot bind to port " << port << endl;
		close(sock);
		sock = 0;
		return false;
	}
	
	// listen
	const int max_pending_connections = 3;
	if (listen(sock, max_pending_connections) < 0) {
		logmsg << "HttpServer: Cannot listen" << endl;
		close(sock);
		sock = 0;
		return false;
	}

	// initialize thread system
	if (!thread_supported())
		thread_init();

	// accept incoming connections
	try_accept();

	// success!
	logmsg << "HttpServer: started" << endl;
	return true;
}

void Rainbow::HttpServer::stop ()
{
	if (sock <= 0)
		return; // already stopped
	close(sock);
	sock = 0;

	// tell threads we died
	for (set<ServerThread*>::iterator i = threads.begin(); i != threads.end(); ++i) {
		(*i)->mutex.lock();
		(*i)->this_exists = false;
		(*i)->mutex.unlock();
		close((*i)->fd);
	}
	threads.clear();

	logmsg << "HttpServer: stopped" << endl;
}

void Rainbow::HttpServer::try_accept ()
{
	struct sockaddr_in clientname;
	socklen_t size = sizeof(clientname);
	int conn;

	do {
		conn = accept(sock, (struct sockaddr*) &clientname, &size);
		if (conn < 0) {
			if (errno == EAGAIN || errno == EWOULDBLOCK) {
				signal_io().connect(sigc::mem_fun(*this, &HttpServer::on_sock_event),
						  sock, IO_IN | IO_PRI | IO_ERR | IO_HUP | IO_NVAL);
			} else
				stop();
		} else
			serve(conn, string(inet_ntoa(clientname.sin_addr)));
	} while (conn >= 0);
}

void Rainbow::HttpServer::serve (int fd, const string &hostname)
{
	const int concurrent_serves = 3;
	if (threads.size() >= concurrent_serves) {
		logmsg << "HttpServer: Rejecting connection from " << hostname << endl;
		close(fd);
		return;
	}

	// set socket to blocking
	int fd_flags = fcntl(fd, F_GETFL, 0);
	if (fd_flags == -1) {
		logmsg << "HttpServer: fcntl failed" << endl;
		close(fd);
		return;
	}
	if (fd_flags & O_NONBLOCK)
		fcntl(fd, F_SETFL, fd_flags & ~O_NONBLOCK);

	logmsg << "HttpServer: Creating thread to service request from " << hostname << endl;
	ServerThread *th = new ServerThread(fd, this);
	threads.insert(th);
	Thread::create(sigc::bind(sigc::mem_fun(*this, &HttpServer::serve_thread), th), false);
}

bool Rainbow::HttpServer::on_sock_event (IOCondition cond)
{
	if (cond == IO_IN || cond == IO_PRI) {
		try_accept();
	} else {
		stop();
	}
	
	return false;
}

void Rainbow::HttpServer::serve_thread (ServerThread *th)
{
	serve_thread_do(th);
	close(th->fd);
	th->dispatch_request_done();
}

void Rainbow::HttpServer::serve_thread_do (ServerThread *th)
{
	const int buf_size = 16384;
	char buf[buf_size];

	const int request_size_limit = 30000;
	string unparsed_request;

	string request_line;
	map<string,string> header;
	int status = 200;

	// wait for and parse request
	for (;;) {
		if (unparsed_request.size() > request_size_limit)
			return;

		ssize_t r = read(th->fd, buf, buf_size);

		if (r <= 0)
			return;

		// parse request headers
		unparsed_request.append(buf, r);
	header_loop:
		string::size_type p = unparsed_request.find(CRLF);
		if (p == string::npos)
			continue; // no more headers left in this read
		string line = unparsed_request.substr(0, p);
		unparsed_request = unparsed_request.substr(p + 2);
		if (line.size() == 0)
			break; // last header
		if (request_line.size() == 0) {
			request_line = line;
			goto header_loop;
		}
		string::size_type colon = line.find(": ");
		if (colon == string::npos) {
			status = 400;
			break;
		}
		transform(line.begin(), (line.begin() + colon),
			  line.begin(), ::tolower);
		header.insert(make_pair(line.substr(0, colon), line.substr(colon + 2)));
		goto header_loop;
	}

	FILE *file = 0;
	size_t content_length = 0;
	ostringstream content_range;
	string snd;
	string url;
	bool headers_only = false;
	if (status == 200) {
		// figure out url
		string::size_type space1 = request_line.find(' ');
		string::size_type space2 = request_line.rfind(' ');
		if (space1 != string::npos && space2 != string::npos)
			url = request_line.substr(space1 + 1, space2 - space1 - 1);
		
		// determine method
		if (request_line.substr(0, space1) == "HEAD")
			headers_only = true;

		Glib::ustring filename;
		Glib::ustring url_utf8;
		try {
			url_utf8 = Glib::ustring(url);
			if (url_utf8.validate())
				url_utf8 = "http:/" + url_utf8;
			else {
				url = "";
				url_utf8 = "";
			}

			th->mutex.lock();
			if (url.size()
			    && th->this_exists
			    && hc->get_filename_threadsafe(url_utf8, filename, true))
				file = fopen(filename_from_utf8(filename).c_str(), "rb");
			th->mutex.unlock();

			if (file) {
				content_length = hc->get_size_threadsafe(url_utf8);

				map<string,string>::iterator r = header.find("range");
				if (r != header.end()) {
					status = 206;
					string::size_type r_b = r->second.find("bytes=");
					if (r_b == string::npos) {
						status = 400;
					} else if (r->second.find(',') == string::npos) {
						string range = r->second.substr(6);
						string::size_type dash = range.find('-');
						if (dash != string::npos) {
							string begin_str = range.substr(0, dash);
							string end_str = range.substr(dash + 1);
							size_t begin = atoi(begin_str.c_str());
							size_t end = atoi(end_str.c_str());
							if (begin_str.size() == 0) {
								begin = content_length - end;
								end = content_length - 1;
							}
							if (end_str.size() == 0)
								end = content_length - 1;

							if (begin > end) {
								status = 400;
							} else if (begin < content_length
								   && end < content_length) {
								content_length = end + 1 - begin;
								fseek(file, begin, SEEK_SET);
								content_range << "bytes " << begin << '-' << end;
							} else
								status = 500; // byte range does not exist
							
						} else
							status = 400;
					} else
						status = 501; // comma not supported by this server
				}
			} else
				status = 404;
		} catch (...) {
			status = 400;
		}
	}

	if (status == 200) {
		ostringstream cl;
		cl << content_length;
		snd = "HTTP/1.1 200 OK" CRLF "Server: " SERVER CRLF "Content-length: " + cl.str() + CRLF ACCEPT_RANGES CRLF;
	} else if (status == 206) {
		snd = "HTTP/1.1 206 Partial Content" CRLF "Server: " SERVER CRLF "Content-Range: " + content_range.str() + CRLF ACCEPT_RANGES CRLF;
	} else if (status == 400)
		snd = "HTTP/1.1 400 Bad request" CRLF "Server: " SERVER CRLF CRLF "400 Bad request";
	else if (status == 404)
		snd = "HTTP/1.1 404 Not found" CRLF "Server: " SERVER CRLF CRLF "404 Not found";
	else if (status == 500)
		snd = "HTTP/1.1 500 Internal server error" CRLF "Server: " SERVER CRLF CRLF "500 Internal server error";
	else if (status == 501)
		snd = "HTTP/1.1 501 Not implemented" CRLF "Server: " SERVER CRLF CRLF "501 Not implemented";

	logmsg << "HttpServer: Request: file \"" << url << "\", status " << status << endl;

	// send header
	send_data_on_socket(th->fd, snd);

	// send file
	if ((status == 200 || status == 206)
		&& !headers_only) {
		for (;;) {
			size_t r = fread(buf, 1, (buf_size < content_length) ? buf_size : content_length, file);
			if (r == 0)
				break;
			if (r <= 0)
				break; // error, so no keepalive
			if (!send_data_on_socket(th->fd, buf, r))
				break;
			content_length -= r;
			if (content_length == 0)
				break;
		}
	}

	if (file)
		fclose(file);
}

void Rainbow::HttpServer::ServerThread::on_dispatch_done ()
{
	logmsg << "HttpServer: Request thread done" << endl;

	mutex.lock();
	bool t = this_exists;
	mutex.unlock();

	if (t)
		server->threads.erase(this);
	delete this;
}
