/*
 *  Hearts - player.c
 *  Copyright 2006 Sander Marechal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "player.h"
#include "hearts.h"
	
/* create a new player */
Player* player_new(gint direction, gint orientation, GString *script, GtkWidget *widget)
{
	/* create the player with default values */
	Player *player;
	player = g_new(Player, 1);
	player->score_total = 0;
	player->score_round = 0;
	player->point_cards = 0;
	player->lua_state = NULL;
	
	/* assign a hand of cards */
	player->hand = cards_hand_new(direction, orientation);
	
	/* create a pango layout for the name */
	player->layout = gtk_widget_create_pango_layout(widget, NULL);

	/* load the AI */
	if (script != NULL)
	{
		/* create a new Lua state */
		player->lua_state = lua_state_init();
		
		/* load the player script */		
		gchar* path = g_strconcat(PACKAGE_DATA_DIR"/gnome-hearts/scripts/players/", script->str, ".lua", NULL);
		if (lua_dofile(player->lua_state, path) != 0)
		{
			printf(_("Error: %s"), lua_tostring(player->lua_state, -1));
			g_assert_not_reached();
		}
		g_free(path);
		
		/* set the player ID (offset by one because lua tables start at 1) */
		lua_pushnumber(player->lua_state, orientation + 1);
		lua_setglobal(player->lua_state, "me");

		/* set the name */
		char *name = g_strconcat("<span foreground=\"white\"><b>", script->str, "</b></span>", NULL);
		pango_layout_set_markup(player->layout, name, -1);
		g_free(name);
	}
	
	return player;
}

/* Let a player's AI play a card on the trick */
void player_play(Player *player, Trick *trick)
{
	/* if there's no AI loaded, quit */
	g_assert(player != NULL && player->lua_state != NULL);
	
	/* set the global variables */
	game_score_publish(player->lua_state);
	cards_hand_publish(player->hand, player->lua_state);
	trick_publish(trick, player->lua_state);
	
	/* call the player's play_card() function */
	lua_getglobal(player->lua_state, "play_card");		/* "play_card" */
	if (lua_pcall(player->lua_state, 0, 1, 0) != 0)		/* table */
	{
		printf(_("Error: %s"), lua_tostring(player->lua_state, -1));
		g_assert_not_reached();
	}
	
	Card *card = lua_pop_card(player->lua_state);
	g_assert(g_list_find(player->hand->list, card) != NULL);
	
	if (!game_is_valid_card(card, player->hand, trick))
	{
		printf("trying to play invalid card %d,%d\n", card->suit, card->rank);
		g_assert_not_reached();
	}
	
	/* play the actual card */
	player_play_card(player, card, trick);
}

void player_select_cards(Player *player)
{
	gint i;
	
	/* if there's no AI loaded, quit */
	g_assert(player->lua_state != NULL);
	
	/* set the global variables */
	game_score_publish(player->lua_state);
	cards_hand_publish(player->hand, player->lua_state);
	
	/* call the player's select_cards() function */
	lua_getglobal(player->lua_state, "select_cards");	/* select_cards */
	if (lua_pcall(player->lua_state, 0, 1, 0) != 0)		/* table */
	{
		printf(_("Error: %s"), lua_tostring(player->lua_state, -1));
		g_assert_not_reached();
	}
	
	/* start reading from the table */					
	lua_pushnil(player->lua_state);						/* table, nil */

	/* read three cards and mark them as selected */
	for (i = 0; i < 3; i++)
	{
		/* read next card */
		lua_next(player->lua_state, -2);			/* table, key, table */

		Card *card = lua_pop_card(player->lua_state);
		g_assert(g_list_find(player->hand->list, card) != NULL);
		card->selected = TRUE;
	}
	
	/* clean up and return */
	lua_pop(player->lua_state, 2);				/* <empty> */
}

/* show the trick to the AI */
void player_trick_end(Player *player, Trick *trick)
{
	if (player->lua_state == NULL)
		return;
	
	trick_publish(trick, player->lua_state);

	lua_getglobal(player->lua_state, "trick_end");
	
	if (!lua_isfunction(player->lua_state, -1))
		return;
	
	if (lua_pcall(player->lua_state, 0, 1, 0) != 0)
	{
		printf(_("Error: %s"), lua_tostring(player->lua_state, -1));
		g_assert_not_reached();
	}
}

/* Call an AI's round_end function */
void player_round_end(Player *player)
{
	if (player->lua_state == NULL)
		return;
	
	game_score_publish(player->lua_state);

	lua_getglobal(player->lua_state, "round_end");
	
	if (!lua_isfunction(player->lua_state, -1))
		return;
	
	if (lua_pcall(player->lua_state, 0, 1, 0) != 0)
	{
		printf(_("Error: %s"), lua_tostring(player->lua_state, -1));
		g_assert_not_reached();
	}
}

/* play player's card on trick */
void player_play_card(Player *player, Card *card, Trick *trick)
{
	GList *list_item;
	
	if (card == NULL)
		return;
	
	trick->card[player->hand->orientation] = card;
	trick->num_played++;
	
	/* if this is the first card played, save the suit and the one who played it */
	if (trick->first_played == -1)
		trick->first_played = player->hand->orientation;
	if (trick->trump == -1)
		trick->trump = card->suit;
	
	g_assert(trick->first_played >= 0 && trick->first_played < 4);

	/* move the card from the hand to the history stack */
	list_item = g_list_find(player->hand->list, card);
	player->hand->list = g_list_delete_link(player->hand->list, list_item);
}

/* free a player */
void player_free(Player *player)
{
	if (player == NULL)
		return;
	
	if (player->hand != NULL)
		cards_hand_free(player->hand);
	
	if (player->lua_state != NULL)
		lua_close(player->lua_state);

	if (player->layout != NULL)
		g_object_unref(player->layout);
	
	g_free(player);
}
