/*
 *  Hearts - player-api.c
 *  Copyright 2006 Sander Marechal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "player-api.h"
#include "hearts.h"

/* FIXME: this is a temporary function. Remove it a.s.a.p. */
void stack_dump (lua_State *L) 
{
	int i;
    int top = lua_gettop(L);
    for (i = 1; i <= top; i++) {  /* repeat for each level */
        int t = lua_type(L, i);
        switch (t) {
    
          case LUA_TSTRING:  /* strings */
            printf("`%s'", lua_tostring(L, i));
            break;
    
          case LUA_TBOOLEAN:  /* booleans */
            printf(lua_toboolean(L, i) ? "true" : "false");
            break;
    
          case LUA_TNUMBER:  /* numbers */
            printf("%g", lua_tonumber(L, i));
            break;
    
          default:  /* other values */
            printf("%s", lua_typename(L, t));
            break;
    
        }
        printf("  ");  /* put a separator */
    }
    printf("\n");  /* end the listing */
}

/* Lua function that returns the trick winner */
gint lua_trick_get_winner(lua_State *state)
{
	gint winner;
	
	/* return nil if the trick hasn't finished */
	if (game_trick.num_played != 4)
	{
		lua_pushnil(state);
		return 1;
	}
	
	winner = trick_get_winner(&game_trick);
	lua_pushnumber(state, winner);
	return 1;
}

/* Lua function that returns the trick starter (offset by one because lua tables start at index 1) */
gint lua_trick_get_trump(lua_State *state)
{
	gint trump = game_trick.trump;
	
	if (trump == -1)
		lua_pushnil(state);
	else
		lua_pushnumber(state, trump);
	
	return 1;
}

gint lua_trick_get_num_played(lua_State *state)
{
	lua_pushnumber(state, game_trick.num_played);
	
	return 1;
}

/* Lua function that returns the value of the trick */
gint lua_trick_get_score(lua_State *state)
{
	gint score;
	
	score = trick_get_score(&game_trick, game_rules.ruleset);
	lua_pushnumber(state, score);
	return 1;
}

/* lua function that returns if a card is valid or not */
gint lua_is_valid_card(lua_State *state)
{
	gint suit, rank, p;
	Card *card;
	
	/* start reading from the table */			
	lua_pushnil(state);								/* table, nil */

	/* read the first value off the stack */
	lua_next(state, -2);							/* table, key, val */
	suit = lua_tonumber(state, -1);					/* table, key, val */
	lua_pop(state, 1);								/* table, key */

	/* read the second value off the stack */
	lua_next(state, -2);							/* table, key, val */
	rank = lua_tonumber(state, -1);					/* table, key, val */
	lua_pop(state, 3);								/* <empty> */

	/* get the player ID (offset of one) */
	lua_getglobal(state, "me");							/* me */
	p = lua_tonumber(state, -1) - 1;
	lua_pop(state, 1);									/* <empty> */
	
	/* find the card in the deck */
	card = &g_array_index(game_deck, Card, CARD_ID(suit, rank));
	lua_pushboolean(state, game_is_valid_card(card, player[p]->hand, &game_trick));
	return 1;
}

/* Initialize a lua_State */
lua_State* lua_state_init()
{
	/* create a new Lua state */
	lua_State *state = lua_open();
		
	/* Load the base libraries */
	luaopen_base(state);
	luaopen_string(state);
	luaopen_table(state);
	luaopen_math(state);
	luaopen_io(state);
		
	/* set the package data dir */
	lua_pushstring(state, PACKAGE_DATA_DIR);
	lua_setglobal(state, "datadir");

	/* bind some C functions to Lua */
	lua_register(state, "trick_get_winner", lua_trick_get_winner);
	lua_register(state, "trick_get_trump", lua_trick_get_trump);
	lua_register(state, "trick_get_num_played", lua_trick_get_num_played);
	lua_register(state, "trick_get_score",  lua_trick_get_score);
	lua_register(state, "is_valid_card",    lua_is_valid_card);

	/* load the hearts script */
	if (lua_dofile(state, PACKAGE_DATA_DIR"/gnome-hearts/scripts/hearts.lua") != 0)
	{
		printf(_("Error: %s"), lua_tostring(state, -1));
		g_assert_not_reached();
	}
	
	return state;
}

/* pop a card from the lua state */
Card* lua_pop_card(lua_State *state)
{
	gint suit, rank, i;
	
	g_assert(lua_istable(state, -1));
	
	/* start reading from the table */			
	lua_pushnil(state);						/* table, nil */

	/* read the first value off the stack */
	lua_next(state, -2);					/* table, key, val */
	suit = lua_tonumber(state, -1);			/* table, key, val */
	lua_pop(state, 1);						/* table, key */

	/* read the second value off the stack */
	lua_next(state, -2);					/* table, key, val */
	rank = lua_tonumber(state, -1);			/* table, key, val */
	lua_pop(state, 3);						/* <empty> */
	
	/* now find the card */
	for (i = 0; i < game_deck->len; i++)
	{
		Card *card = &g_array_index(game_deck, Card, i);
		
		if (card->suit == suit && card->rank == rank)
			return card;
	}
	
	g_assert_not_reached();
}
