#include <stdio.h>
#include <stdlib.h>

#include <string.h>

#include <ctype.h>

#include "gnapster.h"
#include "upload.h"
#include "queue.h"
#include "servers.h"
#include "resume.h"
#include "chan.h"
#include "connect.h"

extern UserInfo user_info;

void socks4_connect(void *data, int source, GdkInputCondition cond) {
   int i, err;
   size_t len;
   unsigned char buf[256];
   SocksData *s;
   
   if (!data)
     return;
   
   s = data;
   
   INPUT_REMOVE(s->input);
   
   switch(s->instance++) {
    case 0:
      len = sizeof(err);
      getsockopt(source, SOL_SOCKET, SO_ERROR, &err, &len);
      if (err) {
	 j_error("socks4_connect", "connect", NULL);
	 break;
      }
      
      i = 0;
      
      buf[i++] = 0x04; /* protocol version: SOCKS4 */
      buf[i++] = 0x01; /* command: CONNECT */
      
      *((guint16 *)(buf + i)) = s->dest_port;
      i += 2;
      
      *((guint32 *)(buf + i)) = s->dest_ip;
      i += 4;
      
      /* add ident userid stuff here */
      buf[i++] = 0x00; /* NULL terminated */
      
      set_blocking(source, 1);
      send(source, buf, i, 0);
      
      /* re-attach */
      s->input = INPUT_ADD(source, GDK_INPUT_READ,
			       socks4_connect, data);
      
      return;
    case 1:
      i = recv(source, buf, sizeof(buf) - 1, 0);
      if (i <= 0) {
	 j_error("socks4_connect", "recv", NULL);
	 break;
      }
      
      if (buf[0] != 0x00)
	break;
      
      switch(buf[1]) {
       case 90: /* request granted */
	 *(s->orig_input) = INPUT_ADD(source, s->condition,
					  s->orig_cb, s->data);
	 break;
       case 91: /* request rejected/failed */
       case 92:
       case 93:
	 fprintf(stderr, "socks4: request rejected/failed\n");
	 break;
       default:
	 break;
      }
      
      break;
    default:
      break;
   }
   
   d_free(s);
}

void socks5_connect(void *data, int source, GdkInputCondition cond) {
   int i, err, len_un, len_pw;
   size_t len;
   unsigned char buf[256];
   SocksData *s;
   
   if (!data)
     return;
   
   s = data;
   
   if (s->instance < 0)
     s->instance = -(s->instance);
   else
     INPUT_REMOVE(s->input);
   
   switch(s->instance++) {
    case 0:
      len = sizeof(err);
      getsockopt(source, SOL_SOCKET, SO_ERROR, &err, &len);
      if (err) {
	 j_error("socks5_connect", "connect", NULL);
	 break;
      }
      
      i = 0;
      buf[i++] = 0x05; /* protocol version: SOCKS5 */
      
      if (!(*s->username) || !(*s->password)) {
	 buf[i++] = 0x01; /* 1 auth method */
	 buf[i++] = 0x00; /* no auth required */
      } else {
	 buf[i++] = 0x02; /* 2 auth methods */
	 buf[i++] = 0x00; /* no auth required */
	 buf[i++] = 0x02; /* username/password auth */
      }
      
      set_blocking(source, 1);
      send(source, buf, i, 0);
      
      s->input = INPUT_ADD(source, GDK_INPUT_READ, 
			       socks5_connect, s);
      
      return;
    case 1:
      i = recv(source, buf, sizeof(buf) - 1, 0);
      if (i <= 0)
	break;
      
      if (buf[0] != 0x05)
	break;
      
      switch(buf[1]) {
       case 0x00: /* no auth required */
	 s->instance = -3;
	 socks5_connect(s, source, cond);
	 
	 return;
       case 0x02: /* username/password required */
	 len_un = strlen(s->username);
	 len_pw = strlen(s->password);
	 
	 if ((len_un + len_pw) > 253)
	   break;
	 
	 i = 0;
	 
	 buf[i++] = 0x01;
	 
	 buf[i++] = (unsigned char)len_un;
	 memcpy(buf + i, s->username, len_un);
	 i += len_un;
	 
	 buf[i++] = (unsigned char)len_pw;
	 memcpy(buf + i, s->password, len_pw);
	 i += len_pw;
	 
	 send(source, buf, i, 0);
	 
	 s->input = INPUT_ADD(source, GDK_INPUT_READ, 
				  socks5_connect, s);
	 
	 return;
       case 0xff: /* none of the methods are acceptable */
	 fprintf(stderr, "socks5: no auth method was acceptable!\n");
	 break;
       default:
	 fprintf(stderr, "socks5: auth method not supported yet\n");
	 break;
      }
    case 2:
      i = recv(source, buf, sizeof(buf) - 1, 0);
      if (i <= 0)
	break;
      
      if (buf[0] != 0x01)
	break;
      
      if (buf[1] != 0x00) {
	 fprintf(stderr, "socks5: socks server failed username/password auth\n");
	 break;
      }
      
      /* do not break, auth succeeded */
    case 3: /* build command send it */
      i = 0;
      
      buf[i++] = 0x05; /* protocol version: SOCKS5 */
      buf[i++] = 0x01; /* command: CONNECT */
      buf[i++] = 0x00; /* reserved */
      buf[i++] = 0x01; /* address type: IP V4 */
      
      *((guint32 *)(buf + i)) = s->dest_ip;
      i += 4;
      
      *((guint16 *)(buf + i)) = s->dest_port;
      i += 2;
      
      send(source, buf, i, 0);
      
      s->input = INPUT_ADD(source, GDK_INPUT_READ,
			       socks5_connect, s);
      
      return;
    case 4: /* collect, and eval connection reply */
      i = recv(source, buf, sizeof(buf) - 1, 0);
      if (i <= 0)
	break;
      
      if (buf[0] != 0x05)
	break;
      
      switch(buf[1]) {
       case 0x00: /* request granted */
	 *(s->orig_input) = INPUT_ADD(source, s->condition,
					  s->orig_cb, s->data);
	 
	 break;
       case 0x01: /* request rejected/failed */
       case 0x02:
       case 0x03:
       case 0x04:
       case 0x05:
       case 0x06:
       case 0x07:
       case 0x08:
	 fprintf(stderr, "socks5: request rejected/failed\n");
	 break;
       default:
	 break;
      }
    default:
      break;
   }
   
   d_free(s);
}

int dispatch_connect(unsigned long int ip, unsigned short int port, int *sock, int *input, GdkInputCondition condition, void *cb, void *data) {
   int s, in;
   struct sockaddr_in server;
   SocksData *conn = NULL;
   
   s = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
   if (s < 0)
     return 0;

   set_blocking(s, 0);
   
   memset(&server, 0, sizeof(struct sockaddr_in));
   server.sin_family = AF_INET;
   server.sin_addr.s_addr = ip;
   server.sin_port = port;
   
   /* ok this is where we will handle our SOCKS4/5 stuff as well as
    * regular connects */
   if (user_info.conf[FIREWALL] && user_info.conf[SOCKS_FIREWALL]) {
      char *s_ip, *un, *pw;
      int s_port;
      
      s_ip = d_config_get_string("/gnapster/Connection/socks_server");
      s_port = j_config_get_int("/gnapster/Connection/socks_port");
      
      if (!s_ip) {
	 j_error_dialog("No SOCKS server found!");
	 return 0;
      }
      
      un = pw = NULL;
      
      if (user_info.conf[SOCKS5]) {
	 un = d_config_get_string("/gnapster/Connection/socks5_username");
	 pw = d_config_get_string("/gnapster/Connection/socks5_password");
      }
      
      conn = d_malloc(sizeof(SocksData));
      memset(conn, 0, sizeof(SocksData));
      
      conn->orig_cb = cb;
      conn->cb = user_info.conf[SOCKS4] ? 
	socks4_connect : socks5_connect;
      conn->data = data;
      conn->condition = condition;
      conn->sock = sock;
      conn->dest_ip = ip;
      conn->dest_port = port;
      conn->username = un;
      conn->password = pw;
      
      server.sin_addr.s_addr = inet_addr(s_ip);
      server.sin_port = htons(s_port);
   }

   connect(s, (struct sockaddr *)&server, sizeof(struct sockaddr_in));
   
   if (sock)
     *sock = s;

   if (!conn) {
      in = INPUT_ADD(s, condition, cb, data);

      if (input)
	*input = in;
   } else {
      conn->input = INPUT_ADD(s, condition, conn->cb, conn);
      
      if (input)
	conn->orig_input = input;
   }
   
   return 1;
}
