/* callbacks.c - controls GTK callback functions (excluding network) */

#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

#include <ctype.h>

#include "xtext.h"
#include "gnapster.h"
#include "browse.h"
#include "commands.h"
#include "queue.h"
#include "download.h"
#include "servers.h"
#include "chan.h"
#include "themes.h"

#include "debug.h"

#include <gdk/gdkx.h>

#ifndef HAVE_GNOME
#include <gdk/gdkkeysyms.h>
#endif /* !HAVE_GNOME */

extern UserInfo user_info;

extern GnapsterMain *gmain;

extern PropDlg prop_dlg;
extern BuildDlg build_dlg;
extern BrowseDlg browse_dlg;
extern AddDlg add_dlg;
extern EditDlg edit_dlg;
extern ListChanDlg list_chan_dlg;
extern ResumeDlg resume_dlg;
extern RefreshDlg refresh_dlg;

extern int building;
extern GList *ignore_list;
extern GList *hooks;

int nick_curr = 0, nick_end = 0;
GList *nick_history = NULL;

int loc_curr = 0, loc_end = 0;
GList *console_history = NULL;

#ifndef HAVE_GNOME
/* i really dont like how this works, oh well */
void *realfunc[] = {
   search_download_cb, browse_user_menu_cb, whois_user_menu_cb,
   browse_download_cb, retry_download_cb, cancel_download_cb,
   cancel_remove_cb, cancel_upload_cb, force_queue_cb, cancel_queue_cb,
   cancel_movequeue_cb
};

void thunk(gpointer data, guint function, GtkWidget *item) {
   void *data_pass;
   
   data_pass = gtk_item_factory_popup_data_from_widget(item);
   GTK_SIGNAL_FUNC(realfunc[function])(item, data_pass);
}
#endif /* !HAVE_GNOME */

CBPack *cb_pack(char *file, int line, char *func_name, void *input_func, void *timeout_func, void *data) {
   CBPack *cb;
   
   cb = d_new(CBPACK);
   
   cb->file = d_strdup(file);
   cb->line = line;
   
   cb->func_name = d_strdup(func_name);
   cb->input_func = input_func;
   cb->timeout_func = timeout_func;
   
   cb->data = data;
   
   return cb;
}

void cb_input_debug(void *data, int source, GdkInputCondition cond) {
   DCBPACK();
   void *d;
   
   d_assert(data != NULL);
   
   CBPACK(d);
   
   fprintf(stderr, "cb_input_debug (%i, %i): %s:%i: %s()\n", source, cond, file, line, func_name);
   
   /* call the real function now */
   (input_func) (d, source, cond);
}

int cb_timeout_debug(void *data) {
   DCBPACK();
   int ret;
   void *d;
   
   d_assert_return(data != NULL, 0);
   
   CBPACK(d);
   
   fprintf(stderr, "cb_timeout_debug: %s:%i: %s()...", file, line, func_name);
   fflush(stderr);
   
   ret = (timeout_func) (d);
   
   fprintf(stderr, "%i\n", ret);
   
   return ret;
}

void disable_ul_cb(GtkWidget *w, void *data) {
   int active;
   
   active = GTK_TOGGLE_BUTTON(w)->active;
   
   /* by the nature of clicked_cb, this func will be called for all stabs
    * automatically, we do not need to loop for it. */

   FORSTABS(if (active)
	      unshare_files(stab);
	    else
	      shared_list_handle(stab, user_info.upload_dir, SUBMIT_LIST));
}

void total_xfer_stats() {
   create_xfer_stats(NULL);
}

void xfer_stats() {
   STab *stab;
   
   stab = get_current_stab();
   d_assert(stab != NULL);
   
   create_xfer_stats(stab);
}

void switch_stab_gnome(GtkWidget *w, void *data) {
   STab *stab;
   
   d_assert(data != NULL);
   
   stab = data;
   
   gtk_notebook_set_page(GTK_NOTEBOOK(gmain->srv_notebook), stab->pn + 4);
}

void switch_stab_gtk(void *data, int callback_action, GtkWidget *w) {
   STab *stab;

   d_assert(data != NULL);
   
   stab = data;
   
   gtk_notebook_set_page(GTK_NOTEBOOK(gmain->srv_notebook), stab->pn + 4);
}

void show_stabs(GtkWidget *w) {
   int st;
   
   st = GTK_CHECK_MENU_ITEM(w)->active;
   
   gtk_notebook_set_show_tabs(GTK_NOTEBOOK(gmain->srv_notebook), st);
   
   j_config_set_int("/gnapster/Options/show_stabs", st);
   j_config_sync();
   
   user_info.conf[SHOW_STABS] = st;
}

void show_stabs_gnome(GtkWidget *w, void *data) {
   show_stabs(w);
}

void show_stabs_gtk(void *data, int cb_action, GtkWidget *w) {
   show_stabs(w);
}

FILE *open_stabs(char *flags) {
   FILE *f;
   char *conf;
   
   conf = local_path("stabs", NULL);
   
   f = fopen(conf, flags);
   
   if (!f)
     j_error("fopen", conf, NULL);
   
   d_free(conf);
   
   return f;
}

void account_show(GtkWidget *w, int row, int col, GdkEvent *ev, void *data) {
   GtkWidget *window, *user, *pass, *primary;
   Account *acnt;
   
   d_assert(data != NULL);
   
   window = data;
   
   GET_WIDGET(window, user);
   GET_WIDGET(window, pass);
   GET_WIDGET(window, primary);

   acnt = gnapster_get_selection_data(w, NULL);
   if (!acnt)
     return;
   
   gtk_entry_set_text(GTK_ENTRY(user), acnt->user);
   gtk_entry_set_text(GTK_ENTRY(pass), acnt->pass);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(primary), acnt->primary);
}

int account_rem(GtkWidget *w, void *data) {
   GtkWidget *window, *clist, *user, *pass, *combo;
   FILE *f;
   Account *acnt;
   char *dptr, *userstr, *passstr;
   GList *felem, *ptr;
   FileEntry *fent;
   int row, primary, ret;
   
   d_assert_return(data != NULL, 0);
   
   window = data;
   
   GET_WIDGET(window, user);
   GET_WIDGET(window, pass);
   GET_WIDGET(window, clist);
   GET_WIDGET(window, combo);
   
   acnt = gnapster_get_selection_data(clist, &row);
   if (!acnt)
     return 0;

   /* w == NULL if called internally */
   if (w && acnt->primary) {
      j_error_dialog(_("Cannot remove your primary acount!"));
      return 0;
   }
   
   f = open_local_path("accounts", NULL, "r");
   if (!f)
     return 0;

   felem = read_file(f);
   
   fclose(f);
   
   f = open_local_path("accounts", NULL, "w");

   ret = 0;
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      userstr = next_arg_full(dptr, &dptr, ':');
      passstr = next_arg(dptr, &dptr);
      convert(next_arg(dptr, &dptr), "%i", &primary); 
      
      NA_ERR_HANDLE_BREAK();
      
      if (strcmp(acnt->user, userstr))
	fprintf(f, "%s:%s %i\n", userstr, passstr, primary);
      else
	ret = 1;
   }
   
   fclose(f);
   
   if (combo) {
      gtk_list_clear_items(GTK_LIST(GTK_COMBO(combo)->list), 0, -1);
      fill_accounts(combo, 1);
   }
   
   gtk_clist_remove(GTK_CLIST(clist), row);
  
   if (!w)
     return ret;
   
   gtk_entry_set_text(GTK_ENTRY(user), "");
   gtk_entry_set_text(GTK_ENTRY(pass), "");
   
   return ret;
}

void account_change(GtkWidget *w, void *data) {
   if (account_rem(NULL, data))
     account_add(NULL, data);
}

void account_add(GtkWidget *w, void *data) {
   GtkWidget *window, *user, *pass, *primary, *clist, *combo;
   FILE *f;
   char *u, *p;
   int pr;
   
   d_assert(data != NULL);
   
   window = data;
   
   GET_WIDGET(window, user);
   GET_WIDGET(window, pass);
   GET_WIDGET(window, clist);
   GET_WIDGET(window, primary);
   GET_WIDGET(window, combo);

   u = gtk_entry_get_text(GTK_ENTRY(user));
   p = gtk_entry_get_text(GTK_ENTRY(pass));
   pr = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(primary));
   
   if (!u || !p)
     return;
   if (!*u || !*p)
     return;
   
   f = open_local_path("accounts", NULL, "a");
   
   fprintf(f, "%s:%s %i\n", u, p, pr);
   
   fclose (f);
   
   if (combo) {
      gtk_list_clear_items(GTK_LIST(GTK_COMBO(combo)->list), 0, -1);
      fill_accounts(combo, 1);
   }
   
   gtk_clist_clear(GTK_CLIST(clist));
   
   fill_accounts(clist, 0);
   
   gtk_entry_set_text(GTK_ENTRY(user), "");
   gtk_entry_set_text(GTK_ENTRY(pass), "");
   gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(primary), 0);
}

void stab_snapshot() {
   FILE *f;
   
   f = open_stabs("w");
   
   FORSTABS(if (ci && ci->ip && ci->port && ci->account) fprintf(f, "%s:%hu \"%s\" %s\n", ci->ip, ci->port, ci->server, ci->account->user));
   
   fclose(f);
   
   j_warning_dialog(_("Server snapshot has been saved"));
}

void browse_clear(GtkWidget *w, void *data) {
   STab *stab;
   
   stab = get_current_stab();
   
   gtk_clist_clear(GTK_CLIST(gmain->bt->ctree));
   
   if (browsing(stab->ci))
     end_browse(stab);
}

/* hooray for xchat */
int property_notify(GtkWidget *w, GdkEventProperty *event, void *data) {
   static Atom prop = None;
   
   if (prop == None)
     prop = XInternAtom(GDK_DISPLAY(), "_XROOTPMAP_ID", True);
   
   if (event->atom == prop)
     update_xtext(NULL);
   
   return FALSE;
}

void gnapster_exit_cb() {
   int w = -1, h = -1;

   FORSTABS(if (!connected(ci)) continue;
	    CLOSE(ci->sock);
	    INPUT_REMOVE(ci->sock_input));
   
   CLOSE(gmain->bind_sock);
   INPUT_REMOVE(gmain->bind_input);

   gdk_window_get_size(gmain->window->window, &w, &h);
   j_config_set_int("/gnapster/Geometry/w", w);
   j_config_set_int("/gnapster/Geometry/h", h);
   j_config_sync();
   
   gtk_main_quit();
}

void safe_exit_check() {
   if (!gmain->safe_exit)
     return;
   
   if (CLIST_LENGTH(gmain->ut->clist) ||
       CLIST_LENGTH(gmain->dt->clist) ||
       CLIST_LENGTH(gmain->dt->queue_clist))
     return;
   
   gnapster_exit_cb();
}

void safe_exit_cb() {
   FORSTABS(unshare_files(stab);
	    hook_text_insert(stab, CURR, SYSTEM, "general_message", "%s",
			     "Waiting for all transfers to complete..."));
   
   gmain->safe_exit = 1;
   
   safe_exit_check();
}

void set_trans(GtkWidget *text) {
   GtkXText *xtext;
   
   xtext = GTK_XTEXT(text);
   
   gtk_xtext_set_background(xtext, NULL, user_info.conf[TINT],
			    user_info.conf[TINT]);
   
   xtext->tint_red = user_info.conf[TINT_RED];
   xtext->tint_green = user_info.conf[TINT_GREEN];
   xtext->tint_blue = user_info.conf[TINT_BLUE];

   if (xtext->transparent) {
      gtk_xtext_free_trans(xtext);
      gtk_xtext_load_trans(xtext);
   }
   
   gtk_xtext_render_page(xtext, xtext->adj->value);
}

void update_xtext(GtkWidget *w) {
   GList *cl;
   
   FORSTABS(set_trans(stab->ct->text);
	    for(cl=stab->ct->channel_list; cl; cl=cl->next)
  	      set_trans(((ChannelInfo *)cl->data)->text));
}

void browse_font_ok_cb(GtkWidget *w, GtkWidget *dlg) {
   char *font;
   
   font = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(dlg));
   
   if (prop_dlg.font_entry && font && strlen(font))
     gtk_entry_set_text(GTK_ENTRY(prop_dlg.font_entry), font);
   
   gtk_widget_destroy(dlg);
}

void browse_font_cancel_cb(GtkWidget *w, GtkWidget *dlg) {
   gtk_widget_destroy(dlg);
}

void browse_font_cb() {
   GtkWidget *dlg;
   
   dlg = gtk_font_selection_dialog_new(_("Select Font"));
   
   gtk_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(dlg)->ok_button), "clicked",
		      GTK_SIGNAL_FUNC(browse_font_ok_cb), dlg);
   gtk_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(dlg)->cancel_button), "clicked",
		      GTK_SIGNAL_FUNC(browse_font_cancel_cb), dlg);
   
   gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(dlg),
					   gtk_entry_get_text(GTK_ENTRY(prop_dlg.font_entry)));
   
   gtk_widget_show(dlg);
}

void update_shares_cb() {
   STab *stab;
   
   stab = get_current_stab();
   
   shared_list_handle(stab, user_info.upload_dir, BUILD_LIST | UPDATE_LIST);
}

void connect_last_cb() {
   STab *stab;
   
   stab = get_current_stab();
   
   if (connected(stab->ci) || connecting(stab->ci))
     already_connected(stab, NULL);
   else
     connect_cb(stab);
}

void already_connected_cb(gint reply, void *data) {
   int treply;
   Server *nsrv;
   STab *stab;
   
   if (!data)
     return;
   
   stab = get_current_stab();
   
#ifdef HAVE_GNOME
   treply = reply;
#else
   treply = data ? DIALOG_YES : DIALOG_NO;
#endif /* HAVE_GNOME */
   
   nsrv = (Server *)data;
   if (!nsrv)
     return;
   
   if (treply == DIALOG_YES) {
      disconnect(stab);
      if (strcmp(nsrv->network, "connect_cb_regular")) {
	 Server *nsrv;
	 
	 nsrv = (Server *)data;
	 browse_dlg_connect_cb(NULL, nsrv);
      } else {
	 connect_cb(stab);
	 j_free(SERVER, nsrv);
      }
   }
}

void already_connected(STab *stab, Server *nsrv) {
   GtkWidget *dialog;
   
   if (!nsrv) {
      Server *srv;
      
      srv = d_new(SERVER);
      srv->network = d_strdup("connect_cb_regular");
      
      nsrv = srv;
   }
   
   dialog = j_question_dialog(_("You're already connected to a server. Disconnect?"),
			      already_connected_cb,
			      nsrv,
			      GTK_WINDOW(gmain->window));
}

void connect_official_cb() {
   STab *stab;
   
   stab = get_current_stab();
   
   set_conn_info(stab, SERVER_NAPSTER_COM, 8875, "server.napster.com", NULL);
   if (!stab->ci->account) {
      j_error_dialog("Failure to open default primary account!");
      return;
   }
   
   if (connected(stab->ci) || connecting(stab->ci)) {
      already_connected(stab, NULL);
      
      return;
   }

   connect_cb(stab);
}

void switch_page_cb(GtkWidget *widget, GtkNotebookPage *page, int page_num, gpointer data) {
   STab *stab;
   GtkWidget *nth_page, *label;
   GtkStyle *old_style;
   GtkWidget *focus_widget;   
   
   nth_page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(widget), page_num);
   
   label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(widget), nth_page);
   
   old_style = gtk_widget_get_style(label);
   
   if (old_style == gmain->highlight)
     gtk_widget_set_style(label, gmain->normal);
   
   if (widget == gmain->srv_notebook) {
      stab = get_stab(page_num - 4);
      
      change_status(stab);
      change_title(stab);
   } else
     stab = get_current_stab();
   
   if (!stab)
     return;
   
   /* only modify focus if selecting a server tab */
   if (gtk_notebook_get_current_page(GTK_NOTEBOOK(gmain->srv_notebook)) < 4 &&
       widget != gmain->srv_notebook)
     return;
   
   GET_WIDGET(stab->vbox, focus_widget);
   if (focus_widget)
     gtk_widget_grab_focus(focus_widget);
}

void build_list_cb() {
   STab *stab;
   
   if (building) {
      building = 0;
      return;
   }
   
   stab = get_current_stab();

   shared_list_handle(stab, user_info.upload_dir,
		      BUILD_LIST);
}

void update_list_cb() {
   STab *stab;
   
   stab = get_current_stab();
   
   shared_list_handle(stab, user_info.upload_dir,
		      BUILD_LIST | UPDATE_LIST);
}

void query_user_cb(GtkWidget *w, gpointer data) {
   STab *stab;
   
   stab = get_current_stab();
   
   gtk_notebook_set_page(GTK_NOTEBOOK(gmain->srv_notebook), 4 + stab->pn);
   
   gtk_entry_set_text(GTK_ENTRY(stab->ct->entry), "/query ");
   
   gtk_widget_grab_focus(stab->ct->entry);
}

void unquery_user_cb(GtkWidget *w, gpointer data) {
   STab *stab;
   
   stab = get_current_stab();
   
   if (!remove_console_tab(stab, NULL, 1))
     j_error_dialog("You must click on a user query tab before attempting to close it");
}

/*void bp_event(GtkWidget *widget, GdkEventButton *ev, void *data) {
   GtkWidget *menu;
   int row = -1, col;
   
   gtk_clist_get_selection_info(GTK_CLIST(widget), ev->x, ev->y, &row, &col);
   
   if (ev->button == 3) {
#ifdef HAVE_GNOME
      menu = gnome_popup_menu_new(data);
      gnome_popup_menu_do_popup_modal(menu, NULL, NULL, ev, widget);
      gtk_widget_destroy(menu);
#else
      UNUSED(menu);
      gtk_item_factory_popup_with_data(GTK_ITEM_FACTORY(data), widget, NULL, 
				       ev->x_root, ev->y_root, ev->button, 
				       ev->time);
#endif
   }
}*/

void menu_popup_cb(GtkWidget *widget, GdkEventButton *event, gpointer data) {
   GtkWidget *temp_menu;
   int row = -1, col;

   gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y,
				&row, &col);

   if (event->type == GDK_2BUTTON_PRESS && event->button == 1) {
      Search *s = NULL;
      
      if (widget == gmain->bt->ctree) {
	 GtkCTreeNode *node = 
	   GTK_CTREE_NODE(g_list_nth(GTK_CLIST(widget)->row_list, row));
	 s = gtk_ctree_node_get_row_data(GTK_CTREE(widget), node);
      } else if (widget == gmain->st->clist)
	s = gtk_clist_get_row_data(GTK_CLIST(widget), row);
      
      if (!s)
	return;
	 
      download_true_cb(widget, s);
   } else if (event->button == 3) {
#ifdef HAVE_GNOME
      temp_menu = gnome_popup_menu_new(data);
      gnome_popup_menu_do_popup_modal(temp_menu, NULL, NULL, event, widget);
      gtk_widget_destroy(temp_menu);
#else
      UNUSED(temp_menu);
      gtk_item_factory_popup_with_data(GTK_ITEM_FACTORY(data), widget, NULL, 
				       (guint)event->x_root, (guint)event->y_root,
				       event->button, event->time);
#endif /* HAVE_GNOME */
   }
}

void console_activate_cb() {
   STab *stab;
   char *console_text = NULL, *command, *message;
   
   stab = get_current_stab();
   
   console_text = gtk_entry_get_text(GTK_ENTRY(stab->ct->entry));
   
   if (!strlen(console_text)) 
     return;
   
   if (loc_end > MAX_HISTORY) {
      GList *link;
      
      link = g_list_nth(console_history, 0);
      console_history = g_list_remove_link(console_history, link);
      d_free(link->data);
      loc_end--;
   }
   console_history = g_list_append(console_history, d_strdup(console_text));
   loc_end++;
   loc_curr = loc_end;

   if (*console_text != '/') {
      handle_command(stab, "/say", d_strdup(console_text));
      gtk_entry_set_text(GTK_ENTRY(stab->ct->entry), "");
      return;
   }
   
   command = next_arg(console_text, &console_text);
   if (!command) {
      gtk_entry_set_text(GTK_ENTRY(stab->ct->entry), "");
      return;
   }
   message = last_arg(console_text, &console_text);
   
   NA_RESET();

   if (!handle_command(stab, command, message))
     hook_text_insert(stab, CURR, SYSTEM, "unknown_cmd", "%s",
		      command);
   
   gtk_entry_set_text(GTK_ENTRY(stab->ct->entry), "");
}

GList *add_nick_history(GList *nh, char *user) {
   if (nick_end > MAX_HISTORY) {
      GList *link;

      link = g_list_nth(nh, 0);
      nh = g_list_remove_link(nh, link);

      d_free(link->data);

      nick_end--;
   }

   nh = remove_nick(nh, user);

   nh = g_list_append(nh, d_strdup(user));

   nick_end = g_list_length(nh);
   nick_curr = nick_end;
   
   return nh;
}

void msg_cycle(STab *stab) {
   char *nick;
   
   nick_curr--;
   
   /* wrap cycling */
   if (nick_curr < 0)
     nick_curr = (nick_end - 1);
   
   nick = g_list_nth_data(nick_history, nick_curr);
   if (!nick)
     return;
   
   entry_set_text(stab->ct->entry, "/m %s ", nick);
}

void tab_complete(STab *stab, char *text) {
   ChannelInfo *chan;
   GList *ptr, *nicks, *cnicks = NULL;
   char *dptr, *lw, *cnick;
   int pn;
   
   pn = gnapster_get_pagenum(stab);
   if (pn <= 1)
     return;
   
   chan = g_list_nth_data(stab->ct->channel_list, pn - 2);
   if (chan->user)
     return;
   
   /* find the last complete word */
   lw = strrchr(text, ' ');
   lw = (!lw) ? text : lw + 1;
   
   if (!lw)
     return;

   nicks = j_comp_items(stab, GTK_CLIST(chan->clist));
   cnick = j_complete(nicks, &cnicks, lw);
   
   if (!cnick) {
      j_complete_free();
      return;
   }

   if (cnicks) {
      int i;
      
      for(i=0, ptr=cnicks; ptr; i=!i, ptr=ptr->next) {
	 dptr = ptr->data;
	 
	 hook_text_insert(stab, CURR, (i) ? MESSAGE : SYSTEM,
			  "user_list", "%-20s", dptr);
	 
	 if (i || !ptr->next)
	   gnapster_text_insert(stab, CURR, MESSAGE, "\n");
      }
      
      j_complete_free();
      
      return;
   }
   
   *lw = 0;
   
   entry_set_text(stab->ct->entry, "%s%s ", text, cnick);
   
   j_complete_free();
}

int console_keypress_cb(GtkWidget *w, GdkEventKey *key) {
   GList *ptr;
   STab *stab;
   
   stab = get_current_stab();
   
   if (key->keyval != GDK_Tab)
     nick_curr = nick_end;

   if (key->keyval >= GDK_1 && key->keyval <= GDK_9 && 
       key->state & GDK_MOD1_MASK) {
      gtk_notebook_set_page(GTK_NOTEBOOK(stab->ct->notebook),
			    (key->keyval - 49));
      
      return 1;
   }
   
   if (key->keyval == GDK_Tab) {
      char *text;

      gtk_signal_emit_stop_by_name(GTK_OBJECT(w), "key_press_event");
    
      text = gtk_entry_get_text(GTK_ENTRY(stab->ct->entry));
      if (!strlen(text) || nick_curr != nick_end)
	msg_cycle(stab);
      else
	tab_complete(stab, text);
      
      return 1;
   }
   
   if (key->keyval != GDK_Up && key->keyval != GDK_Down)
     return 0;
   
   switch(key->keyval) {
    case GDK_Up:
      loc_curr--;
      if (loc_curr < 0) loc_curr = 0;
      break;
    case GDK_Down:
      loc_curr++;
      if (loc_curr > loc_end) loc_curr = loc_end;
      break;
   }
   
   ptr = g_list_nth(console_history, loc_curr);
   
   if (ptr && ptr->data)
     gtk_entry_set_text(GTK_ENTRY(w), ptr->data);
   else if (loc_curr == loc_end)
     gtk_entry_set_text(GTK_ENTRY(w), "");
	
   gtk_signal_emit_stop_by_name(GTK_OBJECT(w), "key_press_event");
   
   return 1;
}

Search *get_data(const void *row) {
   return ((GtkCListRow *)row)->data;
}

User *get_chan_data(const void *row) {
   return ((GtkCListRow *)row)->data;
}

int compare_chan_user(GtkCList *clist, const void *row1, const void *row2) {
   User *s1 = get_chan_data(row1), *s2 = get_chan_data(row2);
   if (!s1 || !s2)
     return 0;
   
   return g_strcasecmp(s1->user, s2->user);
}

int compare_chan_conn(GtkCList *clist, const void *row1, const void *row2) {
   User *s1 = get_chan_data(row1), *s2 = get_chan_data(row2);
   if (!s1 || !s2)
     return 0;
   
   if (s1->conn <= s2->conn) return 1;
   if (s1->conn > s2->conn) return -1;
   
   return 0;
}

int compare_chan_shares(GtkCList *clist, const void *row1, const void *row2) {
   User *s1 = get_chan_data(row1), *s2 = get_chan_data(row2);
   if (!s1 || !s2) 
     return 0;
   
   if (s1->shares <= s2->shares) return 1;
   if (s1->shares > s2->shares) return -1;
   
   return 0;
}

void channel_sort(GtkCList *clist, int column) {
   void *sortfunc[3] = {
      compare_chan_user, compare_chan_shares, compare_chan_conn
   };
   
   if (column >= 3 || column < 0)
     return;
   
   gtk_clist_set_compare_func(clist, sortfunc[column]);
   
   if (clist->sort_column == column) {
      clist->sort_type = (clist->sort_type == GTK_SORT_ASCENDING) ? 
			 GTK_SORT_DESCENDING : GTK_SORT_ASCENDING;
   }
   
   gtk_clist_set_sort_column(clist, column);
   gtk_clist_sort(clist);
}

int compare_file(GtkCList *clist, const void *row1, const void *row2) {
   Search *s1 = get_data(row1), *s2 = get_data(row2);
   
   if (!s1 || !s2)
     return 0;
   
   return g_strcasecmp(s1->trunc_file, s2->trunc_file);
}

int compare_user(GtkCList *clist, const void *row1, const void *row2) {
   Search *s1 = get_data(row1), *s2 = get_data(row2);
   
   if (!s1 || !s2)
     return 0;
   
   return g_strcasecmp(s1->user, s2->user);
}

int compare_size(GtkCList *clist, const void *row1, const void *row2) {
   Search *s1 = get_data(row1), *s2 = get_data(row2);
   
   if (!s1 || !s2)
     return 0;
   
   if (s1->size <= s2->size) return 1;
   if (s1->size > s2->size) return -1;
   
   return 1;
}

int compare_length(GtkCList *clist, const void *row1, const void *row2) {
   Search *s1 = get_data(row1), *s2 = get_data(row2);
   int seconds1, seconds2;
   
   if (!s1 || !s2)
     return 0;
   
   seconds1 = (s1->size / ((s1->kbps * 1000) / 8));
   seconds2 = (s2->size / ((s2->kbps * 1000) / 8));
   
   if (seconds1 <= seconds2) return 1;
   if (seconds1 > seconds2) return -1;
   
   return 1;
}

int compare_bitrate(GtkCList *clist, const void *row1, const void *row2) {
   Search *s1 = get_data(row1), *s2 = get_data(row2);
   
   if (!s1 || !s2) 
     return 0;
   
   if (s1->kbps <= s2->kbps) return 1;
   if (s1->kbps > s2->kbps) return -1;
   
   return 0;
}

int compare_connection(GtkCList *clist, const void *row1, const void *row2) {
   Search *s1 = get_data(row1), *s2 = get_data(row2);
   int ret;
   
   if (!s1 || !s2)
     return 0;
   
   if (s1->connection <= s2->connection) return 1;
   if (s1->connection > s2->connection) return -1;
   
   ret = compare_user(clist, row1, row2);
   if (!ret)
     ret = compare_file(clist, row1, row2);
   
   return ret;
}

void search_sort(GtkCList *clist, int column) {
   void *sortfunc[6] = {
      compare_file, compare_connection, compare_user, compare_size, 
      compare_length, compare_bitrate
   };
   
   if (column >= 6 || column < 0)
     return;

   gtk_clist_set_compare_func(clist, sortfunc[column]);

   if (clist->sort_column == column) {
      clist->sort_type = (clist->sort_type == GTK_SORT_ASCENDING) ?
		    	 GTK_SORT_DESCENDING : GTK_SORT_ASCENDING;
   }
   
   gtk_clist_set_sort_column(clist, column);
   
   gtk_clist_sort(clist);
}

void cancel_search_cb() {
   STab *stab;
   
   stab = get_current_stab();
   
   if (!searching(stab->ci))
     return;
   
   cancels_inc(stab->ci, 1);
   
   end_search(stab, NULL);
}

void search_click_cb() {
/*   STab *stab;
   
   stab = get_current_stab();*/
   
   add_entry(gmain->st->gtk_entry, gmain->st->entry);
   
   search_activate_cb();
}

void browse_user_menu(GtkWidget *clist) {
   void *temp;
   Type *n;
   int row = 0;
   char user[1024];
   STab *stab;
   ConnInfo *ci;
   
   stab = get_current_stab();
   ci = stab->ci;
   
   if (!connected(ci) || browsing(ci))
     return;

   temp = gnapster_get_selection_data(clist, &row);
   if (!temp) 
     return;
   
   n = (Type *)temp;
   
   if (n->type == N_SEARCH )
     g_snprintf(user, sizeof(user), "%s", ((Search *)temp)->user);
   else if (n->type == TRANSFER)
     g_snprintf(user, sizeof(user), "%s", ((Transfer *)temp)->user);
   else if (n->type == N_USER)
     g_snprintf(user, sizeof(user), "%s", ((User *)temp)->user);
   else 
     return;
   
   browse_request(stab, user, 1);
}

void browse_user_menu_cb(GtkWidget *w, gpointer data) {
   GtkWidget *data_widget;

   if (!data)
     return;

   data_widget = (GtkWidget *)data;
   browse_user_menu(data_widget);
}

void browse_user_cb() {
   char *string;
   STab *stab;
   ConnInfo *ci;
   
   if (!(stab = GET_DATA(gmain->bt->omenu, "stab")))
     return;
   
   ci = stab->ci;
   
   if (!connected(ci) || browsing(ci))
     return;
   
   string = gtk_entry_get_text(GTK_ENTRY(gmain->bt->entry));
   if (!strlen(string)) 
     return;

   browse_request(stab, string, 1);
}

void whois_user_menu_cb(GtkWidget *w, gpointer data) {
   GtkWidget *clist;
   int row = 0;
   Search *temp;
   Type *n;
   char *user;
   STab *stab;
   
   if (!data) 
     return;
   
   clist = (GtkWidget *)data;
   if (!clist) 
     return;
   
   temp = gnapster_get_selection_data(clist, &row);
   if (!temp) 
     return;
   
   n = (Type *)temp;
   
   if (n->type == N_SEARCH)
     stab = temp->st, user = temp->user;
   else if (n->type == N_USER)
     stab = get_current_stab(), user = ((User *)temp)->user;
   else if (n->type == TRANSFER)
     stab = ((Transfer *)temp)->st, user = ((Transfer *)temp)->user;
   else
     return;
   
   hook_text_insert(stab, CURR, SYSTEM, "whois_ack", "%s",
		    user);
   napster_send(stab->ci->sock, NAPSTER_WHOIS_REQUEST, user);
}

void connect_cb(STab *stab) {
   ConnInfo *ci;
   
   ci = stab->ci;
   
   if (connected(ci) || connecting(ci))
     return;
   
/*   if (!user_info.user) {
      update_status(1, stab, _("Please edit your preferences before attempting to connect!"), 0);
      return;
   }*/
   
   CLOSE(ci->sock);
   INPUT_REMOVE(ci->sock_input);
   
   ci->state &= ~CONNECTED_MASK;
   ci->state |= CONNECTING_MASK;
   
   notebook_set_sensitive(gmain->srv_notebook, stab->pn + 4, 0);
   notebook_set_label(gmain->srv_notebook, stab->pn + 4, ci->server);

   if (ci->port == 8875) {
      update_status(1, stab, _("Finding best host..."), 1);
      napster_get_best_host(stab, ci->ip, ci->port);
   } else {
      update_status(1, stab, _("Connecting to server..."), 1);
      napster_connect(stab, ci->ip, ci->port);
   }

/*   j_config_set_string("/gnapster/Options/server", ci->ip);
   j_config_set_int("/gnapster/Options/server_port", ci->port);
   j_config_sync();*/
   
/*   d_free(user_info.server);
   user_info.server = d_config_get_string("/gnapster/Options/server");
   user_info.server_port = j_config_get_int("/gnapster/Options/server_port");
   
   if (!user_info.server || !strcmp(user_info.server, "208.184.216.222")) {
      j_config_set_string("/gnapster/Options/server", SERVER_NAPSTER_COM);
      j_config_set_int("/gnapster/Options/server_port", 8875);
      j_config_sync();
      user_info.server = d_config_get_string("/gnapster/Options/server");
      user_info.server_port = j_config_get_int("/gnapster/Options/server_port");
   }
   
   if (!user_info.server_port) {
      j_config_set_int("/gnapster/Options/server_port", 8875);
      j_config_sync();
      user_info.server_port = j_config_get_int("/gnapster/Options/server_port");
   }
   
   if (user_info.server_port == 8875) {
      update_status(1, stab, _("Finding best host..."), 1);
      napster_get_best_host(stab);
   } else {
      if (ci->ip)
	d_free(ci->ip);
      
      ci->ip = d_strdup(user_info.server);
      ci->port = user_info.server_port;
      
      update_status(1, stab, _("Connecting to server..."), 1);
      napster_connect(stab, ci->ip, ci->port);
   }*/
}

int reconnect_cb(void *data) {
   STab *stab;
   
   if (!data)
     return 0;
   
   stab = data;
   
   connect_cb(stab);
   
   TIMEOUT_REMOVE(stab->ci->reconnect_timeout);
   
   return 0;
}

void disconnect(STab *stab) {
   ConnInfo *ci;
   
   if (!stab)
     return;
   
   ci = stab->ci;
   
   if (!connected(ci) && !connecting(ci))
     return;
   
   /* we can assume these are set because the above statement failed */
   CLOSE(ci->sock);
   INPUT_REMOVE(ci->sock_input);

   /* remove it from the Servers submenu */
   remove_server_menu();
   
   /* if this was the last connected STab, unbind the Napster data port */
   if (!total_conn())
     napster_unbind();
   
   INPUT_REMOVE(ci->upload_input);
   
   nb_destroy(ci->sock + 1);
   
   gtk_label_set_text(GTK_LABEL(gmain->ut->label), "");
   
   gtk_entry_set_editable(GTK_ENTRY(gmain->st->gtk_entry), 1);
   gtk_entry_set_editable(GTK_ENTRY(gmain->bt->entry), 1);

   /* opennap only feature */
/*   gtk_widget_hide(gmain->st->not_label);
   gtk_widget_hide(gmain->st->not_entry);*/

   update_status(1, stab, _("Disconnected"), 0);
   
   shared_free(stab);
   
/*   gtk_xtext_remove_lines(GTK_XTEXT(stab->mt->text), -1, TRUE);*/
   
   channel_list_clear(stab);
   
   /* reconnection timeout */
   if (!safe_exit(ci))
     ci->reconnect_timeout = TIMEOUT_ADD(10000, reconnect_cb, stab);

   /* this holds things like opennap version, connected || connecting,
    * searching, browsing, etc etc */
   ci->state = 0;

   notebook_set_sensitive(gmain->srv_notebook, stab->pn + 4, 0);
/*   notebook_set_label(gmain->srv_notebook, stab->pn, "no server");*/
   
   append_server_menu();
}

void disconnect_all_cb() {
   FORSTABS(disconnect(stab);
	    TIMEOUT_REMOVE(stab->ci->reconnect_timeout));
}

void reconnect_all_cb() {
   FORSTABS(disconnect(stab);
	    connect_cb(stab));
}

void disconnect_cb() {
   STab *stab;
   
   stab = get_current_stab();
   
   disconnect(stab);
   
   TIMEOUT_REMOVE(stab->ci->reconnect_timeout);
}

void search_download_cb(GtkWidget *w, gpointer data) {
   download_true_cb(gmain->st->clist, NULL);
}

void browse_download_cb(GtkWidget *w, gpointer data) {
   download_true_cb(gmain->bt->ctree, NULL);
}

void cancel_upload_cb() {
   Transfer *temp;
   GtkWidget *clist;
   
   clist = gmain->ut->clist;
   
   temp = gnapster_get_selection_data(clist, NULL);
   if (!temp)
     return;
   
   upload_close(temp, TRANSFER_CANCEL);
}

char *listint_to_str(int i, int type) {
   char *cmp[] = { 
      "At least", "Equal to", "At best", NULL
   }, *conn[] = {
      "Unknown", "14.4", "28.8", "33.6", "56k", "64k ISDN", "128k ISDN", "Cable", "DSL",
      "T1", "T3+", NULL
   };
   
   switch(type) {
    case 0:
      return cmp[i];
    case 1:
      return conn[i];
    default:
      return NULL;
   }
}

int liststr_to_int(char *s) {
   char *conn[] = {
      "Unknown", "14.4", "28.8", "33.6", "56k", "64k ISDN", "128k ISDN", "Cable", "DSL",
      "T1", "T3+", NULL
   };
   char *bitrate[] = {
      "96", "112", "128", "160", "192", NULL
   };
   int i = 0;

   if (!strcmp(s, "At least"))
     return 0;
   else if (!strcmp(s, "Equal to"))
     return 1;
   else if (!strcmp(s, "At best"))
     return 2;
   
   while(conn[i]) {
      if (!strcmp(s, conn[i])) return i;
      i++;
   }
   i = 0;
   while(bitrate[i]) {
      if (!strcmp(s, bitrate[i])) return i;
      i++;
   }

   return 0;
}

void prop_set_state(GtkWidget *widget, WidgetProperty *prop, char *s, float f) {
   /* this routine may emit this signal, so we need to remove it */
   gtk_signal_disconnect_by_func(GTK_OBJECT(widget), 
				 GTK_SIGNAL_FUNC(combo_changed), NULL);
   
   if (GTK_IS_SPIN_BUTTON(widget))
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), f);
   else
     gtk_entry_set_text(GTK_ENTRY(widget), s);

   gtk_signal_connect(GTK_OBJECT(widget), "changed", 
		      GTK_SIGNAL_FUNC(combo_changed), NULL);
}

void combo_changed(GtkWidget *widget, gpointer data) {
   WidgetProperty *prop;
   char *s;
   int val;
   
   GET_WIDGET(widget, prop);
   if (!prop)
     return;
   
   s = NULL;
 
   if (GTK_IS_SPIN_BUTTON(widget))
     val = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
   else
     val = liststr_to_int((s = gtk_entry_get_text(GTK_ENTRY(widget))));
   
   user_info.conf[prop->key] = val;

   j_config_set_int(prop->conf, val);
   j_config_sync();
/*
   FORSTABS(if ((w = find_prop(stab, prop->key)) && (w != widget)) 
	       prop_set_state(w, prop, s, val));*/
}

void preferences_changed_cb() {
   j_property_box_changed(prop_dlg.window);
}

/* Preferences Save */

#ifdef HAVE_GNOME
void preferences_button_cb(GtkWidget *widget, gint arg1, gpointer data) {
#else
void preferences_button_cb(GtkWidget *widget, gpointer data) {
#endif /* HAVE_GNOME */
   char *nap_port, *conn, *download_dir, *upload_dir;
   char *font, *theme, *socks_server, *socks_port;
   char *socks5_username, *socks5_password;

   if (prop_dlg.calling++) {
      prop_dlg.calling = 0;
      
      return;
   }

   nap_port = gtk_entry_get_text(GTK_ENTRY(prop_dlg.nap_port));
   conn = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(prop_dlg.connection)->entry));
   download_dir = gtk_entry_get_text(GTK_ENTRY(j_file_entry_gtk_entry(prop_dlg.download_dir)));
   upload_dir = share_buildpathstr(prop_dlg.share_paths);
   
   socks_server = gtk_entry_get_text(GTK_ENTRY(prop_dlg.socks_server));
   socks_port = gtk_entry_get_text(GTK_ENTRY(prop_dlg.socks_port));
   
   socks5_username = gtk_entry_get_text(GTK_ENTRY(prop_dlg.socks5_username));
   socks5_password = gtk_entry_get_text(GTK_ENTRY(prop_dlg.socks5_password));
   
   font = gtk_entry_get_text(GTK_ENTRY(prop_dlg.font_entry));
   
   theme = gtk_entry_get_text(GTK_ENTRY(prop_dlg.curr_theme));
   
   if (!strlen(conn) || 
       !strlen(download_dir) || !strlen(nap_port) || !strlen(font) || 
       !strlen(theme)) {
      return;
   }
   
   if (user_info.conf[FIREWALL] && user_info.conf[SOCKS_FIREWALL]) {
      if (!strlen(socks_server) || !strlen(socks_port))
	user_info.conf[SOCKS_FIREWALL] = 0;
      else {
	 j_config_set_string("/gnapster/Connection/socks_server", socks_server);
	 j_config_set_string("/gnapster/Connection/socks_port", socks_port);
      }
      
      if (user_info.conf[SOCKS5]) {
	 j_config_set_string("/gnapster/Connection/socks5_username", 
			     socks5_username);
	 j_config_set_string("/gnapster/Connection/socks5_password", 
			     socks5_password);
      }
   }
   
   user_info.connection = napster_conn_as_int(conn);
   
   handle_rebind(my_atoi(nap_port));
   
   j_config_set_int("/gnapster/User/connection", user_info.connection);
   j_config_set_string("/gnapster/User/dl_dir", download_dir);
   
   if (j_strlen(upload_dir))
     j_config_set_string("/gnapster/User/ul_dir", upload_dir);
   
   d_free(upload_dir);
   
   j_config_set_string("/gnapster/xtext/font", font);
   
   j_config_set_string("/gnapster/Display/theme", theme);

   gnapster_config_set_int(0);

   if (!user_info.theme || strcmp(user_info.theme, theme)) {
      destroy_theme_hook(hooks);
      load_theme();
   }

   d_free(user_info.download_dir);
   d_free(user_info.upload_dir);
   
   user_info.download_dir = d_config_get_string("/gnapster/User/dl_dir");
   user_info.upload_dir = d_config_get_string("/gnapster/User/ul_dir");

#ifndef HAVE_GNOME
   if (!strcmp(data, "ok"))
     j_dialog_close(prop_dlg.window);
#endif /* !HAVE_GNOME */
}

void preferences_destroy_cb(GtkWidget *widget, gint arg1, gpointer data) {
#ifndef HAVE_GNOME
   j_dialog_close(prop_dlg.window);
#endif /* !HAVE_GNOME */
   
   prop_dlg.window = NULL;
}

void set_prop_tog_state(GtkWidget *w, WidgetProperty *prop, int sens, int data) {
   GList *ptr;
   
   for(ptr=prop->sensitive; ptr; ptr=ptr->next)
     gtk_widget_set_sensitive(GTK_WIDGET(ptr->data), data);
   
   if (!sens)
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), data);
}
   
void clicked_cb(GtkWidget *widget, int *data) {
   GtkToggleButton *button;
   WidgetProperty *prop;
   
   button = GTK_TOGGLE_BUTTON(widget);
   
   /* so that option buttons work still */
   *data = button->active ? 1 : 0;
   
   GET_WIDGET(widget, prop);
   if (!prop)
     return;
   
   j_config_set_int(prop->conf, *data);
   j_config_sync();
   
   set_prop_tog_state(widget, prop, 1, *data);
   
   /* make sure all the tabs are in sync with this one */
/*   FORSTABS(if ((w = find_prop(stab, prop->key)) && (w != widget)) set_prop_tog_state(w, prop, 0, *data));*/
}

void preferences_clicked_cb(GtkWidget *widget, int *data) {
   clicked_cb(widget, data);
   preferences_changed_cb();
}

void preferences_cb() {
   if (dialog_open(prop_dlg.window)) 
     return;
   
   create_prop_dlg(preferences_changed_cb,
		   preferences_button_cb,
		   preferences_destroy_cb,
		   preferences_clicked_cb,
		   &(user_info.new_user));
   
   gtk_widget_show_all(prop_dlg.window);
}

void build_dlg_cancel_cb() {
   building = 0;
   
   j_dialog_close(build_dlg.window);
   
   INPUT_REMOVE(build_dlg.pipe_input);
   
   if (build_dlg.cpid)
     kill(build_dlg.cpid, SIGUSR1);
}

void about_cb() {
   GtkWidget *about_box;
   const char *authors[] = {
      "Josh Guilfoyle (jasta) <jasta@gotlinux.org> - Main developer / project maintainer",
/*      "Aaron Lehmann <aaronl@vitelus.com> - anti-gnome bitch ;)", */
/*      "Jeff Hughes (valaxer) <valaxer@nwinet.com> - gnapster testing",*/
      NULL,
   };
   
   about_box = j_about_new(_("Gnapster"), VERSION,
			   _("Copyright (C) 1999-2001 Josh Guilfoyle"),
			   authors,
			   _("Napster (online mp3 community) client written for GTK+/GNOME"),
			   NULL);
   
   gtk_widget_show_all(about_box);
}
