/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>

#include "config.h"

#include "configdb.h"

#define BUFFER_SIZE 255
#define SEPERATOR   '='

FILE *open_file_for_reading(const char *filename, int secure) {
    FILE *f;

    f = fopen(filename, "a+");

    if(f == NULL) {
	printf("gmoo: Error in opening file: %s\n", strerror(errno));
	exit(1);
    }
    f = freopen(filename, "r", f);
    if(f == NULL) {
	printf("gmoo: Error in opening file: %s\n", strerror(errno));
	exit(1);
    }
    if(secure) chmod(filename, 00600);
    return f;
}

FILE *open_file_for_writing(const char *filename, int secure) {
    FILE *f;

    f = fopen(filename, "w+");
    
    if(f == NULL) {
	printf("gmoo: Error in opening file: %s\n", strerror(errno));
	exit(1);
    }
    f = freopen(filename, "w", f);
    if(f == NULL) {
	printf("gmoo: Error in opening file: %s\n", strerror(errno));
	exit(1);
    }
    if(secure) chmod(filename, 00600);
    return f;
}

void close_file(FILE *f) {
    if(fclose(f) != 0) {
	printf("gmoo: Error in closing file: %s\n", strerror(errno));
	exit(1);
    }
}

ConfigDBNode *parse_to_node(char *buffer) {
    char *s;
    int len;
    ConfigDBNode *node = NULL;

    s = strchr(buffer, SEPERATOR);
    if(s != NULL) {
	node = g_malloc(sizeof(ConfigDBNode));
	s[0] = '\0';
	node->key   = g_strdup(buffer);
	node->value = g_strdup(s + 1);
	len = strlen(node->value);
	if(len > 0) {
	    node->value[len - 1] = '\0';
	}
    }
    return node;
}

ConfigDBNode *read_node_from_file(FILE *f) {
    int i = 1;
    char *buffer = NULL;
    char *result;
    int len = 0;
    ConfigDBNode *node;

    while(1) {
	buffer = g_realloc(buffer, i * BUFFER_SIZE + 1);
	result = fgets(buffer + (i - 1) * BUFFER_SIZE, BUFFER_SIZE, f);
	len = strlen(buffer);
	if(result == NULL) { /* EOF */
	    return NULL;
	} else if(len == 1 && buffer[0] == '\n') { /* empty line */
	    g_free(buffer);
	    buffer = NULL;
	    i = 1;
	    continue;
	} else if(len > 0 && buffer[len - 1] == '\n') { /* Done reading */
	    node = parse_to_node(buffer);
	    g_free(buffer);
	    return node;
	}
	i++;
    }
}

ConfigDB *configdb_load(const char *filename, int secure) {
    GList *db = NULL;
    ConfigDBNode *node;
    FILE *f;

    f = open_file_for_reading(filename, secure);
    while((node = read_node_from_file(f)) != NULL) {
        db = g_list_append(db, (void *) node);

    }
    close_file(f);

    return db;
}

void free_node(void *data) {
    ConfigDBNode *node = data;

    g_free(node->key);
    g_free(node->value);
    g_free(node);
}

void configdb_free(ConfigDB *db) {
    g_list_foreach(db, (GFunc) free_node, NULL);
    g_list_free(db);
}


ConfigDBNode *find_node_with_key(ConfigDB *db, const char *key) {
    while(db != NULL) {
	if(strcmp(key, ((ConfigDBNode *)(db->data))->key) == 0)
	    return (ConfigDBNode *)(db->data);
	db = g_list_next(db);
    }
    return NULL;
}

int get_int_for(ConfigDB *db, const char *key, int *value) {
    ConfigDBNode *node = find_node_with_key(db, key);
    if(node == NULL) {
	return FALSE;
    }
    *value = atoi(node->value);
    return TRUE;
}

int get_string_for(ConfigDB *db, const char *key, char **value) {
    ConfigDBNode *node = find_node_with_key(db, key);
    if(node == NULL) {
	return FALSE;
    }
    *value = g_strdup(node->value);
    return TRUE;
}

int configdb_get_int(ConfigDB *db, const char *key, int default_value) {
    int return_value;
    if(!get_int_for(db, key, &return_value)) {
	return_value = default_value;
    }
    return return_value;
}

char *configdb_get_string(ConfigDB *db, const char *key,
			   const char *default_value) {
    char *return_value;
    if(!get_string_for(db, key, &return_value)) {
	return_value = g_strdup(default_value);
    }
    if(strcmp(_NULL, return_value) == 0) {
	g_free(return_value);
	return_value = NULL;
    }
    return return_value;
}

void  configdb_get_rgb(ConfigDB *db, const char *key, 
		       const char *default_value,
		       gushort *r, gushort *g, gushort *b) {
    char *s = configdb_get_string(db, key, default_value);
    if(s == NULL) {
	*r = 0; *g = 0; *b = 0;
    } else {
	sscanf(s, "%hx/%hx/%hx", r, g, b);
	g_free(s);
    }    
}



ConfigDB *configdb_set_int(ConfigDB *db, const char *key, int value) {
    ConfigDBNode *node;

    node = find_node_with_key(db, key);
    if(node == NULL) { /* add node */
	node = g_malloc(sizeof(ConfigDBNode));
	node->key = g_strdup(key);
	db = g_list_append(db, (void *) node);
    } else {
	g_free(node->value);
    }
    node->value = g_strdup_printf("%d", value);
    return db;
}

ConfigDB *configdb_set_string(ConfigDB *db, const char *key, 
			      const char *value) {
    ConfigDBNode *node;

    node = find_node_with_key(db, key);
    if(node == NULL) { /* add node */
	node = g_malloc(sizeof(ConfigDBNode));
	node->key = g_strdup(key);
	db = g_list_append(db, (void *) node);
    } else {
	g_free(node->value);
    }
    node->value = value == NULL ? g_strdup(_NULL) : g_strdup(value);
    return db;
}

ConfigDB *configdb_set_rgb(ConfigDB *db, const char *key, 
			   gushort r, gushort g, gushort b) {
    ConfigDBNode *node;

    node = find_node_with_key(db, key);
    if(node == NULL) { /* add node */
	node = g_malloc(sizeof(ConfigDBNode));
	node->key = g_strdup(key);
	db = g_list_append(db, (void *) node);
    } else {
	g_free(node->value);
    }
    node->value = g_strdup_printf("%.4hx/%.4hx/%.4hx", r, g, b);
    return db;
    
}


void store_node(void *data, void *file) {
    fprintf((FILE *) file, "%s=%s\n", 
            ((ConfigDBNode *)data)->key, 
            ((ConfigDBNode *)data)->value);
}

ConfigDB *configdb_delete(ConfigDB *db, const char *key) {
    ConfigDBNode *node = find_node_with_key(db, key);
    if(node != NULL) {
	db = g_list_remove(db, (void *) node);
	free_node((void *) node);
    }
    return db;
}

void configdb_store(ConfigDB *db, const char *filename, int secure) {
    FILE *f;

    f = open_file_for_writing(filename, secure);

    g_list_foreach(db, (GFunc) store_node, (void *) f);

    close_file(f);
}
