/*
    Glurp - A GTK+ client for Music Player Daemon
    Copyright (C) 2004, 2005 Andrej Kacian

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    http://musicpd.org/glurp.shtml

*/

#include <glib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>

#include "structs.h"
#include "support.h"
#include "comm.h"
#include "gui.h"

extern GlurpState *glurp;
extern GladeXML *guixml, *plxml;

gint glurp_connect() {
  if(CONNECTED) {
    debug("Already connected");
    statusbar_print("Already connected to server");
    return 0;
  }

  if(DISCONNECTING || CONNECTING) {
    debug("Disconnecting");
    statusbar_print("Connecting or disconnecting from server, please wait");
    return 0;
  }

  if(!glurp->config->server_host || !strlen(glurp->config->server_host)) {
    debug("Can not connect, no server set in config");
    statusbar_print("Can not connect, no server set in config");
    return 0;
  }

  if(!glurp->config->server_port || glurp->config->server_port > 65535) {
    debug("Invalid port specified in config");
    return 0;
  }

  glurp->conn_state = GLURP_CONN_STATE_CONNECTING;
  gui_set_connecting();

  glurp->conn = mpd_newConnection(glurp->config->server_host, glurp->config->server_port, 10);

  if(strlen(glurp->config->server_pass)) {
    debug("Sending password to server");
    mpd_sendPasswordCommand(glurp->conn, glurp->config->server_pass);
    mpd_finishCommand(glurp->conn);
  }

  if( check_mpd_error() ) {
    glurp_disconnect();
    return 0;
  }

  debug("Connected to %s:%d", glurp->config->server_host, glurp->config->server_port);
  statusbar_print("Connected to server %s:%d", glurp->config->server_host, glurp->config->server_port);

  debug("Server version: %d.%d.%d", glurp->conn->version[0], glurp->conn->version[1], glurp->conn->version[2]);

  title_print(glade_xml_get_widget(guixml, "glurp_window_main"), "%s:%d", glurp->config->server_host, glurp->config->server_port);

  if( !STREAM_CAPABLE_MPD ) {
    debug("MPD VERSION TOO OLD, DISCONNECTING");
    glurp_disconnect();
    statusbar_print("MPD version too old, disconnecting");
  }

  glurp->conn_state = GLURP_CONN_STATE_CONNECTED;
  gui_set_connected();

  debug("Adding a timeout call to gui_update().");
  g_timeout_add(glurp->config->refresh_rate, gui_update_cb, NULL);

  return 1;
}

void glurp_disconnect() {
  if( DISCONNECTED ) {
    debug("Not connected");
    if( glurp->conn ) {
      debug("Closing stale connection");
      mpd_closeConnection(glurp->conn);
      glurp->conn = NULL;
    }
    gui_set_disconnected();
    return;
  }

  if( CONNECTED || CONNECTING || DISCONNECTING ) {
    glurp->conn_state = GLURP_CONN_STATE_DISCONNECTING;
    debug("Disconnecting from server...");
    mpd_closeConnection(glurp->conn);
  }

  glurp->conn = NULL;

  glurp->prev_song_num = -1;
  glurp->playlist_version = 0;

  glurp->conn_state = GLURP_CONN_STATE_DISCONNECTED;
  glurp->play_state = MPD_STATUS_STATE_STOP;
  debug("Disconnected");

  gui_set_disconnected();
}

mpd_Status *get_status(gboolean standalone) {
  mpd_Status *status = NULL;

  if( !glurp->conn ) {
    debug("We're not connected");
    return NULL;
  }

  if(standalone) mpd_sendStatusCommand(glurp->conn);
  if( !(status = mpd_getStatus(glurp->conn)) ) {
    debug("Cannot retrieve mpd status!");
    if( check_mpd_error() ) {
      glurp_disconnect();
      return NULL;
    }
    if(standalone) mpd_finishCommand(glurp->conn);
    return NULL;
  }
  if(standalone) mpd_finishCommand(glurp->conn);

  return status;
}

void update_playlist() {
  mpd_InfoEntity *entity;

  if( !glurp->conn ) {
    debug("Not connected to server, returning");
    return;
  }

  statusbar_print("Updating playlist...");

  clear_playlist();

  mpd_sendPlaylistInfoCommand(glurp->conn, -1);

  while( (entity = mpd_getNextInfoEntity(glurp->conn)) ) {
    if( entity->type != MPD_INFO_ENTITY_TYPE_SONG ) {
      mpd_freeInfoEntity(entity);
      continue;
    }

    add_song(entity->info.song);

    mpd_freeInfoEntity(entity);
  }

  mpd_finishCommand(glurp->conn);
  if( !check_mpd_error() ) debug("playlist succesfully updated");
  else glurp_disconnect();
}

void clear_playlist() {
  GlurpSong *s, *ps = NULL;

  for( s = glurp->playlist; s; s = s->next ) {
    if(ps) g_free(ps);
    ps = s;
    if(s && !s->next) g_free(s);
  }

  glurp->playlist = NULL;
}

void get_playlist_list() {
  mpd_InfoEntity *entity;
  GlurpPl *npl, *pl;

  clear_playlist_list();

  if( !glurp->conn ) {
    debug("Not connected to server, returning");
    return;
  }

  mpd_sendLsInfoCommand(glurp->conn, "");

  while( (entity = mpd_getNextInfoEntity(glurp->conn)) ) {
    if( entity->type == MPD_INFO_ENTITY_TYPE_PLAYLISTFILE ) {
      npl = malloc(sizeof(GlurpPl));
      npl->name = g_strdup(entity->info.playlistFile->path);
      npl->next = NULL;

      debug("Received playlist '%s'", npl->name);

      for( pl = glurp->playlists; pl && pl->next; pl = pl->next ) {}

      if(pl) pl->next = npl;
      else glurp->playlists = npl;
    }
    mpd_freeInfoEntity(entity);
  }

  mpd_finishCommand(glurp->conn);
  if( !check_mpd_error() ) debug("list of playlists loaded");
  else glurp_disconnect();
}

void clear_playlist_list() {
  GlurpPl *s, *ps = NULL;

  for( s = glurp->playlists; s; s = s->next ) {
    if(ps) g_free(ps);
    ps = s;
    if(s && !s->next) g_free(s);
  }

  glurp->playlists = NULL;
}

void load_playlist(gchar *name) {
  if( !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(plxml, "checkbutton_append_playlist"))) ) {
    debug("clearing mpd playlist");
    mpd_sendClearCommand(glurp->conn);
    mpd_finishCommand(glurp->conn);
    if( check_mpd_error() ) {
      glurp_disconnect();
      return;
    }
  }

  debug("loading playlist '%s'", name);
  mpd_sendLoadCommand(glurp->conn, name);
  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) {
    glurp_disconnect();
    return;
  }
}

GlurpSong *glurp_get_nth_song(gint n) {
  GlurpSong *s = NULL;
  gint i = 0;

  for( s = glurp->playlist; s && i<n; s = s->next, i++ ) {}
  return s;
}

gboolean check_mpd_error() {
  if( glurp && glurp->conn && glurp->conn->error ) {
    debug("!!! MPD ERROR: %s", glurp->conn->errorStr);
    statusbar_print("Server error: %s", glurp->conn->errorStr);
    return TRUE;
  }
  return FALSE;
}

void glurp_add_add_dir(gchar *path, GtkTreePath *gpath) {
  mpd_InfoEntity *entity;

  if( !gpath && gtk_tree_model_iter_n_children(GTK_TREE_MODEL(glurp->gui_addtree), NULL) ) return;

  debug("Starting to list '%s'", path);

  mpd_sendLsInfoCommand(glurp->conn, path);

  check_mpd_error();

  while( (entity = mpd_getNextInfoEntity(glurp->conn)) ) {
    switch(entity->type) {
      case MPD_INFO_ENTITY_TYPE_DIRECTORY:
        gui_add_append(g_strdup(entity->info.directory->path), gpath, FALSE);
        break;
      case MPD_INFO_ENTITY_TYPE_SONG:
        gui_add_append(g_strdup(entity->info.song->file), gpath, TRUE);
	break;
    }

    mpd_freeInfoEntity(entity);
  }

  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) {
    glurp_disconnect();
    return;
  }
  
}

void glurp_add_search_result_dir(const gchar *what, gint type, GtkTreePath *gpath) {
  mpd_InfoEntity *entity;

  debug("Starting to search for '%s' with type %d", what, type);
  if( what == NULL || !strlen(what) ) return;

  mpd_sendSearchCommand(glurp->conn, type, what);

  check_mpd_error();

  while( (entity = mpd_getNextInfoEntity(glurp->conn)) ) {
    switch(entity->type) {
      case MPD_INFO_ENTITY_TYPE_DIRECTORY:
        gui_add_append(g_strdup(entity->info.directory->path), gpath, FALSE);
        break;
      case MPD_INFO_ENTITY_TYPE_SONG:
        gui_add_append(g_strdup(entity->info.song->file), gpath, TRUE);
	break;
    }

    mpd_freeInfoEntity(entity);
  }

  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) {
    glurp_disconnect();
    return;
  }
}

gboolean glurp_process_plchanges(mpd_Status *status) {
  mpd_InfoEntity *entity;
  gboolean changed = FALSE;

  if( status->playlistLength != get_num_songs() ) changed = TRUE;

  while( (entity = mpd_getNextInfoEntity(glurp->conn)) ) {
    switch(entity->type) {
      case MPD_INFO_ENTITY_TYPE_SONG:
        debug("Updating song (id:%d, pos:%d)", entity->info.song->id, entity->info.song->pos);
        update_song(entity->info.song);
	changed = TRUE;
	break;
    }
    mpd_freeInfoEntity(entity);
  }

  if( status->playlistLength < get_num_songs() ) {
    glurp_trim_playlist_end(status->playlistLength - 1);
    gui_trim_playlist_end(status->playlistLength - 1);
  }

  if( changed ) debug_print_playlist();

  glurp->playlist_version = status->playlist;
  return changed;
}

void glurp_update_song(mpd_Song *song) {
  GlurpSong *s;

  if( !(s = get_song_by_pos(song->pos)) ) {
    debug("Song not in playlist, adding");
    add_song(song);
    return;
  }

  debug("Updating song (pos:%d) to id:%d", song->pos, song->id);

  if( s->file && song->file && strcmp(s->file, song->file) ) g_free(s->file);
  if( song->file ) s->file = g_strdup(song->file);
  else s->file = g_strdup("");

  if( s->artist && song->artist && strcmp(s->artist, song->artist) ) g_free(s->artist);
  if( song->artist ) s->artist = g_strdup(song->artist);
  else s->artist = g_strdup("");

  if( s->title && song->title && strcmp(s->title, song->title) ) g_free(s->title);
  if( song->title ) s->title = g_strdup(song->title);
  else s->title = g_strdup("");

  if( s->album && song->album && strcmp(s->album, song->album) ) g_free(s->album);
  if( song->album ) s->album = g_strdup(song->album);
  else s->album = g_strdup("");

  if( s->track && song->track && strcmp(s->track, song->track) ) g_free(s->track);
  if( song->track ) s->track = g_strdup(song->track);
  else s->track = g_strdup("");

  if( s->name && song->name && strcmp(s->name, song->name) ) g_free(s->name);
  if( song->name ) s->name = g_strdup(song->name);
  else s->name = g_strdup("");

  s->pos = song->pos;
  s->id = song->id;

  s->time = song->time;
}

void glurp_trim_playlist_end(gint last) {
  GlurpSong *lasts = get_song_by_pos(last), *s = lasts->next, *ns;

  debug("Starting trimming beyond pos:%d", last);

  /* cut the linked list */
  if( last == -1 ) {
    debug("Purging entire playlist");
    s = glurp->playlist;
    glurp->playlist = NULL;
  } else if( lasts ) lasts->next = NULL;

  ns = s->next;

  while( s ) {
    debug("killing id:%d", s->id);
    g_free(s);
    s = ns;
    if( ns ) ns = ns->next;
  }
}
