/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/***********************
 * Texture management. *
 ***********************/

#include "gliv.h"

#include <math.h>               /* ceil() */
#include <GL/gl.h>

extern rt_struct *rt;

/* Lowest power of two bigger than the argument. */
G_GNUC_CONST static guint p2(guint p)
{
    guint ret = 1;

    while (ret < p)
        ret <<= 1;

    return ret;
}

/* Draws a piece of a multi-textures image or a whole mono-texture one. */
static void draw_rectangle(gdouble tex_x0, gdouble tex_x1,
                           gdouble tex_y0, gdouble tex_y1,
                           gdouble vert_x0, gdouble vert_x1,
                           gdouble vert_y0, gdouble vert_y1)
{
    /*
     * (tex_x0  ; tex_y0)  : Origin of the interesting part of the texture.
     * (tex_x1  ; tex_y1)  : Extremity of the interesting part of the texture.
     * (vert_x0 ; vert_y0) : Origin of the rectangle.
     * (vert_x1 ; vert_y1) : Extremity of the rectangle.
     */

    glBegin(GL_QUADS);

    glTexCoord2d(tex_x0, tex_y0);
    glVertex2d(vert_x0, vert_y0);

    glTexCoord2d(tex_x1, tex_y0);
    glVertex2d(vert_x1, vert_y0);

    glTexCoord2d(tex_x1, tex_y1);
    glVertex2d(vert_x1, vert_y1);

    glTexCoord2d(tex_x0, tex_y1);
    glVertex2d(vert_x0, vert_y1);

    glEnd();
}

/* 
 * The tiler was originally based on code from
 * evas by Carsten Haitzler (The Rasterman).
 */

/* 
 * Called twice for each rectangle to get rectangle
 * coordinates to put in the display list.
 */

static void compute_coordinates(gliv_image * im,
                                gdouble * vert0, gdouble * vert1,
                                gdouble * tex0, gdouble * tex1,
                                gboolean is_x, guint id)
{
    guint tiles, left, edge;
    gint dim;
    gdouble ratio;

    if (is_x == TRUE) {
        tiles = im->init->x_tiles;
        dim = im->width;
        left = im->init->x_left;
        edge = im->init->x_edge;
    } else {
        tiles = im->init->y_tiles;
        dim = im->height;
        left = im->init->y_left;
        edge = im->init->y_edge;
    }

    if (id == tiles - 1) {
        *vert0 = -dim / 2.0 + id * (rt->max_texture_size - 2.0);
        if (id == 0) {
            /* Single tile. */
            *tex0 = 0.0;
            *tex1 = (gdouble) left / edge;
            *vert1 = dim / 2.0;
        } else {
            /* Last tile. */
            *tex0 = 1.0 / edge;
            *tex1 = (left - 1.0) / (edge - 1.0);
            *vert1 = dim / 2.0 - 1.0;
        }

        /*
         * glTexImage2D(3G): All implementations support texture images
         * that are at least 64 texels (wide|high).
         */
        ratio = edge / 64.0;
        if (ratio < 1.0) {
            *tex0 *= ratio;
            *tex1 *= ratio;
        }
    } else {
        *tex1 = (rt->max_texture_size - 2.0) / (rt->max_texture_size - 1.0);
        if (id == 0) {
            /* First tile. */
            *vert0 = -dim / 2.0 - 1.0;
            *vert1 = *vert0 + rt->max_texture_size - 1.0;
            *tex0 = 0.0;
        } else {
            /* Middle tiles. */
            *vert0 = -dim / 2.0 + id * (rt->max_texture_size - 2.0);
            *vert1 = *vert0 + rt->max_texture_size - 2.0;
            *tex0 = 1.0 / (rt->max_texture_size + 1.0);
        }
    }
}

static void rectangle(gliv_image * im, tile_dim * tile, guint i, guint j)
{
    gdouble ty0, ty1, tx0, tx1;
    gdouble x0, x1, y0, y1;

    compute_coordinates(im, &x0, &x1, &tx0, &tx1, TRUE, i);
    compute_coordinates(im, &y0, &y1, &ty0, &ty1, FALSE, j);

    draw_rectangle(tx0, tx1, ty0, ty1, x0, x1, y0, y1);

    /* Used when drawing, to know which tiles are hidden. */
    tile->x0 = x0;
    tile->y0 = y0;
    tile->x1 = x1;
    tile->y1 = y1;
}

/* Shortcut to OpenGL parameters common to all textures. */
static void texture_parameter(void)
{
    /* These filters will be changed but it's important to initialize them. */
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
}

static void make_texture(gliv_image * im, guint i, guint j)
{
    static GdkPixbuf *tile_max_size = NULL;
    gboolean last_tile;
    GdkPixbuf *tile;
    gint x, y;
    guint w, h, gl_w, gl_h;

    if (tile_max_size == NULL && im->nb_tiles != 1 &&
        im->width >= rt->max_texture_size && im->height >= rt->max_texture_size)
        /* First tile. */
        tile_max_size = gdk_pixbuf_new(GDK_COLORSPACE_RGB, im->has_alpha, 8,
                                       rt->max_texture_size,
                                       rt->max_texture_size);

    x = i * (rt->max_texture_size - 2);
    y = j * (rt->max_texture_size - 2);

    if (i + 1 == im->init->x_tiles) {
        /* Last horizontal tile. */
        w = im->init->x_left;
        gl_w = im->init->x_edge;
        last_tile = TRUE;

        if (gl_w < 64)
            gl_w = 64;
    } else {
        w = gl_w = rt->max_texture_size;
        last_tile = FALSE;
    }

    if (j + 1 == im->init->y_tiles) {
        /* Last vertical tile. */
        h = im->init->y_left;
        gl_h = im->init->y_edge;

        if (gl_h < 64)
            gl_h = 64;
    } else {
        h = gl_h = rt->max_texture_size;
        last_tile = FALSE;
    }

    texture_parameter();

    if (gl_w == rt->max_texture_size && gl_h == rt->max_texture_size &&
        tile_max_size != NULL)

        tile = tile_max_size;
    else
        tile = gdk_pixbuf_new(GDK_COLORSPACE_RGB, im->has_alpha, 8, gl_w, gl_h);

    gdk_pixbuf_copy_area(im->init->im, x, y, w, h, tile, 0, 0);

    if (gl_w != w)
        /* Right border : copy the last column. */
        gdk_pixbuf_copy_area(tile, w - 1, 0, 1, h, tile, w, 0);

    if (gl_h != h) {
        /* Lower corner : copy the last line. */
        gdk_pixbuf_copy_area(tile, 0, h - 1, w, 1, tile, 0, h);

        if (gl_w != w)
            /* Lower-right corner : copy the last pixel. */
            gdk_pixbuf_copy_area(tile, w - 1, h - 1, 1, 1, tile, w, h);
    }

    glTexImage2D(GL_TEXTURE_2D, 0, 3 + im->has_alpha, gl_w, gl_h, 0,
                 (im->has_alpha == TRUE) ? GL_RGBA : GL_RGB, GL_UNSIGNED_BYTE,
                 gdk_pixbuf_get_pixels(tile));

    if (tile != tile_max_size)
        gdk_pixbuf_unref(tile);

    if (last_tile == TRUE && tile_max_size != NULL) {
        gdk_pixbuf_unref(tile_max_size);
        tile_max_size = NULL;
    }
}

static void tiles_parameters(gliv_image * im, gboolean is_x)
{
    guint *tiles, *left, *edge;
    gint dim;

    if (is_x == TRUE) {
        tiles = &im->init->x_tiles;
        left = &im->init->x_left;
        edge = &im->init->x_edge;
        dim = im->width;
    } else {
        tiles = &im->init->y_tiles;
        left = &im->init->y_left;
        edge = &im->init->y_edge;
        dim = im->height;
    }

    if (dim <= 2)
        *tiles = 1;
    else
        *tiles = (guint) ceil((dim - 2.0) / (rt->max_texture_size - 2.0));

    *left = dim - (*tiles - 1) * (rt->max_texture_size - 2);

    *edge = p2(*left);
}

void prioritize_textures(gliv_image * im, gboolean is_prio)
{
    GLclampf *priorities;
    guint i;

    priorities = g_new(GLclampf, im->nb_tiles);
    for (i = 0; i < im->nb_tiles; i++)
        priorities[i] = (gfloat) is_prio;

    glPrioritizeTextures(im->nb_tiles, im->tex_ids, priorities);
    g_free(priorities);
}


static void compute_gl_dimensions(gliv_image * im)
{
    tiles_parameters(im, TRUE);
    tiles_parameters(im, FALSE);

    im->nb_tiles = im->init->x_tiles * im->init->y_tiles;

    im->tex_ids = g_new(guint, im->nb_tiles);
    glGenTextures(im->nb_tiles, im->tex_ids);
    prioritize_textures(im, FALSE);

    im->list = glGenLists(im->nb_tiles);
}

void create_display_list(gliv_image * im)
{
    guint i, j;
    guint id = 0;

    compute_gl_dimensions(im);

    im->tiles = g_new(tile_dim, im->nb_tiles);

    for (j = 0; j < im->init->y_tiles; j++)
        for (i = 0; i < im->init->x_tiles; i++) {
            glBindTexture(GL_TEXTURE_2D, im->tex_ids[id]);
            make_texture(im, i, j);

            glNewList(im->list + id, GL_COMPILE);

            /* Redundant but need to be in the display list. */
            glBindTexture(GL_TEXTURE_2D, im->tex_ids[id]);

            rectangle(im, im->tiles + id, i, j);
            glEndList();

            id++;
        }
}
