/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*********************
 * Main source file. *
 *********************/

#include "gliv.h"
#include "cmdline.h"

#ifdef MAGICK_CONVERT
#include <signal.h>             /* SIG*, signal() */
#endif

#include <string.h>             /* memcpy(), strlen(), size_t */
#include <ctype.h>              /* isalnum() */

rt_struct *rt;
options_struct *options;
gliv_image *current_image;
GtkWidget *gl_widget;
GtkMenuBar *menu_bar;

static void check_flag(gboolean given, gboolean * flag, gboolean value)
{
    if (given == TRUE)
        *flag = value;
}

typedef struct {
    gboolean given;
    gboolean *flag;
    gboolean value;
} ggo_arg;

static void fill_flags(struct gengetopt_args_info *ggo)
{
/* *INDENT-OFF* */
    ggo_arg args[] = {
{ ggo->menu_given,         &options->menu_bar,     ggo->menu_flag         },
{ ggo->info_given,         &options->status_bar,   ggo->info_flag         },
{ ggo->zoom_pointer_given, &options->zoom_pointer, ggo->zoom_pointer_flag },
{ ggo->alpha_checks_given, &options->alpha_checks, ggo->alpha_checks_flag },
{ ggo->dither_given,       &options->dither,       ggo->dither_flag       },
{ ggo->force_load_given,   &options->force,        ggo->force_load_flag   },
{ ggo->build_menus_given,  &options->build_menus,  ggo->build_menus_flag  },
{ FALSE,                   NULL,                   FALSE                  }
};
/* *INDENT-ON* */

    ggo_arg *arg;

    for (arg = args; arg->flag != NULL; arg++)
        check_flag(arg->given, arg->flag, arg->value);
}

static void init_args(gint argc, gchar ** argv)
{
    struct gengetopt_args_info ggo;
    options_struct *rc_file;

    /* Command line (only the glivrc flag). */
    cmdline_parser(argc, argv, &ggo);

    /* Configuration file. */
    rc_file = load_rc(ggo.glivrc_flag);
    options = g_memdup(rc_file, sizeof(options_struct));

    /* Command line (remaining options). */

    if (ggo.full_screen_given == TRUE || ggo.make_fit_given == TRUE ||
        ggo.maximize_given == TRUE || ggo.scale_down_given == TRUE) {

        options->fullscreen = ggo.full_screen_flag || ggo.make_fit_flag ||
            ggo.maximize_flag || ggo.scale_down_flag;

        if (ggo.make_fit_given == TRUE)
            options->maximize = options->scaledown = ggo.make_fit_flag;
        else {

            check_flag(ggo.maximize_given, &options->maximize,
                       ggo.maximize_flag);

            check_flag(ggo.scale_down_given, &options->scaledown,
                       ggo.scale_down_flag);
        }

    } else if (options->maximize == TRUE || options->scaledown == TRUE)
        options->fullscreen = TRUE;

    if (ggo.delay_given == TRUE && ggo.delay_arg >= 0)
        options->delay = ggo.delay_arg;

    if (ggo.history_given == TRUE && ggo.history_arg >= -1)
        options->history_size = ggo.history_arg;

    fill_flags(&ggo);

    rt = g_new(rt_struct, 1);

    rt->cursor_hidden = FALSE;
    rt->help = FALSE;
    rt->alpha_checks_changed = TRUE;
    rt->scr_width = gdk_screen_width();
    rt->scr_height = gdk_screen_height();

    if (ggo.inputs_num == 0)
        show_open_dialog_first();
    else
        init_list(ggo.inputs, ggo.inputs_num,
                  ggo.recursive_flag, ggo.shuffle_flag);

    g_free(ggo.inputs);
}

#ifdef MAGICK_CONVERT
static void signal_finish(gint sig)
{
    image_magick_finish();
    signal(sig, SIG_DFL);
    raise(sig);
}
#endif

/* The returned string should not be freed. */
gchar *add_mnemonic(gchar * str)
{
#ifdef GTK2
    static gchar *result = NULL;
    size_t len, pos;

    /* + 1 for the underscore. */
    len = strlen(str) + 1;

    result = g_renew(gchar, result, len + 1);

    for (pos = 0; pos < len; pos++) {
        if (isalnum(str[pos])) {
            result[pos] = '_';
            break;
        } else
            result[pos] = str[pos];
    }

    memcpy(result + pos + 1, str + pos, (len - pos) * sizeof(gchar));
    return result;
#else
    return str;
#endif
}

/* The returned string should not be freed. */
gchar *locale_to_utf8(gchar * str)
{
#if defined(GTK2) && defined(ENABLE_NLS)
    static gchar *result = NULL;
    size_t len;
    gchar *tmp;

    tmp = g_locale_to_utf8(str, -1, NULL, NULL, NULL);
    len = strlen(tmp);
    result = g_renew(gchar, result, len + 1);
    memcpy(result, tmp, (len + 1) * sizeof(gchar));
    g_free(tmp);

    return result;
#else
    return str;
#endif
}

gint main(gint argc, gchar ** argv)
{
    /* i18n */
#ifdef ENABLE_NLS
    gtk_set_locale();
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif

    gtk_init(&argc, &argv);

    init_args(argc, argv);

#ifdef MAGICK_CONVERT
    signal(SIGHUP, signal_finish);
    signal(SIGINT, signal_finish);
    signal(SIGTERM, signal_finish);
    g_atexit(image_magick_finish);
#endif

    create_windows();

    gtk_main();

#ifdef MAGICK_CONVERT
    image_magick_finish();
#endif

    return 0;
}
