/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*****************************************************************
 * Callbacks to control the gl_widget through its parent window. *
 *****************************************************************/

#include "gliv.h"

#include <math.h>               /* atan2() */
#include <gdk/gdkkeysyms.h>     /* GDK_* */

extern rt_struct *rt;
extern GtkWidget *gl_widget;

/* Left button saved position in the root window. */
static gdouble old1_root_x, old1_root_y;

/* Left button saved position in the window. */
static gdouble old1_win_x, old1_win_y;

/* Right button saved position in the window. */
static gint old3_win_x, old3_win_y;

/* If the middle button is released. */
static gboolean mouse_wheel_zoom = TRUE;

/* Number of times the pointer warped in each direction. */
static gint nb_scr_w = 0, nb_scr_h = 0;

/* True if the first click was in the menu bar or the status bar. */
static gboolean motion_blocked = TRUE;

static gboolean button_press_event(GtkWidget * unused, GdkEventButton * event)
{
    motion_blocked = (event->window != gl_widget->window);

    switch (event->button) {
    case 1:
        /*
         * Remember the pointer position to know
         * the sliding distance, or the angle.
         */
        old1_root_x = event->x_root;
        old1_root_y = event->y_root;
        old1_win_x = event->x;
        old1_win_y = event->y;
        break;

    case 2:
        mouse_wheel_zoom = FALSE;
        break;

    case 3:
        /* We now have one of the zoom rectangle vertex. */
        old3_win_x = (gint) event->x;
        old3_win_y = (gint) event->y;
        break;

#ifndef GTK2
    case 4:
        /* Mouse-wheel up. */
        if (mouse_wheel_zoom == TRUE)
            zoom_in(1 / ZOOM_FACTOR);
        else {
            /*
             * Sometimes there is a pending event here so we
             * processe it to have gtk_events_pending() == 0.
             */
            gtk_main_iteration_do(FALSE);
            load_direction(-1);
        }
        break;

    case 5:
        /* Mouse-wheel down. */
        if (mouse_wheel_zoom == TRUE)
            zoom_in(ZOOM_FACTOR);
        else {
            /*
             * Sometimes there is a pending event here so we
             * processe it to have gtk_events_pending() == 0.
             */
            gtk_main_iteration_do(FALSE);
            load_direction(1);
        }
#endif
    }

    return TRUE;
}

static gboolean button_release_event(GtkWidget * unused, GdkEventButton * event)
{
    motion_blocked = TRUE;

    switch (event->button) {
    case 1:
        /* Reset mouse warping. */
        nb_scr_w = 0;
        nb_scr_h = 0;

        refresh(APPEND_HISTORY);
        break;

    case 2:
        mouse_wheel_zoom = TRUE;
        break;

    case 3:
        zoom_frame();
    }

    return TRUE;
}

/* Checks wether the pointer is on a border then moves it and returns TRUE. */
static gboolean warp_pointer(gint x, gint y)
{
    gint new_x, new_y;

    new_x = x;
    new_y = y;

    if (x == 0) {
        nb_scr_w--;
        new_x = rt->scr_width - 1;
    } else if (x == rt->scr_width - 1) {
        nb_scr_w++;
        new_x = 0;
    }

    if (y == 0) {
        nb_scr_h--;
        new_y = rt->scr_height - 1;
    } else if (y == rt->scr_height - 1) {
        nb_scr_h++;
        new_y = 0;
    }

    if (new_x == x && new_y == y)
        /* The pointer is not on a border. */
        return FALSE;

    move_pointer(new_x, new_y);
    return TRUE;
}

static gboolean process_move(GdkEventMotion * event)
{
    gdouble x1, y1;

    x1 = event->x_root + nb_scr_w * rt->scr_width;
    y1 = event->y_root + nb_scr_h * rt->scr_height;

    matrix_move(x1 - old1_root_x, y1 - old1_root_y);

    old1_root_x = x1;
    old1_root_y = y1;

    refresh(REFRESH_IMAGE);

    return warp_pointer((gint) event->x_root, (gint) event->y_root);
}

static void process_rotation(GdkEventMotion * event)
{
    gdouble x0, y0, x1, y1, angle;

    x0 = old1_win_x - rt->wid_size->width / 2.0;
    y0 = old1_win_y - rt->wid_size->height / 2.0;

    x1 = event->x - rt->wid_size->width / 2.0;
    y1 = event->y - rt->wid_size->height / 2.0;

    angle = atan2(y0, x0) + atan2(x1, y1) - PI / 2.0;
    matrix_rotate(angle);
    refresh(REFRESH_IMAGE | REFRESH_STATUS);
}

static gboolean motion_notify_event(GtkWidget * unused, GdkEventMotion * event)
{
    /* move_pointer() generates a motion_notify_event, useless here. */
    static gboolean skip = FALSE;

    if (skip == TRUE) {
        /* We come from a move_pointer(). */
        skip = FALSE;
        return TRUE;
    }

    if (rt->cursor_hidden == TRUE)
        toggle_cursor(TRUE);
    else
        schedule_hide_cursor();

    if (motion_blocked == TRUE)
        return TRUE;

    if ((event->state & GDK_BUTTON1_MASK) != 0) {
        if ((event->state & GDK_CONTROL_MASK) != 0) {

            if (event->window == gl_widget->window)
                process_rotation(event);

            old1_root_x = event->x_root + nb_scr_w * rt->scr_width;
            old1_root_y = event->y_root + nb_scr_h * rt->scr_height;

        } else if ((event->state & GDK_CONTROL_MASK) == 0)
            skip = process_move(event);

        /* Update saved coordinates. */

        old1_win_x = event->x;
        old1_win_y = event->y;
    }

    if ((event->state & GDK_BUTTON3_MASK) != 0 &&
        event->window == gl_widget->window) {
        /* We draw the zoom rectangle. */
        set_zoom_frame(old3_win_x, old3_win_y,
                       (gint) event->x - old3_win_x,
                       (gint) event->y - old3_win_y);

        draw_zoom_frame(TRUE);
    }

    return TRUE;
}

static void move_or_rotate(guint state, gdouble x, gdouble y, gdouble angle)
{
    guint what = REFRESH_IMAGE | APPEND_HISTORY;

    if ((state & GDK_CONTROL_MASK) == 0)
        matrix_move(x, y);
    else {
        matrix_rotate(angle);
        what |= REFRESH_STATUS;
    }

    refresh(what);
}

#define BIG_INCREMENT   (PI / 4.0)      /* 45 */
#define SMALL_INCREMENT (PI / 1800.0)   /* 0.1 */

static gboolean process_arrow_key(GdkEventKey * event)
{
    switch (event->keyval) {
    case GDK_Up:
    case GDK_KP_Up:
        move_or_rotate(event->state, 0.0, MOVE_OFFSET, BIG_INCREMENT);
        break;

    case GDK_Left:
    case GDK_KP_Left:
        move_or_rotate(event->state, MOVE_OFFSET, 0.0, SMALL_INCREMENT);
        break;

    case GDK_Right:
    case GDK_KP_Right:
        move_or_rotate(event->state, -MOVE_OFFSET, 0.0, -SMALL_INCREMENT);
        break;

    case GDK_Down:
    case GDK_KP_Down:
        move_or_rotate(event->state, 0.0, -MOVE_OFFSET, -BIG_INCREMENT);
        break;

    default:
        /* Key not found, try a keyboard accelerator. */
        return FALSE;
    }

    return TRUE;
}

static gboolean key_press_event(GtkWidget * widget, GdkEventKey * event)
{
    /*
     * The keys q, d, h, o, t, f, b, i, m, n, p, u, y, l, M
     * are processed through keyboard accelerators.
     */

    switch (event->keyval) {
    case GDK_Escape:
        gtk_main_quit();
        break;

    case GDK_KP_Space:
    case GDK_space:
        load_direction(1);
        break;

    case GDK_BackSpace:
        load_direction(-1);
        break;

    case GDK_r:
    case GDK_R:
        matrix_reset();
        refresh(REFRESH_IMAGE | REFRESH_STATUS | APPEND_HISTORY);
        break;

    case GDK_minus:
    case GDK_KP_Subtract:
        zoom_in(1 / ZOOM_FACTOR);
        break;

    case GDK_plus:
    case GDK_KP_Add:
        zoom_in(ZOOM_FACTOR);
        break;

    default:
        /* Key not found, try an arrow key and then a keyboard accelerator. */
        if (process_arrow_key(event) == FALSE)
            return FALSE;
    }

#ifndef GTK2
    gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
#endif
    return TRUE;
}

#ifdef GTK2
static gboolean scroll_event(GtkWidget * unused, GdkEventScroll * event)
{
    if (mouse_wheel_zoom == TRUE) {

        if (event->direction == GDK_SCROLL_UP)
            zoom_in(1 / ZOOM_FACTOR);
        else if (event->direction == GDK_SCROLL_DOWN)
            zoom_in(ZOOM_FACTOR);

    } else {
        /*
         * mouse_wheel_zoom == FALSE
         *
         * Sometimes there are two pending events here so we
         * processe them to have gtk_events_pending() == 0.
         */
        gtk_main_iteration_do(FALSE);
        gtk_main_iteration_do(FALSE);

        if (event->direction == GDK_SCROLL_UP)
            load_direction(-1);
        else if (event->direction == GDK_SCROLL_DOWN)
            load_direction(1);
    }

    return TRUE;
}
#endif

void install_callbacks(gpointer object)
{
    g_signal_connect(object, "button-press-event",
                     G_CALLBACK(button_press_event), NULL);

    g_signal_connect(object, "button-release-event",
                     G_CALLBACK(button_release_event), NULL);

    g_signal_connect(object, "motion-notify-event",
                     G_CALLBACK(motion_notify_event), NULL);

    g_signal_connect(object, "key-press-event",
                     G_CALLBACK(key_press_event), NULL);

    g_signal_connect(object, "delete-event", G_CALLBACK(gtk_main_quit), NULL);

#ifdef GTK2
    g_signal_connect(object, "scroll-event", G_CALLBACK(scroll_event), NULL);
#endif
}
