module Win32Window where

import StdDIS
import Win32Types
import GDITypes
import Win32WinMessage


----------------------------------------------------------------
-- Window Class
----------------------------------------------------------------

-- The classname must not be deallocated until the corresponding class
-- is deallocated.  For this reason, we represent classnames by pointers
-- and explicitly allocate the className.

type ClassName   = LPCTSTR

-- Note: this is one of those rare functions which doesnt free all 
-- its String arguments.
mkClassName :: String -> ClassName
mkClassName gc_arg1 =
  unsafePerformIO(
    (marshall_string_ gc_arg1) >>= \ (arg1) ->
    prim_mkClassName arg1
    >>= \  gc_res1  ->
    (return (gc_res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_mkClassName" prim_mkClassName :: Addr -> IO (Addr)

type ClassStyle   = UINT

cS_VREDRAW :: ClassStyle
cS_VREDRAW =
  unsafePerformIO(
    prim_cS_VREDRAW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_VREDRAW" prim_cS_VREDRAW :: IO (Word32)
cS_HREDRAW :: ClassStyle
cS_HREDRAW =
  unsafePerformIO(
    prim_cS_HREDRAW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_HREDRAW" prim_cS_HREDRAW :: IO (Word32)
cS_OWNDC :: ClassStyle
cS_OWNDC =
  unsafePerformIO(
    prim_cS_OWNDC
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_OWNDC" prim_cS_OWNDC :: IO (Word32)
cS_CLASSDC :: ClassStyle
cS_CLASSDC =
  unsafePerformIO(
    prim_cS_CLASSDC
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_CLASSDC" prim_cS_CLASSDC :: IO (Word32)
cS_PARENTDC :: ClassStyle
cS_PARENTDC =
  unsafePerformIO(
    prim_cS_PARENTDC
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_PARENTDC" prim_cS_PARENTDC :: IO (Word32)
cS_SAVEBITS :: ClassStyle
cS_SAVEBITS =
  unsafePerformIO(
    prim_cS_SAVEBITS
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_SAVEBITS" prim_cS_SAVEBITS :: IO (Word32)
cS_DBLCLKS :: ClassStyle
cS_DBLCLKS =
  unsafePerformIO(
    prim_cS_DBLCLKS
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_DBLCLKS" prim_cS_DBLCLKS :: IO (Word32)
cS_BYTEALIGNCLIENT :: ClassStyle
cS_BYTEALIGNCLIENT =
  unsafePerformIO(
    prim_cS_BYTEALIGNCLIENT
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_BYTEALIGNCLIENT" prim_cS_BYTEALIGNCLIENT :: IO (Word32)
cS_BYTEALIGNWINDOW :: ClassStyle
cS_BYTEALIGNWINDOW =
  unsafePerformIO(
    prim_cS_BYTEALIGNWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_BYTEALIGNWINDOW" prim_cS_BYTEALIGNWINDOW :: IO (Word32)
cS_NOCLOSE :: ClassStyle
cS_NOCLOSE =
  unsafePerformIO(
    prim_cS_NOCLOSE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_NOCLOSE" prim_cS_NOCLOSE :: IO (Word32)
cS_GLOBALCLASS :: ClassStyle
cS_GLOBALCLASS =
  unsafePerformIO(
    prim_cS_GLOBALCLASS
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cS_GLOBALCLASS" prim_cS_GLOBALCLASS :: IO (Word32)


type WNDCLASS =
 (ClassStyle,  -- style
  HINSTANCE,   -- hInstance
  MbHICON,     -- hIcon
  MbHCURSOR,   -- hCursor
  MbHBRUSH,    -- hbrBackground
  MbLPCSTR,    -- lpszMenuName
  ClassName)   -- lpszClassName

--ToDo!
--To avoid confusion with NULL, WNDCLASS requires you to add 1 to a SystemColor
--(which can be NULL)
-- %fun mkMbHBRUSH :: SystemColor -> MbHBRUSH
-- %code
-- %result ((HBRUSH)($0+1));

marshall_wndClass_ :: WNDCLASS -> IO Addr
marshall_wndClass_ gc_arg1 =
  case gc_arg1 of { (style,hInstance,hIcon,hCursor,hbrBackground,lpszMenuName,lpszClassName) ->
  (case hIcon of {
      Nothing -> (return (nullHANDLE));
      (Just hIcon) -> (return ((hIcon)))
   }) >>= \ (hIcon) ->
  (case hCursor of {
      Nothing -> (return (nullHANDLE));
      (Just hCursor) -> (return ((hCursor)))
   }) >>= \ (hCursor) ->
  (case hbrBackground of {
      Nothing -> (return (nullHANDLE));
      (Just hbrBackground) -> (return ((hbrBackground)))
   }) >>= \ (hbrBackground) ->
  (case lpszMenuName of {
      Nothing -> (return (nullAddr));
      (Just lpszMenuName) -> (return ((lpszMenuName)))
   }) >>= \ (lpszMenuName) ->
  prim_marshall_wndClass_ style hInstance hIcon hCursor hbrBackground lpszMenuName lpszClassName
  >>= \ gc_result ->
  access_prim_marshall_wndClass__c (gc_result :: Addr) >>= \ c ->
  access_prim_marshall_wndClass__gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_marshall_wndClass__gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (c))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_marshall_wndClass_" prim_marshall_wndClass_ :: Word32 -> Addr -> Addr -> Addr -> Addr -> Addr -> Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_marshall_wndClass__c" access_prim_marshall_wndClass__c :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_marshall_wndClass__gc_failed" access_prim_marshall_wndClass__gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_marshall_wndClass__gc_failstring" access_prim_marshall_wndClass__gc_failstring :: Addr -> IO (Addr)




registerClass :: WNDCLASS -> IO MbATOM
registerClass gc_arg1 =
  (marshall_wndClass_ gc_arg1) >>= \ (arg1) ->
  prim_registerClass arg1
  >>= \  res1  ->
  (if 0 == (res1)
   then return Nothing
   else (return ((Just res1)))) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_registerClass" prim_registerClass :: Addr -> IO (Word32)

unregisterClass :: ClassName -> HINSTANCE -> IO ()
unregisterClass arg1 arg2 =
  prim_unregisterClass arg1 arg2
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_unregisterClass" prim_unregisterClass :: Addr -> Addr -> IO ()

----------------------------------------------------------------
-- Window Style
----------------------------------------------------------------

type WindowStyle   = DWORD

wS_OVERLAPPED :: WindowStyle
wS_OVERLAPPED =
  unsafePerformIO(
    prim_wS_OVERLAPPED
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_OVERLAPPED" prim_wS_OVERLAPPED :: IO (Word32)
wS_POPUP :: WindowStyle
wS_POPUP =
  unsafePerformIO(
    prim_wS_POPUP
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_POPUP" prim_wS_POPUP :: IO (Word32)
wS_CHILD :: WindowStyle
wS_CHILD =
  unsafePerformIO(
    prim_wS_CHILD
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_CHILD" prim_wS_CHILD :: IO (Word32)
wS_CLIPSIBLINGS :: WindowStyle
wS_CLIPSIBLINGS =
  unsafePerformIO(
    prim_wS_CLIPSIBLINGS
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_CLIPSIBLINGS" prim_wS_CLIPSIBLINGS :: IO (Word32)
wS_CLIPCHILDREN :: WindowStyle
wS_CLIPCHILDREN =
  unsafePerformIO(
    prim_wS_CLIPCHILDREN
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_CLIPCHILDREN" prim_wS_CLIPCHILDREN :: IO (Word32)
wS_VISIBLE :: WindowStyle
wS_VISIBLE =
  unsafePerformIO(
    prim_wS_VISIBLE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_VISIBLE" prim_wS_VISIBLE :: IO (Word32)
wS_DISABLED :: WindowStyle
wS_DISABLED =
  unsafePerformIO(
    prim_wS_DISABLED
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_DISABLED" prim_wS_DISABLED :: IO (Word32)
wS_MINIMIZE :: WindowStyle
wS_MINIMIZE =
  unsafePerformIO(
    prim_wS_MINIMIZE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_MINIMIZE" prim_wS_MINIMIZE :: IO (Word32)
wS_MAXIMIZE :: WindowStyle
wS_MAXIMIZE =
  unsafePerformIO(
    prim_wS_MAXIMIZE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_MAXIMIZE" prim_wS_MAXIMIZE :: IO (Word32)
wS_CAPTION :: WindowStyle
wS_CAPTION =
  unsafePerformIO(
    prim_wS_CAPTION
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_CAPTION" prim_wS_CAPTION :: IO (Word32)
wS_BORDER :: WindowStyle
wS_BORDER =
  unsafePerformIO(
    prim_wS_BORDER
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_BORDER" prim_wS_BORDER :: IO (Word32)
wS_DLGFRAME :: WindowStyle
wS_DLGFRAME =
  unsafePerformIO(
    prim_wS_DLGFRAME
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_DLGFRAME" prim_wS_DLGFRAME :: IO (Word32)
wS_VSCROLL :: WindowStyle
wS_VSCROLL =
  unsafePerformIO(
    prim_wS_VSCROLL
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_VSCROLL" prim_wS_VSCROLL :: IO (Word32)
wS_HSCROLL :: WindowStyle
wS_HSCROLL =
  unsafePerformIO(
    prim_wS_HSCROLL
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_HSCROLL" prim_wS_HSCROLL :: IO (Word32)
wS_SYSMENU :: WindowStyle
wS_SYSMENU =
  unsafePerformIO(
    prim_wS_SYSMENU
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_SYSMENU" prim_wS_SYSMENU :: IO (Word32)
wS_THICKFRAME :: WindowStyle
wS_THICKFRAME =
  unsafePerformIO(
    prim_wS_THICKFRAME
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_THICKFRAME" prim_wS_THICKFRAME :: IO (Word32)
wS_MINIMIZEBOX :: WindowStyle
wS_MINIMIZEBOX =
  unsafePerformIO(
    prim_wS_MINIMIZEBOX
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_MINIMIZEBOX" prim_wS_MINIMIZEBOX :: IO (Word32)
wS_MAXIMIZEBOX :: WindowStyle
wS_MAXIMIZEBOX =
  unsafePerformIO(
    prim_wS_MAXIMIZEBOX
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_MAXIMIZEBOX" prim_wS_MAXIMIZEBOX :: IO (Word32)
wS_GROUP :: WindowStyle
wS_GROUP =
  unsafePerformIO(
    prim_wS_GROUP
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_GROUP" prim_wS_GROUP :: IO (Word32)
wS_TABSTOP :: WindowStyle
wS_TABSTOP =
  unsafePerformIO(
    prim_wS_TABSTOP
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_TABSTOP" prim_wS_TABSTOP :: IO (Word32)
wS_OVERLAPPEDWINDOW :: WindowStyle
wS_OVERLAPPEDWINDOW =
  unsafePerformIO(
    prim_wS_OVERLAPPEDWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_OVERLAPPEDWINDOW" prim_wS_OVERLAPPEDWINDOW :: IO (Word32)
wS_POPUPWINDOW :: WindowStyle
wS_POPUPWINDOW =
  unsafePerformIO(
    prim_wS_POPUPWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_POPUPWINDOW" prim_wS_POPUPWINDOW :: IO (Word32)
wS_CHILDWINDOW :: WindowStyle
wS_CHILDWINDOW =
  unsafePerformIO(
    prim_wS_CHILDWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_CHILDWINDOW" prim_wS_CHILDWINDOW :: IO (Word32)
wS_TILED :: WindowStyle
wS_TILED =
  unsafePerformIO(
    prim_wS_TILED
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_TILED" prim_wS_TILED :: IO (Word32)
wS_ICONIC :: WindowStyle
wS_ICONIC =
  unsafePerformIO(
    prim_wS_ICONIC
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_ICONIC" prim_wS_ICONIC :: IO (Word32)
wS_SIZEBOX :: WindowStyle
wS_SIZEBOX =
  unsafePerformIO(
    prim_wS_SIZEBOX
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_SIZEBOX" prim_wS_SIZEBOX :: IO (Word32)
wS_TILEDWINDOW :: WindowStyle
wS_TILEDWINDOW =
  unsafePerformIO(
    prim_wS_TILEDWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_TILEDWINDOW" prim_wS_TILEDWINDOW :: IO (Word32)

type WindowStyleEx   = DWORD

wS_EX_DLGMODALFRAME :: WindowStyleEx
wS_EX_DLGMODALFRAME =
  unsafePerformIO(
    prim_wS_EX_DLGMODALFRAME
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_DLGMODALFRAME" prim_wS_EX_DLGMODALFRAME :: IO (Word32)
wS_EX_NOPARENTNOTIFY :: WindowStyleEx
wS_EX_NOPARENTNOTIFY =
  unsafePerformIO(
    prim_wS_EX_NOPARENTNOTIFY
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_NOPARENTNOTIFY" prim_wS_EX_NOPARENTNOTIFY :: IO (Word32)
wS_EX_TOPMOST :: WindowStyleEx
wS_EX_TOPMOST =
  unsafePerformIO(
    prim_wS_EX_TOPMOST
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_TOPMOST" prim_wS_EX_TOPMOST :: IO (Word32)
wS_EX_ACCEPTFILES :: WindowStyleEx
wS_EX_ACCEPTFILES =
  unsafePerformIO(
    prim_wS_EX_ACCEPTFILES
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_ACCEPTFILES" prim_wS_EX_ACCEPTFILES :: IO (Word32)
wS_EX_TRANSPARENT :: WindowStyleEx
wS_EX_TRANSPARENT =
  unsafePerformIO(
    prim_wS_EX_TRANSPARENT
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_TRANSPARENT" prim_wS_EX_TRANSPARENT :: IO (Word32)
wS_EX_MDICHILD :: WindowStyleEx
wS_EX_MDICHILD =
  unsafePerformIO(
    prim_wS_EX_MDICHILD
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_MDICHILD" prim_wS_EX_MDICHILD :: IO (Word32)
wS_EX_TOOLWINDOW :: WindowStyleEx
wS_EX_TOOLWINDOW =
  unsafePerformIO(
    prim_wS_EX_TOOLWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_TOOLWINDOW" prim_wS_EX_TOOLWINDOW :: IO (Word32)
wS_EX_WINDOWEDGE :: WindowStyleEx
wS_EX_WINDOWEDGE =
  unsafePerformIO(
    prim_wS_EX_WINDOWEDGE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_WINDOWEDGE" prim_wS_EX_WINDOWEDGE :: IO (Word32)
wS_EX_CLIENTEDGE :: WindowStyleEx
wS_EX_CLIENTEDGE =
  unsafePerformIO(
    prim_wS_EX_CLIENTEDGE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_CLIENTEDGE" prim_wS_EX_CLIENTEDGE :: IO (Word32)
wS_EX_CONTEXTHELP :: WindowStyleEx
wS_EX_CONTEXTHELP =
  unsafePerformIO(
    prim_wS_EX_CONTEXTHELP
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_CONTEXTHELP" prim_wS_EX_CONTEXTHELP :: IO (Word32)
wS_EX_RIGHT :: WindowStyleEx
wS_EX_RIGHT =
  unsafePerformIO(
    prim_wS_EX_RIGHT
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_RIGHT" prim_wS_EX_RIGHT :: IO (Word32)
wS_EX_LEFT :: WindowStyleEx
wS_EX_LEFT =
  unsafePerformIO(
    prim_wS_EX_LEFT
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_LEFT" prim_wS_EX_LEFT :: IO (Word32)
wS_EX_RTLREADING :: WindowStyleEx
wS_EX_RTLREADING =
  unsafePerformIO(
    prim_wS_EX_RTLREADING
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_RTLREADING" prim_wS_EX_RTLREADING :: IO (Word32)
wS_EX_LTRREADING :: WindowStyleEx
wS_EX_LTRREADING =
  unsafePerformIO(
    prim_wS_EX_LTRREADING
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_LTRREADING" prim_wS_EX_LTRREADING :: IO (Word32)
wS_EX_LEFTSCROLLBAR :: WindowStyleEx
wS_EX_LEFTSCROLLBAR =
  unsafePerformIO(
    prim_wS_EX_LEFTSCROLLBAR
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_LEFTSCROLLBAR" prim_wS_EX_LEFTSCROLLBAR :: IO (Word32)
wS_EX_RIGHTSCROLLBAR :: WindowStyleEx
wS_EX_RIGHTSCROLLBAR =
  unsafePerformIO(
    prim_wS_EX_RIGHTSCROLLBAR
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_RIGHTSCROLLBAR" prim_wS_EX_RIGHTSCROLLBAR :: IO (Word32)
wS_EX_CONTROLPARENT :: WindowStyleEx
wS_EX_CONTROLPARENT =
  unsafePerformIO(
    prim_wS_EX_CONTROLPARENT
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_CONTROLPARENT" prim_wS_EX_CONTROLPARENT :: IO (Word32)
wS_EX_STATICEDGE :: WindowStyleEx
wS_EX_STATICEDGE =
  unsafePerformIO(
    prim_wS_EX_STATICEDGE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_STATICEDGE" prim_wS_EX_STATICEDGE :: IO (Word32)
wS_EX_APPWINDOW :: WindowStyleEx
wS_EX_APPWINDOW =
  unsafePerformIO(
    prim_wS_EX_APPWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_APPWINDOW" prim_wS_EX_APPWINDOW :: IO (Word32)
wS_EX_OVERLAPPEDWINDOW :: WindowStyleEx
wS_EX_OVERLAPPEDWINDOW =
  unsafePerformIO(
    prim_wS_EX_OVERLAPPEDWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_OVERLAPPEDWINDOW" prim_wS_EX_OVERLAPPEDWINDOW :: IO (Word32)
wS_EX_PALETTEWINDOW :: WindowStyleEx
wS_EX_PALETTEWINDOW =
  unsafePerformIO(
    prim_wS_EX_PALETTEWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_wS_EX_PALETTEWINDOW" prim_wS_EX_PALETTEWINDOW :: IO (Word32)

cW_USEDEFAULT :: Pos
cW_USEDEFAULT =
  unsafePerformIO(
    prim_cW_USEDEFAULT
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_cW_USEDEFAULT" prim_cW_USEDEFAULT :: IO (Int)

type Pos = Int

type MbPos = Maybe Pos

type WindowClosure = HWND -> WindowMessage -> WPARAM -> LPARAM -> IO LRESULT


 
marshall_windowClosure_ :: WindowClosure -> IO Addr
marshall_windowClosure_ cl = mkWindowClosure cl

foreign export ccall dynamic mkWindowClosure :: WindowClosure -> IO Addr




setWindowClosure :: HWND -> WindowClosure -> IO ()
setWindowClosure hwnd gc_arg1 =
  (marshall_windowClosure_ gc_arg1) >>= \ (closure) ->
  prim_setWindowClosure hwnd closure
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_setWindowClosure" prim_setWindowClosure :: Addr -> Addr -> IO ()

createWindow :: ClassName -> String -> WindowStyle -> Maybe Pos -> Maybe Pos -> Maybe Pos -> Maybe Pos -> MbHWND -> MbHMENU -> HINSTANCE -> WindowClosure -> IO HWND
createWindow name gc_arg1 style x y width height hwndParent hmenu hinst gc_arg2 =
  (marshall_string_ gc_arg1) >>= \ (windowName) ->
  (case x of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just x) -> (return ((x)))
   }) >>= \ (x) ->
  (case y of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just y) -> (return ((y)))
   }) >>= \ (y) ->
  (case width of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just width) -> (return ((width)))
   }) >>= \ (width) ->
  (case height of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just height) -> (return ((height)))
   }) >>= \ (height) ->
  (case hwndParent of {
      Nothing -> (return (nullHANDLE));
      (Just hwndParent) -> (return ((hwndParent)))
   }) >>= \ (hwndParent) ->
  (case hmenu of {
      Nothing -> (return (nullHANDLE));
      (Just hmenu) -> (return ((hmenu)))
   }) >>= \ (hmenu) ->
  (marshall_windowClosure_ gc_arg2) >>= \ (closure) ->
  prim_createWindow name windowName style x y width height hwndParent hmenu hinst closure
  >>= \ gc_result ->
  access_prim_createWindow_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_createWindow_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_createWindow_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_createWindow" prim_createWindow :: Addr -> Addr -> Word32 -> Int -> Int -> Int -> Int -> Addr -> Addr -> Addr -> Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_createWindow_res1" access_prim_createWindow_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_createWindow_gc_failed" access_prim_createWindow_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_createWindow_gc_failstring" access_prim_createWindow_gc_failstring :: Addr -> IO (Addr)

-- Freeing the title/window name has been reported
-- to cause a crash, so let's not do it.
-- %end free(windowName)  

createWindowEx :: WindowStyle -> ClassName -> String -> WindowStyle -> Maybe Pos -> Maybe Pos -> Maybe Pos -> Maybe Pos -> MbHWND -> MbHMENU -> HINSTANCE -> WindowClosure -> IO HWND
createWindowEx estyle cls gc_arg1 wstyle x y nWidth nHeight hwndParent hmenu hinstance gc_arg2 =
  (marshall_string_ gc_arg1) >>= \ (wname) ->
  (case x of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just x) -> (return ((x)))
   }) >>= \ (x) ->
  (case y of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just y) -> (return ((y)))
   }) >>= \ (y) ->
  (case nWidth of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just nWidth) -> (return ((nWidth)))
   }) >>= \ (nWidth) ->
  (case nHeight of {
      Nothing -> (return (cW_USEDEFAULT));
      (Just nHeight) -> (return ((nHeight)))
   }) >>= \ (nHeight) ->
  (case hwndParent of {
      Nothing -> (return (nullHANDLE));
      (Just hwndParent) -> (return ((hwndParent)))
   }) >>= \ (hwndParent) ->
  (case hmenu of {
      Nothing -> (return (nullHANDLE));
      (Just hmenu) -> (return ((hmenu)))
   }) >>= \ (hmenu) ->
  (marshall_windowClosure_ gc_arg2) >>= \ (closure) ->
  prim_createWindowEx estyle cls wname wstyle x y nWidth nHeight hwndParent hmenu hinstance closure
  >>= \ gc_result ->
  access_prim_createWindowEx_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_createWindowEx_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_createWindowEx_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_createWindowEx" prim_createWindowEx :: Word32 -> Addr -> Addr -> Word32 -> Int -> Int -> Int -> Int -> Addr -> Addr -> Addr -> Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_createWindowEx_res1" access_prim_createWindowEx_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_createWindowEx_gc_failed" access_prim_createWindowEx_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_createWindowEx_gc_failstring" access_prim_createWindowEx_gc_failstring :: Addr -> IO (Addr)

-- see CreateWindow comment.
-- %end free(wname)  

----------------------------------------------------------------

defWindowProc :: MbHWND -> WindowMessage -> WPARAM -> LPARAM -> IO LRESULT
defWindowProc arg1 arg2 arg3 arg4 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_defWindowProc arg1 arg2 arg3 arg4
  >>= \  res1  ->
  (return (res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_defWindowProc" prim_defWindowProc :: Addr -> Word32 -> Word32 -> Int32 -> IO (Int32)

----------------------------------------------------------------

getClientRect :: HWND -> IO RECT
getClientRect arg1 =
  prim_getClientRect arg1
  >>= \ gc_result ->
  access_prim_getClientRect_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_getClientRect_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_getClientRect_gc_res3 (gc_result :: Addr) >>= \ gc_res3 ->
  access_prim_getClientRect_gc_res4 (gc_result :: Addr) >>= \ gc_res4 ->
  access_prim_getClientRect_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getClientRect_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2,gc_res3,gc_res4)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getClientRect" prim_getClientRect :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getClientRect_gc_res1" access_prim_getClientRect_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getClientRect_gc_res2" access_prim_getClientRect_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getClientRect_gc_res3" access_prim_getClientRect_gc_res3 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getClientRect_gc_res4" access_prim_getClientRect_gc_res4 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getClientRect_gc_failed" access_prim_getClientRect_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getClientRect_gc_failstring" access_prim_getClientRect_gc_failstring :: Addr -> IO (Addr)

getWindowRect :: HWND -> IO RECT
getWindowRect arg1 =
  prim_getWindowRect arg1
  >>= \ gc_result ->
  access_prim_getWindowRect_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_getWindowRect_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_getWindowRect_gc_res3 (gc_result :: Addr) >>= \ gc_res3 ->
  access_prim_getWindowRect_gc_res4 (gc_result :: Addr) >>= \ gc_res4 ->
  access_prim_getWindowRect_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getWindowRect_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2,gc_res3,gc_res4)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getWindowRect" prim_getWindowRect :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getWindowRect_gc_res1" access_prim_getWindowRect_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getWindowRect_gc_res2" access_prim_getWindowRect_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getWindowRect_gc_res3" access_prim_getWindowRect_gc_res3 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getWindowRect_gc_res4" access_prim_getWindowRect_gc_res4 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getWindowRect_gc_failed" access_prim_getWindowRect_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getWindowRect_gc_failstring" access_prim_getWindowRect_gc_failstring :: Addr -> IO (Addr)

-- Should it be MbRECT instead?
invalidateRect :: MbHWND -> MbLPRECT -> Bool -> IO ()
invalidateRect arg1 arg2 gc_arg1 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  (case arg2 of {
      Nothing -> (return (nullAddr));
      (Just arg2) -> (return ((arg2)))
   }) >>= \ (arg2) ->
  (marshall_bool_ gc_arg1) >>= \ (arg3) ->
  prim_invalidateRect arg1 arg2 arg3
  >>= \ gc_result ->
  access_prim_invalidateRect_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_invalidateRect_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_invalidateRect" prim_invalidateRect :: Addr -> Addr -> Int -> IO (Addr)
foreign import ccall unsafe "access_prim_invalidateRect_gc_failed" access_prim_invalidateRect_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_invalidateRect_gc_failstring" access_prim_invalidateRect_gc_failstring :: Addr -> IO (Addr)

screenToClient :: HWND -> POINT -> IO POINT
screenToClient arg1 gc_arg1 =
  case gc_arg1 of { (gc_arg2,gc_arg3) ->
  prim_screenToClient arg1 gc_arg2 gc_arg3
  >>= \ gc_result ->
  access_prim_screenToClient_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_screenToClient_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_screenToClient_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_screenToClient_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2)))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_screenToClient" prim_screenToClient :: Addr -> Int32 -> Int32 -> IO (Addr)
foreign import ccall unsafe "access_prim_screenToClient_gc_res1" access_prim_screenToClient_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_screenToClient_gc_res2" access_prim_screenToClient_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_screenToClient_gc_failed" access_prim_screenToClient_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_screenToClient_gc_failstring" access_prim_screenToClient_gc_failstring :: Addr -> IO (Addr)

clientToScreen :: HWND -> POINT -> IO POINT
clientToScreen hwnd gc_arg1 =
  case gc_arg1 of { (gc_arg2,gc_arg3) ->
  prim_clientToScreen hwnd gc_arg2 gc_arg3
  >>= \ gc_result ->
  access_prim_clientToScreen_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_clientToScreen_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_clientToScreen_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_clientToScreen_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2)))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_clientToScreen" prim_clientToScreen :: Addr -> Int32 -> Int32 -> IO (Addr)
foreign import ccall unsafe "access_prim_clientToScreen_gc_res1" access_prim_clientToScreen_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_clientToScreen_gc_res2" access_prim_clientToScreen_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_clientToScreen_gc_failed" access_prim_clientToScreen_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_clientToScreen_gc_failstring" access_prim_clientToScreen_gc_failstring :: Addr -> IO (Addr)

----------------------------------------------------------------
-- Setting window text/label
----------------------------------------------------------------
-- For setting the title bar text.  But inconvenient to make the LPCTSTR
setWindowText :: HWND -> String -> IO ()
setWindowText arg1 gc_arg1 =
  (marshall_string_ gc_arg1) >>= \ (arg2) ->
  prim_setWindowText arg1 arg2
  >>= \ gc_result ->
  access_prim_setWindowText_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_setWindowText_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_setWindowText" prim_setWindowText :: Addr -> Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_setWindowText_gc_failed" access_prim_setWindowText_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_setWindowText_gc_failstring" access_prim_setWindowText_gc_failstring :: Addr -> IO (Addr)

----------------------------------------------------------------
-- Paint struct
----------------------------------------------------------------

type PAINTSTRUCT =
 ( HDC   -- hdc
 , Bool  -- fErase
 , RECT  -- rcPaint
 )

type LPPAINTSTRUCT   = Addr

sizeofPAINTSTRUCT :: DWORD
sizeofPAINTSTRUCT =
  unsafePerformIO(
    prim_sizeofPAINTSTRUCT
    >>= \  gc_res1  ->
    (return (gc_res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sizeofPAINTSTRUCT" prim_sizeofPAINTSTRUCT :: IO (Word32)

beginPaint :: HWND -> LPPAINTSTRUCT -> IO HDC
beginPaint arg1 arg2 =
  prim_beginPaint arg1 arg2
  >>= \ gc_result ->
  access_prim_beginPaint_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_beginPaint_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_beginPaint_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_beginPaint" prim_beginPaint :: Addr -> Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_beginPaint_res1" access_prim_beginPaint_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_beginPaint_gc_failed" access_prim_beginPaint_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_beginPaint_gc_failstring" access_prim_beginPaint_gc_failstring :: Addr -> IO (Addr)

endPaint :: HWND -> LPPAINTSTRUCT -> IO ()
endPaint arg1 arg2 =
  prim_endPaint arg1 arg2
foreign import  ccall "Win32Window_stub_ffi.h prim_endPaint" prim_endPaint :: Addr -> Addr -> IO ()
-- Apparently always succeeds (return non-zero)

----------------------------------------------------------------
-- ShowWindow
----------------------------------------------------------------

type ShowWindowControl   = DWORD

sW_HIDE :: ShowWindowControl
sW_HIDE =
  unsafePerformIO(
    prim_sW_HIDE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_HIDE" prim_sW_HIDE :: IO (Word32)
sW_SHOWNORMAL :: ShowWindowControl
sW_SHOWNORMAL =
  unsafePerformIO(
    prim_sW_SHOWNORMAL
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOWNORMAL" prim_sW_SHOWNORMAL :: IO (Word32)
sW_SHOWMINIMIZED :: ShowWindowControl
sW_SHOWMINIMIZED =
  unsafePerformIO(
    prim_sW_SHOWMINIMIZED
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOWMINIMIZED" prim_sW_SHOWMINIMIZED :: IO (Word32)
sW_SHOWMAXIMIZED :: ShowWindowControl
sW_SHOWMAXIMIZED =
  unsafePerformIO(
    prim_sW_SHOWMAXIMIZED
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOWMAXIMIZED" prim_sW_SHOWMAXIMIZED :: IO (Word32)
sW_MAXIMIZE :: ShowWindowControl
sW_MAXIMIZE =
  unsafePerformIO(
    prim_sW_MAXIMIZE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_MAXIMIZE" prim_sW_MAXIMIZE :: IO (Word32)
sW_SHOWNOACTIVATE :: ShowWindowControl
sW_SHOWNOACTIVATE =
  unsafePerformIO(
    prim_sW_SHOWNOACTIVATE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOWNOACTIVATE" prim_sW_SHOWNOACTIVATE :: IO (Word32)
sW_SHOW :: ShowWindowControl
sW_SHOW =
  unsafePerformIO(
    prim_sW_SHOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOW" prim_sW_SHOW :: IO (Word32)
sW_MINIMIZE :: ShowWindowControl
sW_MINIMIZE =
  unsafePerformIO(
    prim_sW_MINIMIZE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_MINIMIZE" prim_sW_MINIMIZE :: IO (Word32)
sW_SHOWMINNOACTIVE :: ShowWindowControl
sW_SHOWMINNOACTIVE =
  unsafePerformIO(
    prim_sW_SHOWMINNOACTIVE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOWMINNOACTIVE" prim_sW_SHOWMINNOACTIVE :: IO (Word32)
sW_SHOWNA :: ShowWindowControl
sW_SHOWNA =
  unsafePerformIO(
    prim_sW_SHOWNA
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_SHOWNA" prim_sW_SHOWNA :: IO (Word32)
sW_RESTORE :: ShowWindowControl
sW_RESTORE =
  unsafePerformIO(
    prim_sW_RESTORE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sW_RESTORE" prim_sW_RESTORE :: IO (Word32)

showWindow :: HWND -> ShowWindowControl -> IO Bool
showWindow arg1 arg2 =
  prim_showWindow arg1 arg2
  >>= \  res1  ->
  (unmarshall_bool_ res1) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_showWindow" prim_showWindow :: Addr -> Word32 -> IO (Int)

----------------------------------------------------------------
-- Misc
----------------------------------------------------------------

adjustWindowRect :: RECT -> WindowStyle -> Bool -> IO RECT
adjustWindowRect gc_arg1 arg2 gc_arg7 =
  case gc_arg1 of { (gc_arg2,gc_arg3,gc_arg4,gc_arg5) ->
  (marshall_bool_ gc_arg7) >>= \ (arg3) ->
  prim_adjustWindowRect gc_arg2 gc_arg3 gc_arg4 gc_arg5 arg2 arg3
  >>= \ gc_result ->
  access_prim_adjustWindowRect_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_adjustWindowRect_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_adjustWindowRect_gc_res3 (gc_result :: Addr) >>= \ gc_res3 ->
  access_prim_adjustWindowRect_gc_res4 (gc_result :: Addr) >>= \ gc_res4 ->
  access_prim_adjustWindowRect_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_adjustWindowRect_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2,gc_res3,gc_res4)))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_adjustWindowRect" prim_adjustWindowRect :: Int32 -> Int32 -> Int32 -> Int32 -> Word32 -> Int -> IO (Addr)
foreign import ccall unsafe "access_prim_adjustWindowRect_gc_res1" access_prim_adjustWindowRect_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRect_gc_res2" access_prim_adjustWindowRect_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRect_gc_res3" access_prim_adjustWindowRect_gc_res3 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRect_gc_res4" access_prim_adjustWindowRect_gc_res4 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRect_gc_failed" access_prim_adjustWindowRect_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_adjustWindowRect_gc_failstring" access_prim_adjustWindowRect_gc_failstring :: Addr -> IO (Addr)

adjustWindowRectEx :: RECT -> WindowStyle -> Bool -> WindowStyleEx -> IO RECT
adjustWindowRectEx gc_arg1 arg2 gc_arg7 arg4 =
  case gc_arg1 of { (gc_arg2,gc_arg3,gc_arg4,gc_arg5) ->
  (marshall_bool_ gc_arg7) >>= \ (arg3) ->
  prim_adjustWindowRectEx gc_arg2 gc_arg3 gc_arg4 gc_arg5 arg2 arg3 arg4
  >>= \ gc_result ->
  access_prim_adjustWindowRectEx_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_adjustWindowRectEx_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_adjustWindowRectEx_gc_res3 (gc_result :: Addr) >>= \ gc_res3 ->
  access_prim_adjustWindowRectEx_gc_res4 (gc_result :: Addr) >>= \ gc_res4 ->
  access_prim_adjustWindowRectEx_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_adjustWindowRectEx_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2,gc_res3,gc_res4)))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_adjustWindowRectEx" prim_adjustWindowRectEx :: Int32 -> Int32 -> Int32 -> Int32 -> Word32 -> Int -> Word32 -> IO (Addr)
foreign import ccall unsafe "access_prim_adjustWindowRectEx_gc_res1" access_prim_adjustWindowRectEx_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRectEx_gc_res2" access_prim_adjustWindowRectEx_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRectEx_gc_res3" access_prim_adjustWindowRectEx_gc_res3 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRectEx_gc_res4" access_prim_adjustWindowRectEx_gc_res4 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_adjustWindowRectEx_gc_failed" access_prim_adjustWindowRectEx_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_adjustWindowRectEx_gc_failstring" access_prim_adjustWindowRectEx_gc_failstring :: Addr -> IO (Addr)

-- Win2K and later:
-- %fun AllowSetForegroundWindow :: DWORD -> IO ()

-- %
-- %dis animateWindowType x = dWORD x
-- type AnimateWindowType   = DWORD

-- %const AnimateWindowType
--        [ AW_SLIDE
--        , AW_ACTIVATE
--        , AW_BLEND
--        , AW_HIDE
--        , AW_CENTER
--        , AW_HOR_POSITIVE
--        , AW_HOR_NEGATIVE
--        , AW_VER_POSITIVE
--        , AW_VER_NEGATIVE
--        ]

-- Win98 or Win2K:
-- %fun AnimateWindow :: HWND -> DWORD -> AnimateWindowType -> IO ()
-- %code BOOL success = AnimateWindow(arg1,arg2,arg3)
-- %fail { !success } { ErrorWin("AnimateWindow") }

anyPopup :: IO Bool
anyPopup =
  prim_anyPopup
  >>= \  res1  ->
  (unmarshall_bool_ res1) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_anyPopup" prim_anyPopup :: IO (Int)

arrangeIconicWindows :: HWND -> IO ()
arrangeIconicWindows arg1 =
  prim_arrangeIconicWindows arg1
  >>= \ gc_result ->
  access_prim_arrangeIconicWindows_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_arrangeIconicWindows_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_arrangeIconicWindows" prim_arrangeIconicWindows :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_arrangeIconicWindows_gc_failed" access_prim_arrangeIconicWindows_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_arrangeIconicWindows_gc_failstring" access_prim_arrangeIconicWindows_gc_failstring :: Addr -> IO (Addr)

beginDeferWindowPos :: Int -> IO HDWP
beginDeferWindowPos arg1 =
  prim_beginDeferWindowPos arg1
  >>= \ gc_result ->
  access_prim_beginDeferWindowPos_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_beginDeferWindowPos_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_beginDeferWindowPos_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_beginDeferWindowPos" prim_beginDeferWindowPos :: Int -> IO (Addr)
foreign import ccall unsafe "access_prim_beginDeferWindowPos_res1" access_prim_beginDeferWindowPos_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_beginDeferWindowPos_gc_failed" access_prim_beginDeferWindowPos_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_beginDeferWindowPos_gc_failstring" access_prim_beginDeferWindowPos_gc_failstring :: Addr -> IO (Addr)

bringWindowToTop :: HWND -> IO ()
bringWindowToTop arg1 =
  prim_bringWindowToTop arg1
  >>= \ gc_result ->
  access_prim_bringWindowToTop_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_bringWindowToTop_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_bringWindowToTop" prim_bringWindowToTop :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_bringWindowToTop_gc_failed" access_prim_bringWindowToTop_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_bringWindowToTop_gc_failstring" access_prim_bringWindowToTop_gc_failstring :: Addr -> IO (Addr)

childWindowFromPoint :: HWND -> POINT -> IO MbHWND
childWindowFromPoint hwnd gc_arg1 =
  case gc_arg1 of { (gc_arg2,gc_arg3) ->
  prim_childWindowFromPoint hwnd gc_arg2 gc_arg3
  >>= \ gc_result ->
  access_prim_childWindowFromPoint_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_childWindowFromPoint_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_childWindowFromPoint_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (if nullHANDLE == (res1)
	then return Nothing
	else (return ((Just res1)))) >>= \ gc_res1 ->
       (return (gc_res1))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_childWindowFromPoint" prim_childWindowFromPoint :: Addr -> Int32 -> Int32 -> IO (Addr)
foreign import ccall unsafe "access_prim_childWindowFromPoint_res1" access_prim_childWindowFromPoint_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_childWindowFromPoint_gc_failed" access_prim_childWindowFromPoint_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_childWindowFromPoint_gc_failstring" access_prim_childWindowFromPoint_gc_failstring :: Addr -> IO (Addr)

childWindowFromPointEx :: HWND -> POINT -> DWORD -> IO MbHWND
childWindowFromPointEx hwnd gc_arg1 arg2 =
  case gc_arg1 of { (gc_arg2,gc_arg3) ->
  prim_childWindowFromPointEx hwnd gc_arg2 gc_arg3 arg2
  >>= \ gc_result ->
  access_prim_childWindowFromPointEx_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_childWindowFromPointEx_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_childWindowFromPointEx_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (if nullHANDLE == (res1)
	then return Nothing
	else (return ((Just res1)))) >>= \ gc_res1 ->
       (return (gc_res1))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_childWindowFromPointEx" prim_childWindowFromPointEx :: Addr -> Int32 -> Int32 -> Word32 -> IO (Addr)
foreign import ccall unsafe "access_prim_childWindowFromPointEx_res1" access_prim_childWindowFromPointEx_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_childWindowFromPointEx_gc_failed" access_prim_childWindowFromPointEx_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_childWindowFromPointEx_gc_failstring" access_prim_childWindowFromPointEx_gc_failstring :: Addr -> IO (Addr)

closeWindow :: HWND -> IO ()
closeWindow arg1 =
  prim_closeWindow arg1
  >>= \ gc_result ->
  access_prim_closeWindow_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_closeWindow_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_closeWindow" prim_closeWindow :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_closeWindow_gc_failed" access_prim_closeWindow_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_closeWindow_gc_failstring" access_prim_closeWindow_gc_failstring :: Addr -> IO (Addr)

deferWindowPos :: HDWP -> HWND -> HWND -> Int -> Int -> Int -> Int -> SetWindowPosFlags -> IO HDWP
deferWindowPos arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 =
  prim_deferWindowPos arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8
  >>= \ gc_result ->
  access_prim_deferWindowPos_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_deferWindowPos_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_deferWindowPos_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_deferWindowPos" prim_deferWindowPos :: Addr -> Addr -> Addr -> Int -> Int -> Int -> Int -> Word32 -> IO (Addr)
foreign import ccall unsafe "access_prim_deferWindowPos_res1" access_prim_deferWindowPos_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_deferWindowPos_gc_failed" access_prim_deferWindowPos_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_deferWindowPos_gc_failstring" access_prim_deferWindowPos_gc_failstring :: Addr -> IO (Addr)

destroyWindow :: HWND -> IO ()
destroyWindow arg1 =
  prim_destroyWindow arg1
  >>= \ gc_result ->
  access_prim_destroyWindow_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_destroyWindow_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_destroyWindow" prim_destroyWindow :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_destroyWindow_gc_failed" access_prim_destroyWindow_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_destroyWindow_gc_failstring" access_prim_destroyWindow_gc_failstring :: Addr -> IO (Addr)

endDeferWindowPos :: HDWP -> IO ()
endDeferWindowPos arg1 =
  prim_endDeferWindowPos arg1
  >>= \ gc_result ->
  access_prim_endDeferWindowPos_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_endDeferWindowPos_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_endDeferWindowPos" prim_endDeferWindowPos :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_endDeferWindowPos_gc_failed" access_prim_endDeferWindowPos_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_endDeferWindowPos_gc_failstring" access_prim_endDeferWindowPos_gc_failstring :: Addr -> IO (Addr)

findWindow :: String -> String -> IO MbHWND
findWindow gc_arg1 gc_arg2 =
  (marshall_string_ gc_arg1) >>= \ (arg1) ->
  (marshall_string_ gc_arg2) >>= \ (arg2) ->
  prim_findWindow arg1 arg2
  >>= \  res1  ->
  (if nullHANDLE == (res1)
   then return Nothing
   else (return ((Just res1)))) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_findWindow" prim_findWindow :: Addr -> Addr -> IO (Addr)

findWindowEx :: HWND -> HWND -> String -> String -> IO MbHWND
findWindowEx arg1 arg2 gc_arg1 gc_arg2 =
  (marshall_string_ gc_arg1) >>= \ (arg3) ->
  (marshall_string_ gc_arg2) >>= \ (arg4) ->
  prim_findWindowEx arg1 arg2 arg3 arg4
  >>= \  res1  ->
  (if nullHANDLE == (res1)
   then return Nothing
   else (return ((Just res1)))) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_findWindowEx" prim_findWindowEx :: Addr -> Addr -> Addr -> Addr -> IO (Addr)

flashWindow :: HWND -> Bool -> IO Bool
flashWindow arg1 gc_arg1 =
  (marshall_bool_ gc_arg1) >>= \ (arg2) ->
  prim_flashWindow arg1 arg2
  >>= \  res1  ->
  (unmarshall_bool_ res1) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_flashWindow" prim_flashWindow :: Addr -> Int -> IO (Int)
-- No error code

moveWindow :: HWND -> Int -> Int -> Int -> Int -> Bool -> IO ()
moveWindow arg1 arg2 arg3 arg4 arg5 gc_arg1 =
  (marshall_bool_ gc_arg1) >>= \ (arg6) ->
  prim_moveWindow arg1 arg2 arg3 arg4 arg5 arg6
  >>= \ gc_result ->
  access_prim_moveWindow_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_moveWindow_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_moveWindow" prim_moveWindow :: Addr -> Int -> Int -> Int -> Int -> Int -> IO (Addr)
foreign import ccall unsafe "access_prim_moveWindow_gc_failed" access_prim_moveWindow_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_moveWindow_gc_failstring" access_prim_moveWindow_gc_failstring :: Addr -> IO (Addr)

getDesktopWindow :: IO HWND
getDesktopWindow =
  prim_getDesktopWindow
  >>= \  res1  ->
  (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getDesktopWindow" prim_getDesktopWindow :: IO (Addr)

getForegroundWindow :: IO HWND
getForegroundWindow =
  prim_getForegroundWindow
  >>= \  res1  ->
  (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getForegroundWindow" prim_getForegroundWindow :: IO (Addr)

getParent :: HWND -> IO HWND
getParent arg1 =
  prim_getParent arg1
  >>= \ gc_result ->
  access_prim_getParent_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_getParent_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getParent_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getParent" prim_getParent :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getParent_res1" access_prim_getParent_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getParent_gc_failed" access_prim_getParent_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getParent_gc_failstring" access_prim_getParent_gc_failstring :: Addr -> IO (Addr)

getTopWindow :: HWND -> IO HWND
getTopWindow arg1 =
  prim_getTopWindow arg1
  >>= \ gc_result ->
  access_prim_getTopWindow_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_getTopWindow_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getTopWindow_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getTopWindow" prim_getTopWindow :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getTopWindow_res1" access_prim_getTopWindow_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getTopWindow_gc_failed" access_prim_getTopWindow_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getTopWindow_gc_failstring" access_prim_getTopWindow_gc_failstring :: Addr -> IO (Addr)


type SetWindowPosFlags = UINT

sWP_NOSIZE :: SetWindowPosFlags
sWP_NOSIZE =
  unsafePerformIO(
    prim_sWP_NOSIZE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOSIZE" prim_sWP_NOSIZE :: IO (Word32)
sWP_NOMOVE :: SetWindowPosFlags
sWP_NOMOVE =
  unsafePerformIO(
    prim_sWP_NOMOVE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOMOVE" prim_sWP_NOMOVE :: IO (Word32)
sWP_NOZORDER :: SetWindowPosFlags
sWP_NOZORDER =
  unsafePerformIO(
    prim_sWP_NOZORDER
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOZORDER" prim_sWP_NOZORDER :: IO (Word32)
sWP_NOREDRAW :: SetWindowPosFlags
sWP_NOREDRAW =
  unsafePerformIO(
    prim_sWP_NOREDRAW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOREDRAW" prim_sWP_NOREDRAW :: IO (Word32)
sWP_NOACTIVATE :: SetWindowPosFlags
sWP_NOACTIVATE =
  unsafePerformIO(
    prim_sWP_NOACTIVATE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOACTIVATE" prim_sWP_NOACTIVATE :: IO (Word32)
sWP_FRAMECHANGED :: SetWindowPosFlags
sWP_FRAMECHANGED =
  unsafePerformIO(
    prim_sWP_FRAMECHANGED
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_FRAMECHANGED" prim_sWP_FRAMECHANGED :: IO (Word32)
sWP_SHOWWINDOW :: SetWindowPosFlags
sWP_SHOWWINDOW =
  unsafePerformIO(
    prim_sWP_SHOWWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_SHOWWINDOW" prim_sWP_SHOWWINDOW :: IO (Word32)
sWP_HIDEWINDOW :: SetWindowPosFlags
sWP_HIDEWINDOW =
  unsafePerformIO(
    prim_sWP_HIDEWINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_HIDEWINDOW" prim_sWP_HIDEWINDOW :: IO (Word32)
sWP_NOCOPYBITS :: SetWindowPosFlags
sWP_NOCOPYBITS =
  unsafePerformIO(
    prim_sWP_NOCOPYBITS
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOCOPYBITS" prim_sWP_NOCOPYBITS :: IO (Word32)
sWP_NOOWNERZORDER :: SetWindowPosFlags
sWP_NOOWNERZORDER =
  unsafePerformIO(
    prim_sWP_NOOWNERZORDER
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOOWNERZORDER" prim_sWP_NOOWNERZORDER :: IO (Word32)
sWP_NOSENDCHANGING :: SetWindowPosFlags
sWP_NOSENDCHANGING =
  unsafePerformIO(
    prim_sWP_NOSENDCHANGING
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOSENDCHANGING" prim_sWP_NOSENDCHANGING :: IO (Word32)
sWP_DRAWFRAME :: SetWindowPosFlags
sWP_DRAWFRAME =
  unsafePerformIO(
    prim_sWP_DRAWFRAME
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_DRAWFRAME" prim_sWP_DRAWFRAME :: IO (Word32)
sWP_NOREPOSITION :: SetWindowPosFlags
sWP_NOREPOSITION =
  unsafePerformIO(
    prim_sWP_NOREPOSITION
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_sWP_NOREPOSITION" prim_sWP_NOREPOSITION :: IO (Word32)

----------------------------------------------------------------
-- HDCs
----------------------------------------------------------------

type GetDCExFlags   = DWORD

dCX_WINDOW :: GetDCExFlags
dCX_WINDOW =
  unsafePerformIO(
    prim_dCX_WINDOW
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_WINDOW" prim_dCX_WINDOW :: IO (Word32)
dCX_CACHE :: GetDCExFlags
dCX_CACHE =
  unsafePerformIO(
    prim_dCX_CACHE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_CACHE" prim_dCX_CACHE :: IO (Word32)
dCX_CLIPCHILDREN :: GetDCExFlags
dCX_CLIPCHILDREN =
  unsafePerformIO(
    prim_dCX_CLIPCHILDREN
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_CLIPCHILDREN" prim_dCX_CLIPCHILDREN :: IO (Word32)
dCX_CLIPSIBLINGS :: GetDCExFlags
dCX_CLIPSIBLINGS =
  unsafePerformIO(
    prim_dCX_CLIPSIBLINGS
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_CLIPSIBLINGS" prim_dCX_CLIPSIBLINGS :: IO (Word32)
dCX_PARENTCLIP :: GetDCExFlags
dCX_PARENTCLIP =
  unsafePerformIO(
    prim_dCX_PARENTCLIP
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_PARENTCLIP" prim_dCX_PARENTCLIP :: IO (Word32)
dCX_EXCLUDERGN :: GetDCExFlags
dCX_EXCLUDERGN =
  unsafePerformIO(
    prim_dCX_EXCLUDERGN
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_EXCLUDERGN" prim_dCX_EXCLUDERGN :: IO (Word32)
dCX_INTERSECTRGN :: GetDCExFlags
dCX_INTERSECTRGN =
  unsafePerformIO(
    prim_dCX_INTERSECTRGN
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_INTERSECTRGN" prim_dCX_INTERSECTRGN :: IO (Word32)
dCX_LOCKWINDOWUPDATE :: GetDCExFlags
dCX_LOCKWINDOWUPDATE =
  unsafePerformIO(
    prim_dCX_LOCKWINDOWUPDATE
    >>= \  res1  ->
    (return (res1)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_dCX_LOCKWINDOWUPDATE" prim_dCX_LOCKWINDOWUPDATE :: IO (Word32)

-- apparently mostly fails if you use invalid hwnds
getDCEx :: HWND -> HRGN -> GetDCExFlags -> IO HDC
getDCEx arg1 arg2 arg3 =
  prim_getDCEx arg1 arg2 arg3
  >>= \ gc_result ->
  access_prim_getDCEx_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_getDCEx_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getDCEx_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getDCEx" prim_getDCEx :: Addr -> ForeignObj -> Word32 -> IO (Addr)
foreign import ccall unsafe "access_prim_getDCEx_res1" access_prim_getDCEx_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getDCEx_gc_failed" access_prim_getDCEx_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getDCEx_gc_failstring" access_prim_getDCEx_gc_failstring :: Addr -> IO (Addr)

getDC :: MbHWND -> IO HDC
getDC arg1 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_getDC arg1
  >>= \ gc_result ->
  access_prim_getDC_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_getDC_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getDC_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getDC" prim_getDC :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getDC_res1" access_prim_getDC_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getDC_gc_failed" access_prim_getDC_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getDC_gc_failstring" access_prim_getDC_gc_failstring :: Addr -> IO (Addr)

getWindowDC :: MbHWND -> IO HDC
getWindowDC arg1 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_getWindowDC arg1
  >>= \ gc_result ->
  access_prim_getWindowDC_res1 (gc_result :: Addr) >>= \ res1 ->
  access_prim_getWindowDC_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getWindowDC_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (res1))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getWindowDC" prim_getWindowDC :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getWindowDC_res1" access_prim_getWindowDC_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getWindowDC_gc_failed" access_prim_getWindowDC_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getWindowDC_gc_failstring" access_prim_getWindowDC_gc_failstring :: Addr -> IO (Addr)

releaseDC :: MbHWND -> HDC -> IO ()
releaseDC arg1 arg2 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_releaseDC arg1 arg2
  >>= \ gc_result ->
  access_prim_releaseDC_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_releaseDC_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_releaseDC" prim_releaseDC :: Addr -> Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_releaseDC_gc_failed" access_prim_releaseDC_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_releaseDC_gc_failstring" access_prim_releaseDC_gc_failstring :: Addr -> IO (Addr)

getDCOrgEx :: HDC -> IO POINT
getDCOrgEx arg1 =
  prim_getDCOrgEx arg1
  >>= \ gc_result ->
  access_prim_getDCOrgEx_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_getDCOrgEx_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_getDCOrgEx_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getDCOrgEx_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getDCOrgEx" prim_getDCOrgEx :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getDCOrgEx_gc_res1" access_prim_getDCOrgEx_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getDCOrgEx_gc_res2" access_prim_getDCOrgEx_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getDCOrgEx_gc_failed" access_prim_getDCOrgEx_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getDCOrgEx_gc_failstring" access_prim_getDCOrgEx_gc_failstring :: Addr -> IO (Addr)

----------------------------------------------------------------
-- Caret
----------------------------------------------------------------

hideCaret :: HWND -> IO ()
hideCaret arg1 =
  prim_hideCaret arg1
  >>= \ gc_result ->
  access_prim_hideCaret_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_hideCaret_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_hideCaret" prim_hideCaret :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_hideCaret_gc_failed" access_prim_hideCaret_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_hideCaret_gc_failstring" access_prim_hideCaret_gc_failstring :: Addr -> IO (Addr)

showCaret :: HWND -> IO ()
showCaret arg1 =
  prim_showCaret arg1
  >>= \ gc_result ->
  access_prim_showCaret_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_showCaret_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_showCaret" prim_showCaret :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_showCaret_gc_failed" access_prim_showCaret_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_showCaret_gc_failstring" access_prim_showCaret_gc_failstring :: Addr -> IO (Addr)

-- ToDo: allow arg2 to be NULL or {(HBITMAP)1}
createCaret :: HWND -> HBITMAP -> MbINT -> MbINT -> IO ()
createCaret arg1 arg2 arg3 arg4 =
  (case arg3 of {
      Nothing -> (return (0));
      (Just arg3) -> (return ((arg3)))
   }) >>= \ (arg3) ->
  (case arg4 of {
      Nothing -> (return (0));
      (Just arg4) -> (return ((arg4)))
   }) >>= \ (arg4) ->
  prim_createCaret arg1 arg2 arg3 arg4
  >>= \ gc_result ->
  access_prim_createCaret_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_createCaret_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_createCaret" prim_createCaret :: Addr -> Addr -> Int32 -> Int32 -> IO (Addr)
foreign import ccall unsafe "access_prim_createCaret_gc_failed" access_prim_createCaret_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_createCaret_gc_failstring" access_prim_createCaret_gc_failstring :: Addr -> IO (Addr)

destroyCaret :: IO ()
destroyCaret =
  prim_destroyCaret
  >>= \ gc_result ->
  access_prim_destroyCaret_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_destroyCaret_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_destroyCaret" prim_destroyCaret :: IO (Addr)
foreign import ccall unsafe "access_prim_destroyCaret_gc_failed" access_prim_destroyCaret_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_destroyCaret_gc_failstring" access_prim_destroyCaret_gc_failstring :: Addr -> IO (Addr)

getCaretPos :: IO POINT
getCaretPos =
  prim_getCaretPos
  >>= \ gc_result ->
  access_prim_getCaretPos_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_getCaretPos_gc_res2 (gc_result :: Addr) >>= \ gc_res2 ->
  access_prim_getCaretPos_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getCaretPos_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return ((gc_res1,gc_res2)))
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_getCaretPos" prim_getCaretPos :: IO (Addr)
foreign import ccall unsafe "access_prim_getCaretPos_gc_res1" access_prim_getCaretPos_gc_res1 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getCaretPos_gc_res2" access_prim_getCaretPos_gc_res2 :: Addr -> IO (Int32)
foreign import ccall unsafe "access_prim_getCaretPos_gc_failed" access_prim_getCaretPos_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getCaretPos_gc_failstring" access_prim_getCaretPos_gc_failstring :: Addr -> IO (Addr)

setCaretPos :: POINT -> IO ()
setCaretPos gc_arg1 =
  case gc_arg1 of { (gc_arg2,gc_arg3) ->
  prim_setCaretPos gc_arg2 gc_arg3
  >>= \ gc_result ->
  access_prim_setCaretPos_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_setCaretPos_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))}
foreign import  ccall unsafe "Win32Window_stub_ffi.h prim_setCaretPos" prim_setCaretPos :: Int32 -> Int32 -> IO (Addr)
foreign import ccall unsafe "access_prim_setCaretPos_gc_failed" access_prim_setCaretPos_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_setCaretPos_gc_failstring" access_prim_setCaretPos_gc_failstring :: Addr -> IO (Addr)

-- The remarks on SetCaretBlinkTime are either highly risible or very sad -
-- depending on whether you plan to use this function.

----------------------------------------------------------------
-- MSGs and event loops
--
-- Note that the following functions have to be reentrant:
--
--   DispatchMessage
--   SendMessage
--   UpdateWindow   (I think)
--   RedrawWindow   (I think)
--
-- The following dont have to be reentrant (according to documentation)
--
--   GetMessage
--   PeekMessage
--   TranslateMessage
--
-- For Hugs (and possibly NHC too?) this is no big deal.
-- For GHC, you have to use casm_GC instead of casm.
-- (It might be simpler to just put all this code in another
-- file and build it with the appropriate command line option...)
----------------------------------------------------------------

-- type MSG = 
--   ( HWND   -- hwnd;	
--   , UINT   -- message;
--   , WPARAM -- wParam;
--   , LPARAM -- lParam;
--   , DWORD  -- time;
--   , POINT  -- pt;
--   )

type LPMSG   = Addr

-- A NULL window requests messages for any window belonging to this thread.
-- a "success" value of 0 indicates that WM_QUIT was received
getMessage :: MbHWND -> IO LPMSG
getMessage arg1 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_getMessage arg1
  >>= \ gc_result ->
  access_prim_getMessage_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_getMessage_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getMessage_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (gc_res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_getMessage" prim_getMessage :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getMessage_gc_res1" access_prim_getMessage_gc_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getMessage_gc_failed" access_prim_getMessage_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getMessage_gc_failstring" access_prim_getMessage_gc_failstring :: Addr -> IO (Addr)

getMessage2 :: MbHWND -> IO (LPMSG,Bool)
getMessage2 arg1 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_getMessage2 arg1
  >>= \ gc_result ->
  access_prim_getMessage2_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_getMessage2_gc_res3 (gc_result :: Addr) >>= \ gc_res3 ->
  access_prim_getMessage2_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_getMessage2_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (unmarshall_bool_ gc_res3) >>= \ gc_res2 ->
       (return ((gc_res1,gc_res2)))
foreign import  ccall "Win32Window_stub_ffi.h prim_getMessage2" prim_getMessage2 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getMessage2_gc_res1" access_prim_getMessage2_gc_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_getMessage2_gc_res3" access_prim_getMessage2_gc_res3 :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getMessage2_gc_failed" access_prim_getMessage2_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_getMessage2_gc_failstring" access_prim_getMessage2_gc_failstring :: Addr -> IO (Addr)

-- A NULL window requests messages for any window belonging to this thread.
-- Arguably the code block shouldn't be a 'safe' one, but it shouldn't really
-- hurt.
peekMessage :: MbHWND -> UINT -> UINT -> UINT -> IO LPMSG
peekMessage arg1 arg2 arg3 arg4 =
  (case arg1 of {
      Nothing -> (return (nullHANDLE));
      (Just arg1) -> (return ((arg1)))
   }) >>= \ (arg1) ->
  prim_peekMessage arg1 arg2 arg3 arg4
  >>= \ gc_result ->
  access_prim_peekMessage_gc_res1 (gc_result :: Addr) >>= \ gc_res1 ->
  access_prim_peekMessage_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_peekMessage_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (gc_res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_peekMessage" prim_peekMessage :: Addr -> Word32 -> Word32 -> Word32 -> IO (Addr)
foreign import ccall unsafe "access_prim_peekMessage_gc_res1" access_prim_peekMessage_gc_res1 :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_peekMessage_gc_failed" access_prim_peekMessage_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_peekMessage_gc_failstring" access_prim_peekMessage_gc_failstring :: Addr -> IO (Addr)

-- Note: youre not supposed to call this if youre using accelerators
translateMessage :: LPMSG -> IO BOOL
translateMessage arg1 =
  prim_translateMessage arg1
  >>= \  res1  ->
  (unmarshall_bool_ res1) >>= \ gc_res1 ->
  (return (gc_res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_translateMessage" prim_translateMessage :: Addr -> IO (Int)

updateWindow :: HWND -> IO ()
updateWindow arg1 =
  prim_updateWindow arg1
  >>= \ gc_result ->
  access_prim_updateWindow_gc_failed (gc_result :: Addr) >>= \ gc_failed ->
  access_prim_updateWindow_gc_failstring (gc_result :: Addr) >>= \ gc_failstring ->
  if ( gc_failed /= (0::Int))
  then unmarshall_string_ gc_failstring >>=  ioError  . userError
  else (return (()))
foreign import  ccall "Win32Window_stub_ffi.h prim_updateWindow" prim_updateWindow :: Addr -> IO (Addr)
foreign import ccall unsafe "access_prim_updateWindow_gc_failed" access_prim_updateWindow_gc_failed :: Addr -> IO (Int)
foreign import ccall unsafe "access_prim_updateWindow_gc_failstring" access_prim_updateWindow_gc_failstring :: Addr -> IO (Addr)

-- Return value of DispatchMessage is usually ignored
dispatchMessage :: LPMSG -> IO LONG
dispatchMessage arg1 =
  prim_dispatchMessage arg1
  >>= \  res1  ->
  (return (res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_dispatchMessage" prim_dispatchMessage :: Addr -> IO (Int32)

sendMessage :: HWND -> WindowMessage -> WPARAM -> LPARAM -> IO LRESULT
sendMessage arg1 arg2 arg3 arg4 =
  prim_sendMessage arg1 arg2 arg3 arg4
  >>= \  res1  ->
  (return (res1))
foreign import  ccall "Win32Window_stub_ffi.h prim_sendMessage" prim_sendMessage :: Addr -> Word32 -> Word32 -> Int32 -> IO (Int32)

----------------------------------------------------------------

-- ToDo: figure out reentrancy stuff
-- ToDo: catch error codes
--
-- ToDo: how to send HWND_BROADCAST to PostMessage
-- %fun PostMessage       :: MbHWND -> WindowMessage -> WPARAM -> LPARAM -> IO ()
-- %fun PostQuitMessage   :: Int -> IO ()
-- %fun PostThreadMessage :: DWORD -> WindowMessage -> WPARAM -> LPARAM -> IO ()
-- %fun InSendMessage     :: IO Bool

----------------------------------------------------------------
-- End
----------------------------------------------------------------
