module CleanStdList where


--	********************************************************************************
--	Clean to Haskell Standard Env library.
--	
--	CleanStdList defines all functions on lists in Clean StdList (StdEnv) that are
--	not member of the Haskell Prelude. 
--	********************************************************************************

import CleanExtras

{-	In Clean, % is an overloaded operator used to slice lists and arrays. 
	For this version it has been specialised to lists only.
-}
infixl 9 %
(%) :: [a] -> (Int,Int) -> [a]
(%) list (frm,to) = take (to - frm + 1) (drop frm list)

	
-- ****************************************************************************************
-- standard operators
-- ****************************************************************************************

flatten :: [[a]] -> [a]
flatten (h:t) = h ++ flatten t
flatten []    = []

isEmpty :: [a] -> Bool	-- isEmpty = null
isEmpty [] = True
isEmpty _  = False

-- ****************************************************************************************
-- standard functions
-- ****************************************************************************************

hd :: [a] -> a		-- hd = head
hd (a:_) = a
hd []    = error "hd of []"

tl :: [a] -> [a]	-- tl = tail
tl (_:x) = x
tl []    = error "tl of []"

insert :: (a -> a -> Bool) -> a -> [a] -> [a]
insert r x ls@(y : ys)
	| r x y     = x : ls
	| otherwise = y : insert r x ys
insert _ x []       = [x]

insertAt :: Int -> a -> [a] -> [a]
insertAt 0 x ys      = x:ys
insertAt _ x []      = [x]
insertAt n x (y:ys)  = y : insertAt (n-1) x ys

removeAt :: Int -> [a] -> [a]
removeAt 0 (y : ys) = ys
removeAt n (y : ys) = y : removeAt (n-1) ys
removeAt n []       = []

updateAt :: Int -> a  -> [a] -> [a]
updateAt 0 x []     = []
updateAt 0 x (y:ys) = x:ys
updateAt _ x []     = []
updateAt n x (y:ys) = y : updateAt (n-1) x ys

indexList :: [a] -> [Int]
indexList x
	= f 0 x
	where
		f :: Int -> [a] -> [Int]
		f n (a:x) = n:f (n+1) x
		f n []    = []

repeatn :: Int -> a -> [a]
repeatn n x = take n (repeat x)

scan :: (a -> b -> a) -> a -> [b] -> [a]
scan op r (a:x) = r:scan op (op r a) x
scan op r []    = [r]


isMember :: (Eq a) => a -> [a] -> Bool
isMember x (hd:tl) = hd==x || isMember x tl
isMember x []      = False

isAnyMember :: (Eq a) => [a] -> [a] -> Bool		-- Is one of arg1 an element arg2
isAnyMember (x:xs) list = isMember x list || isAnyMember xs list
isAnyMember []     list = False

removeDup :: (Eq a) => [a] -> [a]
removeDup (x:xs) = x:removeDup (filter ((/=) x) xs)
removeDup _      = []

removeMember :: (Eq a) => a -> [a] -> [a]
removeMember e (a:as)
	| a==e      = as
	| otherwise = a:removeMember e as
removeMember e []
	= []

removeMembers :: (Eq a) => [a] -> [a] -> [a]
removeMembers x []    = x
removeMembers x (b:y) = removeMembers (removeMember b x) y

removeIndex :: (Eq a) => a -> [a] -> (Int,[a])
removeIndex e xs
	= removei e xs 0
	where
		removei :: (Eq a) => a -> [a] -> Int -> (Int,[a])
		removei e (x:xs) i
			| x==e
				= (i,xs)
			| otherwise
				= (j,x:res)
				where
					(j,res) = removei e xs (i+1)
		removei e [] i
			= error "Error in removeIndex: element not found"

limit :: (Eq a) => [a] -> a
limit (a:cons@(b:x))
	| a==b      = a
	| otherwise = limit cons
limit other
	= error "incorrect use of limit"


{-	The following list operations have obtained slightly different class restrictions
	because these are different in Haskell.
-}

prod :: (Num a,One a) => [a] -> a		-- uses only one and (*)
prod xs
	= accprod one xs
	where
		accprod n (x:xs) = accprod (n*x) xs
		accprod n []     = n

avg :: (Zero a,One a,Fractional a) => [a] -> a	-- uses only zero, one, (+), (/)
avg []
	= error "avg called with empty list"
avg xs
	= accavg zero zero xs
	where
		accavg n nelem (x:xs) = accavg (n+x) (nelem+one) xs
		accavg n nelem []     = n / nelem
