% -----------------------------------------------------------------------------
% $Id: Exception.lhs,v 1.20 2001/04/26 14:19:47 simonmar Exp $
%
% (c) The GRAP/AQUA Project, Glasgow University, 1998
%

The External API for exceptions.  The functions provided in this
module allow catching of exceptions in the IO monad.

\begin{code}
module Exception (
	Exception(..),		-- instance Eq, Ord, Show, Typeable
	IOException,		-- instance Eq, Ord, Show, Typeable
	ArithException(..),	-- instance Eq, Ord, Show, Typeable
	ArrayException(..),	-- instance Eq, Ord, Show, Typeable
	AsyncException(..),	-- instance Eq, Ord, Show, Typeable

	try,       -- :: IO a -> IO (Either Exception a)
	tryJust,   -- :: (Exception -> Maybe b) -> a    -> IO (Either b a)

	catch,     -- :: IO a -> (Exception -> IO a) -> IO a
	catchJust, -- :: (Exception -> Maybe b) -> IO a -> (b -> IO a) -> IO a

	evaluate,  -- :: a -> IO a

	-- Exception predicates (for catchJust, tryJust)

	ioErrors,		-- :: Exception -> Maybe IOError
	arithExceptions, 	-- :: Exception -> Maybe ArithException
	errorCalls,		-- :: Exception -> Maybe String
	dynExceptions,		-- :: Exception -> Maybe Dynamic
	assertions,		-- :: Exception -> Maybe String
	asyncExceptions, 	-- :: Exception -> Maybe AsyncException
	userErrors,		-- :: Exception -> Maybe String

	-- Throwing exceptions

	throw,		-- :: Exception -> a
#ifndef __STGHUGS__
	-- for now
	throwTo,	-- :: ThreadId -> Exception -> a
#endif

	-- Dynamic exceptions

	throwDyn, 	-- :: Typeable ex => ex -> b
	throwDynTo, 	-- :: Typeable ex => ThreadId -> ex -> b
	catchDyn, 	-- :: Typeable ex => IO a -> (ex -> IO a) -> IO a
	
	-- Async exception control

        block,          -- :: IO a -> IO a
        unblock,        -- :: IO a -> IO a

	-- Assertions

	-- for now
	assert,		-- :: Bool -> a -> a

	-- Utilities

	finally, 	-- :: IO a -> IO b -> IO b

	bracket,  	-- :: IO a -> (a -> IO b) -> (a -> IO c) -> IO ()
	bracket_, 	-- :: IO a -> IO b -> IO c -> IO ()

	-- DEPRECATED (all deprecated in 5.00, remove in 5.02)

	tryAll,      		-- :: a    -> IO (Either Exception a)
	tryAllIO,    		-- :: IO a -> IO (Either Exception a)

	catchAllIO,  		-- :: IO a -> (Exception -> IO a) -> IO a
	catchAll,    		-- :: a    -> (Exception -> IO a) -> IO a

	raiseInThread,		-- :: ThreadId -> Exception -> IO ()

        blockAsyncExceptions,   -- :: IO a -> IO a
        unblockAsyncExceptions, -- :: IO a -> IO a
  ) where

#ifndef __STGHUGS__
import Prelude 		hiding (catch)
import PrelGHC 		( assert )
import PrelException 	hiding (try, catch, bracket, bracket_)
import PrelConc		( throwTo, ThreadId )
import PrelIOBase	( IO(..) )
#else
import Prelude hiding ( catch )
import PrelPrim	( catchException 
		, Exception(..)
		, throw
		, ArithException(..)
		, AsyncException(..)
		, assert
		)
#endif

import Dynamic	( Dynamic, toDyn, fromDynamic, Typeable(..)
		, TyCon, mkTyCon, mkAppTy
		)
\end{code}

-----------------------------------------------------------------------------
Catching exceptions

PrelException defines 'catchException' for us.

\begin{code}
catch :: IO a -> (Exception -> IO a) -> IO a
catch =  catchException

catchJust :: (Exception -> Maybe b) -> IO a -> (b -> IO a) -> IO a
catchJust p a handler = catch a handler'
  where handler' e = case p e of 
			Nothing -> throw e
			Just b  -> handler b

{-# DEPRECATED catchAllIO "use catch instead" #-}
catchAllIO :: IO a -> (Exception -> IO a) -> IO a
catchAllIO = catch
{-# DEPRECATED catchAll "use (catch . evaluate) instead" #-}
catchAll :: a -> (Exception -> IO a) -> IO a
catchAll e h = catch (evaluate e) h
\end{code}

-----------------------------------------------------------------------------
evaluate

\begin{code}
evaluate :: a -> IO a
evaluate a = a `seq` return a
\end{code}

-----------------------------------------------------------------------------
throwTo

\begin{code}
{-# DEPRECATED raiseInThread "use throwTo instead" #-}
raiseInThread t e = throwTo t e
\end{code}

-----------------------------------------------------------------------------
'try' and variations.

\begin{code}
try :: IO a -> IO (Either Exception a)
try a = catch (a >>= \ v -> return (Right v)) (\e -> return (Left e))

tryJust :: (Exception -> Maybe b) -> IO a -> IO (Either b a)
tryJust p a = do
  r <- tryAllIO a
  case r of
	Right v -> return (Right v)
	Left  e -> case p e of
			Nothing -> throw e
			Just b  -> return (Left b)

{-# DEPRECATED tryAll "use (try . evaluate) instead" #-}
tryAll :: a -> IO (Either Exception a)
tryAll a = try (evaluate a)

{-# DEPRECATED tryAllIO "use try instead" #-}
tryAllIO :: IO a -> IO (Either Exception a)
tryAllIO = try
\end{code}

-----------------------------------------------------------------------------
Dynamic exception types.  Since one of the possible kinds of exception
is a dynamically typed value, we can effectively have polymorphic
exceptions.

throwDyn will raise any value as an exception, provided it is in the
Typeable class (see Dynamic.lhs).  

catchDyn will catch any exception of a given type (determined by the
handler function).  Any raised exceptions that don't match are
re-raised.

\begin{code}
throwDyn :: Typeable exception => exception -> b
throwDyn exception = throw (DynException (toDyn exception))

throwDynTo :: Typeable exception => ThreadId -> exception -> IO ()
throwDynTo t exception = throwTo t (DynException (toDyn exception))

catchDyn :: Typeable exception => IO a -> (exception -> IO a) -> IO a
catchDyn m k = catchException m handle
  where handle ex = case ex of
  			   (DynException dyn) ->
		  	  	case fromDynamic dyn of
				    Just exception  -> k exception
				    Nothing -> throw ex
			   _ -> throw ex
\end{code}

-----------------------------------------------------------------------------
Exception Predicates

\begin{code}
ioErrors		:: Exception -> Maybe IOError
arithExceptions 	:: Exception -> Maybe ArithException
errorCalls		:: Exception -> Maybe String
dynExceptions		:: Exception -> Maybe Dynamic
assertions		:: Exception -> Maybe String
asyncExceptions 	:: Exception -> Maybe AsyncException
userErrors		:: Exception -> Maybe String

ioErrors e@(IOException _) = Just e
ioErrors _ = Nothing

arithExceptions (ArithException e) = Just e
arithExceptions _ = Nothing

errorCalls (ErrorCall e) = Just e
errorCalls _ = Nothing

assertions (AssertionFailed e) = Just e
assertions _ = Nothing

dynExceptions (DynException e) = Just e
dynExceptions _ = Nothing

asyncExceptions (AsyncException e) = Just e
asyncExceptions _ = Nothing

userErrors (UserError e) = Just e
userErrors _ = Nothing
\end{code}

-----------------------------------------------------------------------------
Async Exception control

\begin{code}
{-# DEPRECATED blockAsyncExceptions "renamed to block" #-}
blockAsyncExceptions   = block
{-# DEPRECATED unblockAsyncExceptions "renamed to unblock" #-}
unblockAsyncExceptions = unblock
\end{code}

-----------------------------------------------------------------------------
Some Useful Functions

\begin{code}
bracket :: IO a -> (a -> IO b) -> (a -> IO c) -> IO c
bracket before after thing =
  block (do
    a <- before 
    r <- catch 
	   (unblock (thing a))
	   (\e -> do { after a; throw e })
    after a
    return r
 )
   
-- finally is an instance of bracket, but it's quite common
-- so we give the specialised version for efficiency.
finally :: IO a -> IO b -> IO a
a `finally` sequel =
  block (do
    r <- catch 
	     (unblock a)
	     (\e -> do { sequel; throw e })
    sequel
    return r
  )

bracket_ :: IO a -> IO b -> IO c -> IO c
bracket_ before after thing = bracket before (const after) (const thing)

#ifdef __STGHUGS__
-- For Hugs, we do not implemented blocking (yet!)
block :: IO a -> IO a
block a = a
unblock :: IO a -> IO a
unblock a = a
#endif

\end{code}

-----------------------------------------------------------------------------
Typeable instances

\begin{code}
exceptionTc :: TyCon
exceptionTc = mkTyCon "Exception"

instance Typeable Exception where
  typeOf _ = mkAppTy exceptionTc []

iOExceptionTc :: TyCon
iOExceptionTc = mkTyCon "IOException"

instance Typeable IOException where
  typeOf _ = mkAppTy iOExceptionTc []

arithExceptionTc :: TyCon
arithExceptionTc = mkTyCon "ArithException"

instance Typeable ArithException where
  typeOf _ = mkAppTy arithExceptionTc []

arrayExceptionTc :: TyCon
arrayExceptionTc = mkTyCon "ArrayException"

instance Typeable ArrayException where
  typeOf _ = mkAppTy arrayExceptionTc []

asyncExceptionTc :: TyCon
asyncExceptionTc = mkTyCon "AsyncException"

instance Typeable AsyncException where
  typeOf _ = mkAppTy asyncExceptionTc []
\end{code}
