{-
   AAPoly.hs  (adapted from aapoly.c which is (c) Silicon Graphics, Inc)
   This file is part of HOpenGL - a binding of OpenGL and GLUT for Haskell.
   Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

   This program draws filled polygons with antialiased
   edges.  The special GL_SRC_ALPHA_SATURATE blending 
   function is used.
   Pressing the 't' key turns the antialiasing on and off.
-}

import Char     ( toLower )
import Foreign  ( withArray )
import IORef    ( IORef, newIORef, readIORef, modifyIORef )
import List     ( genericLength )
import System   ( ExitCode(..), exitWith )

import GL
import GLU
import GLUT

myInit :: IO ()
myInit = do
   cullFace Back
   enable CullFace
   blendFunc SrcAlphaSaturate One
   clearColor (Color4 0 0 0 0)

drawCube :: Vertex3 GLfloat -> Vertex3 GLfloat -> IO ()
drawCube (Vertex3 x0 y0 z0) (Vertex3 x1 y1 z1) = do
   enableClientState VertexArray
   enableClientState ColorArray
   withArray [
      Vertex3 x0 y0 z0, Vertex3 x1 y0 z0,
      Vertex3 x1 y1 z0, Vertex3 x0 y1 z0,
      Vertex3 x0 y0 z1, Vertex3 x1 y0 z1,
      Vertex3 x1 y1 z1, Vertex3 x0 y1 z1 ] $ \vertexBuffer -> do
   vertexPointer 3 Float 0 vertexBuffer
   withArray  [
      Color4 0 0 0 1, Color4 1 0 0 1,
      Color4 0 1 0 1, Color4 1 1 0 1,
      Color4 0 0 1 1, Color4 1 0 1 1,
      Color4 0 1 1 1, Color4 1 1 1 (1 :: GLfloat)] $ \colorBuffer -> do
   colorPointer 4 Float 0 colorBuffer
   -- indices of front, top, left, bottom, right, back faces
   let indices = [ 4, 5, 6, 7,
                   2, 3, 7, 6,
                   0, 4, 7, 3,
                   0, 1, 5, 4,
                   1, 5, 6, 2,
                   0, 3, 2, (1 :: GLubyte)]
   withArray indices $ drawElements Quads (genericLength indices) UnsignedByte
   disableClientState VertexArray
   disableClientState ColorArray

-- Note: Polygons must be drawn from front to back
-- for proper blending.
display :: IORef Bool -> DisplayAction
display polySmooth = do
   smooth <- readIORef polySmooth
   if smooth
      then do clear [ ColorBufferBit ]
              enable Blend'
              enable PolygonSmooth
              disable DepthTest
      else do clear [ ColorBufferBit, DepthBufferBit ]
              disable Blend'
              disable PolygonSmooth
              enable DepthTest
   pushMatrix
   translate (Vector3 0 0 (-8 :: GLfloat))
   rotate (30 :: GLfloat) (Vector3 1 0 0)
   rotate (60 :: GLfloat) (Vector3 0 1 0)
   drawCube (Vertex3 (-0.5) (-0.5) (-0.5)) (Vertex3 0.5 0.5 0.5)
   popMatrix

   flush

reshape :: ReshapeAction
reshape screenSize@(WindowSize w h) = do
   viewport (Viewport (WindowPosition 0 0) screenSize)
   matrixMode Projection
   loadIdentity
   perspective 30 (fromIntegral w / fromIntegral h) 1 20
   matrixMode Modelview
   loadIdentity

keyboard :: IORef Bool -> KeyboardAction
keyboard polySmooth c _ = case toLower c of
   't'   -> do modifyIORef polySmooth not; postRedisplay
   '\27' -> exitWith ExitSuccess
   _     -> return ()

--  Main Loop
main :: IO ()
main = do
   (progName, _args) <- GLUT.init Nothing
   polySmooth <- newIORef True
   createWindow progName (display polySmooth) [ Single, GLUT.Rgb, GLUT.Alpha, GLUT.Depth ]
                Nothing (Just (WindowSize 200 200))
   myInit
   reshapeFunc (Just reshape)
   keyboardFunc (Just (keyboard polySmooth))
   mainLoop
