{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 5.3 (Feedback) of the OpenGL 1.2.1 specs.
-}

module GL_Feedback (
   FeedbackType(..),
   unmarshalFeedbackType,   -- internal use only
   FeedbackBuffer(..),
   newFeedbackBuffer, freeFeedbackBuffer, feedbackBuffer, withFeedbackBuffer,
   FeedbackColor, FeedbackInfo(..), FeedbackToken(..), getFeedbackTokens,
   PassThroughValue(..), passThrough
) where

import Foreign          (Ptr, plusPtr, Storable(..), alloca, allocaArray,
                         mallocArray, free)
import Monad            (liftM, liftM2, liftM3, liftM4)

import GL_Constants     (gl_2D, gl_3D, gl_3D_COLOR, gl_3D_COLOR_TEXTURE,
                         gl_4D_COLOR_TEXTURE, gl_POINT_TOKEN, gl_LINE_TOKEN,
			 gl_LINE_RESET_TOKEN, gl_POLYGON_TOKEN, gl_BITMAP_TOKEN,
			 gl_DRAW_PIXEL_TOKEN, gl_COPY_PIXEL_TOKEN,
			 gl_PASS_THROUGH_TOKEN, gl_INDEX_MODE)
import GL_BasicTypes    (GLboolean_, unmarshalGLboolean,
                         GLenum, GLsizei, GLint, GLfloat)
import GL_Marshal       (Marshal, evalMarshal, peekMarshal, eqMarshal)
import GL_VertexSpec    (Vertex2(..), Vertex3(..), Vertex4(..), TexCoord4(..),
                         Color4(..), ColorIndex(..))

---------------------------------------------------------------------------

data FeedbackType =
     TwoD'
   | ThreeD'
   | ThreeDColor'
   | ThreeDColorTexture'
   | FourDColorTexture'
   deriving (Eq, Ord)

marshalFeedbackType :: FeedbackType -> GLenum
marshalFeedbackType TwoD'               = gl_2D
marshalFeedbackType ThreeD'             = gl_3D
marshalFeedbackType ThreeDColor'        = gl_3D_COLOR
marshalFeedbackType ThreeDColorTexture' = gl_3D_COLOR_TEXTURE
marshalFeedbackType FourDColorTexture'  = gl_4D_COLOR_TEXTURE

unmarshalFeedbackType :: GLenum -> FeedbackType
unmarshalFeedbackType t
   | t == gl_2D               = TwoD'
   | t == gl_3D               = ThreeD'
   | t == gl_3D_COLOR         = ThreeDColor'
   | t == gl_3D_COLOR_TEXTURE = ThreeDColorTexture'
   | t == gl_4D_COLOR_TEXTURE = FourDColorTexture'
   | otherwise                = error "unmarshalFeedbackType"

data FeedbackBuffer =
   FeedbackBuffer GLsizei         -- size of buffer in GLfloat units
                  FeedbackType    -- kind of info to be collected
                  (Ptr GLfloat)   -- the malloced buffer itself

newFeedbackBuffer :: GLsizei -> FeedbackType -> IO FeedbackBuffer
newFeedbackBuffer n t = liftM (FeedbackBuffer n t) $ mallocArray (fromIntegral n)

freeFeedbackBuffer :: FeedbackBuffer -> IO ()
freeFeedbackBuffer (FeedbackBuffer _ _ buf) = free buf

feedbackBuffer :: FeedbackBuffer -> IO ()
feedbackBuffer (FeedbackBuffer n t buf) = glFeedbackBuffer n (marshalFeedbackType t) buf

foreign import "glFeedbackBuffer" unsafe glFeedbackBuffer :: GLsizei -> GLenum -> Ptr GLfloat -> IO ()

-- the recommended way...
withFeedbackBuffer :: Int -> FeedbackType -> IO a -> IO a
withFeedbackBuffer n t cont =
   allocaArray n $ \buf -> do
   glFeedbackBuffer (fromIntegral n) (marshalFeedbackType t) buf
   cont

peekF :: Marshal GLfloat
peekF = peekMarshal

type FeedbackColor = Either (ColorIndex GLint) (Color4 GLfloat)

calcColorMarshal :: Bool -> Marshal FeedbackColor
calcColorMarshal False = fmap Right $ liftM4 Color4 peekF peekF peekF peekF
calcColorMarshal True  = fmap Left  $ liftM (ColorIndex . round) peekF

data FeedbackInfo =
     TwoD               (Vertex2 GLfloat)
   | ThreeD             (Vertex3 GLfloat)
   | ThreeDColor        (Vertex3 GLfloat) FeedbackColor
   | ThreeDColorTexture (Vertex3 GLfloat) FeedbackColor (TexCoord4 GLfloat)
   | FourDColorTexture  (Vertex4 GLfloat) FeedbackColor (TexCoord4 GLfloat)
   deriving (Eq,Ord)

calcInfoMarshal :: FeedbackType -> Marshal FeedbackColor -> Marshal FeedbackInfo
calcInfoMarshal TwoD' _ = do
   v <- liftM2 Vertex2 peekF peekF
   return $ TwoD v
calcInfoMarshal ThreeD' _ = do
   v <- liftM3 Vertex3 peekF peekF peekF
   return $ ThreeD v
calcInfoMarshal ThreeDColor' colorMarshal = do
   v <- liftM3 Vertex3 peekF peekF peekF
   c <- colorMarshal
   return $ ThreeDColor v c
calcInfoMarshal ThreeDColorTexture' colorMarshal = do
   v <- liftM3 Vertex3 peekF peekF peekF
   c <- colorMarshal
   t <- liftM4 TexCoord4 peekF peekF peekF peekF
   return $ ThreeDColorTexture v c t
calcInfoMarshal FourDColorTexture' colorMarshal = do
   v <- liftM4 Vertex4 peekF peekF peekF peekF
   c <- colorMarshal
   t <- liftM4 TexCoord4 peekF peekF peekF peekF
   return $ FourDColorTexture v c t

data TokenTag =
     PointTag
   | LineTag
   | LineResetTag
   | PolygonTag
   | BitmapTag
   | DrawPixelTag
   | CopyPixelTag
   | PassThroughTag

unmarshalTokenTag :: GLenum -> TokenTag
unmarshalTokenTag t
   | t == gl_POINT_TOKEN        = PointTag
   | t == gl_LINE_TOKEN         = LineTag
   | t == gl_LINE_RESET_TOKEN   = LineResetTag
   | t == gl_POLYGON_TOKEN      = PolygonTag
   | t == gl_BITMAP_TOKEN       = BitmapTag
   | t == gl_DRAW_PIXEL_TOKEN   = DrawPixelTag
   | t == gl_COPY_PIXEL_TOKEN   = CopyPixelTag
   | t == gl_PASS_THROUGH_TOKEN = PassThroughTag
   | otherwise                  = error "unmarshalTokenTag"

data FeedbackToken =
     PointToken FeedbackInfo
   | LineToken FeedbackInfo FeedbackInfo
   | LineResetToken FeedbackInfo FeedbackInfo
   | PolygonToken [FeedbackInfo]
   | BitmapToken FeedbackInfo
   | DrawPixelToken FeedbackInfo
   | CopyPixelToken FeedbackInfo
   | PassThroughToken PassThroughValue
   deriving (Eq,Ord)

tokenMarshal :: Marshal FeedbackInfo -> Marshal FeedbackToken
tokenMarshal marshalInfo = do
   tag <- peekF
   case unmarshalTokenTag (round tag) of
      PointTag -> liftM PointToken marshalInfo
      LineTag -> liftM2 LineToken marshalInfo marshalInfo
      LineResetTag -> liftM2 LineResetToken marshalInfo marshalInfo
      PolygonTag -> do
         n <- peekF
         liftM PolygonToken $ sequence (replicate (round n) marshalInfo)
      BitmapTag -> liftM BitmapToken marshalInfo
      DrawPixelTag -> liftM DrawPixelToken marshalInfo
      CopyPixelTag -> liftM CopyPixelToken marshalInfo
      PassThroughTag -> liftM (PassThroughToken . PassThroughValue) peekF

-- NOTE: size is the number of GLfloats in buffer, not the number of tokens!
getFeedbackTokens :: GLint -> FeedbackBuffer -> IO (Maybe [FeedbackToken])
getFeedbackTokens size (FeedbackBuffer _ t buf)
   | size < 0  = return Nothing
   | otherwise = do 
      isIndexMode <- alloca $ \booleanBuf -> do
                     glGetBooleanv gl_INDEX_MODE booleanBuf
                     liftM unmarshalGLboolean $ peek booleanBuf
      let endPtr       = buf `plusPtr` (sizeOf (undefined :: GLfloat) * fromIntegral size)
          colorMarshal = calcColorMarshal isIndexMode
          infoMarshal  = calcInfoMarshal t colorMarshal
          loop tokens  = do
             done <- eqMarshal endPtr
             if done
                then return (reverse tokens)
                else do token <- tokenMarshal infoMarshal
                        loop (token:tokens)
      tokens <- evalMarshal (loop []) buf
      return $ Just tokens

foreign import "glGetBooleanv" unsafe glGetBooleanv :: GLenum -> Ptr GLboolean_ -> IO ()

newtype PassThroughValue = PassThroughValue GLfloat deriving (Eq,Ord)

foreign import "glPassThrough" unsafe passThrough :: PassThroughValue -> IO ()
