
/*
  
   1998 Roberto Alameda <roberto@myokay.net>
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 
  

*/ 


#include <config.h>

#if HAVE_UNISTD_H
# include <sys/types.h>
# include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <limits.h>
#include <errno.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "gfont.h"


// *** Imported variables
extern TT_Engine ttengine;

extern GdkVisual *visual;
extern GdkGC *bwGC, *defGC;
extern GtkStyle *stylebold;
extern GdkPixmap *emptypixmap;

extern GtkWidget *mainwindow;
extern GtkWidget *samplet, *stbar, *prbar;

extern bool kerning, antialiasing;
extern int xresolution, yresolution;

extern GdkColor graypalette[];



// *** Module variables
static const char *weight_table[] = {
    "Thin", "Extra Light", "Light", "Regular", "Medium",
    "Semibold", "Bold", "Extra Bold", "Black"
};

static const char *width_table[] = {
  "Ultra Condensed", "Extra Condensed", "Condensed", 
  "Semi Condensed", "Medium", "Semi Expanded", 
  "Expanded", "Extra Expanded", "Ultra Expanded"
};


// Structure associated to a font table window
struct WInfo {
  FontData *fd;
  TT_Instance instance;
  double fontsize;
  int cellwidth, cellheight;
  GSList *pixmaplist;
  GtkWidget *notebook, *drawing_area, *info_label;
  GtkItemFactory *item_factory;
  guint antialiasing:1;  // Whether the fonts in this window are smoothed
  guint hinting:1;
};


// TT_Table contains data used when downloading as Type42 font
struct TT_Table {
  TT_ULong tag;
  TT_ULong checksum;
  TT_ULong offset;
  TT_ULong length;
  const char *name;
  guchar *data;  // The table itself
};


// TTInfo contains data used in the properties window
struct TT_Info {
  GtkWidget *property, *value;
};


// ************** Function prototypes (internally used in this module)
static Callback destroy_fonttable;
static GdkImage* gdk_image_subimage(GdkImage* origimage, gint x, gint y, 
				    gint w, gint h);
static TT_Error findcharmap(FontData* fd, TT_CharMap* charmap);
static TT_Error findtextcharmap(FontData* fd, TT_CharMap* charmap);
static GdkImage* tt_makechar(FontData* fd, TT_UShort index, double size, 
			     bool antialiasing);
static GdkImage* tt_makestring(FontData* fd, const char* string, double size, 
			       bool kerning, bool antialiasing);
static gint mouse_event(GtkWidget *widget, GdkEvent *event, gpointer);
static TT_FWord getkerning(FontData* fd, TT_UShort ch1, TT_UShort ch2);
static void fill_pixmap(WInfo *winfo, int page);

static bool get_table_list(FontData *fd);
static void free_table_list(void);
static void write_string(FILE *ofp, bool pad, TT_Table *tb, 
			 TT_ULong offset, TT_ULong length);
static TT_ULong get_glyph_offset(TT_Table *loca, int ch, int Index_To_Loc_Format);
static void make_offset_table(FontData *fd, char *ot, TT_UShort numTables);

static void make_entries(GtkWidget *frame, struct TT_Info *props, int num);
static GtkWidget* general_info(FontData *fd);
static GtkWidget* specific_info(FontData *fd);





// ********************* Begin of program functions

static GdkImage* gdk_image_subimage(GdkImage* origimage, gint x, gint y, 
				    gint w, gint h)
{
  GdkImage *image;
  GdkImagePrivate *priv, *privorig;
  
  g_return_val_if_fail(origimage != NULL, 0);
  
  priv = g_new(GdkImagePrivate, 1);
  privorig = (GdkImagePrivate*)origimage;
  
  image = (GdkImage *)priv;
  priv->xdisplay = privorig->xdisplay;
  priv->image_put = privorig->image_put;
  image->type = origimage->type;
  image->visual = origimage->visual;
  image->width = w;
  image->height = h;
  image->depth = origimage->depth;
  priv->ximage = XSubImage(privorig->ximage, x, y, w, h);
  priv->ximage->bitmap_bit_order = privorig->ximage->bitmap_bit_order;
  priv->ximage->byte_order = privorig->ximage->byte_order;
  image->byte_order = origimage->byte_order;
  image->mem = priv->ximage->data;
  image->bpl = priv->ximage->bytes_per_line;
  image->bpp = origimage->bpp;
  return image;
}


/*
  Return a string with the requested (idx) name of the font
  It tries to find a Unicode encoded english string of the given type
  idx can be TT_NAME_ID_COPYRIGHT, TT_NAME_ID_FONT_FAMILY, TT_NAME_ID_FONT_SUBFAMILY,
  TT_NAME_ID_UNIQUE_ID, TT_NAME_ID_FULL_NAME, TT_NAME_ID_VERSION_STRING, 
  TT_NAME_ID_PS_NAME, TT_NAME_ID_TRADEMARK
  OpenType names: 
  TT_NAME_ID_MANUFACTURER, TT_NAME_ID_DESIGNER, TT_NAME_ID_DESCRIPTION,
  TT_NAME_ID_VENDOR_URL, TT_NAME_ID_DESIGNER_URL, TT_NAME_ID_LICENSE, 
  TT_NAME_ID_LICENSE_URL
  More: TT_NAME_ID_PREFERRED_FAMILY, TT_NAME_ID_PREFERRED_SUBFAMILY, 
  TT_NAME_ID_MAC_FULL_NAME
*/
const char* getfontname(FontData* fd, int idx)
{
  TT_UShort len;
  char* string;
  TT_UShort platform, encoding, language, id;
  static char *fontName=NULL;
  static int fontNameLen = 256;

  if (fontName == NULL) fontName = (char*)malloc(fontNameLen);
  if (fontName == NULL) {
    fprintf(stderr, "Unable to reserve %d bytes of memory; exiting ...", fontNameLen);
    exit(1);
  }

  TT_Face face = fd->ttdata.face;
  int found = 0;
  int i;
  for (i=0; i<fd->ttdata.properties.num_Names; i++) {
    int error = TT_Get_Name_ID(face, i, &platform, &encoding, &language, &id);
    if (error || id!=idx) continue;
    error = TT_Get_Name_String(face, i, &string, &len);
    if (error) continue;
    if ((platform == TT_PLATFORM_MICROSOFT) && 
	(encoding==TT_MS_ID_SYMBOL_CS||encoding==TT_MS_ID_UNICODE_CS) && 
	((language&0xFF)==0x9)) {
      found = 1;
      break;
    }
    if (platform==TT_PLATFORM_APPLE_UNICODE && language==TT_MAC_LANGID_ENGLISH) {
      found = 1;
      break;
    }
  }
  if (!found) return "";
  // Found a Unicode english name
  while ((len/2+1)>fontNameLen) {
    fontNameLen *= 2;
    fontName = (char*)realloc(fontName, fontNameLen);
    if (fontName == NULL) {
      fprintf(stderr, "Unable to reserve %d bytes of memory; exiting ...", fontNameLen);
      exit(1);
    }
  }
  for (i=0; i<len; i+=2) fontName[i/2] = string[i+1];
  fontName[len/2] = '\0';
  return fontName;
}



// Tries to find a character map and returns it in charmap
// It looks for a Unicode or Symbol MS character map
// This is called when trying to show a character from its code
static TT_Error findcharmap(FontData* fd, TT_CharMap* charmap)
{
  TT_UShort platformID, encodingID;
  TT_UShort i;

  TT_Face face = fd->ttdata.face;
  int num = fd->ttdata.properties.num_CharMaps;
  for (i=0; i<num; i++) {
    int error = TT_Get_CharMap_ID(face, i, &platformID, &encodingID);
    if (error) continue;
    if (platformID==TT_PLATFORM_MICROSOFT && 
	(encodingID==TT_MS_ID_UNICODE_CS||encodingID==TT_MS_ID_SYMBOL_CS))
      return TT_Get_CharMap(face, i, charmap);
  }
  return TT_Err_Invalid_CharMap_Handle;
}



// Tries to find a normal (text) character map and returns it in charmap
// It looks for a Unicode character map
// This is called when trying to show a string
static TT_Error findtextcharmap(FontData* fd, TT_CharMap* charmap)
{
  TT_UShort platformID, encodingID;
  TT_UShort i;
  TT_Face face = fd->ttdata.face;
  int num = fd->ttdata.properties.num_CharMaps;

  // Try first to find a Unicode charmap
  for (i=0; i<num; i++) {
    int error = TT_Get_CharMap_ID(face, i, &platformID, &encodingID);
    if (error) continue;
    if ((platformID==TT_PLATFORM_MICROSOFT && encodingID==TT_MS_ID_UNICODE_CS) || 
	(platformID==TT_PLATFORM_APPLE_UNICODE))
      return TT_Get_CharMap(face, i, charmap);
  }

  // Try now a MacRoman charmap
  for (i=0; i<num; i++) {
    int error = TT_Get_CharMap_ID(face, i, &platformID, &encodingID);
    if (error) continue;
    if (platformID==TT_PLATFORM_MACINTOSH && encodingID==TT_MAC_ID_ROMAN) 
      return TT_Get_CharMap(face, i, charmap);
  }

  // No luck
  return TT_Err_Invalid_CharMap_Handle;
}




// Fills the FontData with data; fd->ttdata.face must be a valid value
void fill_ttfontdata(FontData* fd)
{
  int kerntable;
  int error;

  // Get font properties
  TT_Get_Face_Properties(fd->ttdata.face, &fd->ttdata.properties);
  fd->num_glyphs = fd->ttdata.properties.num_Glyphs;
  fd->fontName = g_strdup(getfontname(fd, TT_NAME_ID_FULL_NAME));
  fd->fontType = TTFONT;
  fd->ttdata.PSloaded = FALSE;

  // Get kerning directory and kerning pairs
  fd->ttdata.kernpairs = 0;
  error = TT_Get_Kerning_Directory(fd->ttdata.face, &fd->ttdata.kerndir);
  if (error) fd->ttdata.kerndir.nTables = 0;
  if (fd->ttdata.kerndir.nTables > 0) {
    for (kerntable=0; kerntable<fd->ttdata.kerndir.nTables; kerntable++) {
      error = TT_Load_Kerning_Table(fd->ttdata.face, kerntable);
      if (error) {
	add_error(fd, TT_ErrToString18(error));
	continue;
      }
      // By now, we only understand format 0, not format 2
      if (fd->ttdata.kerndir.tables[kerntable].format == 0) break;
    }
    if (kerntable == fd->ttdata.kerndir.nTables) 
      add_error(fd, "%s: Cannot find a kern table in format 0", fd->fontName);
    else {
      fd->ttdata.kernpairs = fd->ttdata.kerndir.tables[kerntable].t.kern0.nPairs;
      fd->ttdata.kerntable = kerntable;
    }
  }
}



bool is_ttfontcollection (gchar *pathname, FontData *fd)
{
  int error;
  /* Look if it has a .ttc extension */
  int len = strlen(pathname);
  if (len<4) return FALSE;
  if (!(pathname[len-4] == '.' &&
	(pathname[len-3] == 't' || pathname[len-3] == 'T') &&
	(pathname[len-2] == 't' || pathname[len-2] == 'T') &&
	(pathname[len-1] == 'c' || pathname[len-1] == 'C'))) return FALSE;
  
  error = TT_Open_Face(ttengine, pathname, &fd->ttdata.face);
  if (error) {
    add_error(fd, TT_ErrToString18(error));
    return FALSE;
  }
  TT_Get_Face_Properties(fd->ttdata.face, &fd->ttdata.properties);
  if (fd->ttdata.properties.num_Faces < 1) {
    TT_Close_Face(fd->ttdata.face);
    return FALSE;
  }
  return TRUE;
}


bool is_ttfont (gchar *pathname, FontData *fd)
{
  /* Look if it has a .ttf extension */
  int len = strlen(pathname);
  if (len<4) return FALSE;
  if (!(pathname[len-4] == '.' &&
	(pathname[len-3] == 't' || pathname[len-3] == 'T') &&
	(pathname[len-2] == 't' || pathname[len-2] == 'T') &&
	(pathname[len-1] == 'f' || pathname[len-1] == 'F'))) return FALSE;
  
  int error = TT_Open_Face(ttengine, pathname, &fd->ttdata.face);
  if (error) {
    add_error(fd, TT_ErrToString18(error));
    return FALSE;
  }
  fill_ttfontdata(fd);
  return TRUE;
}



static GdkImage* tt_makechar(FontData* fd, TT_UShort index, double size, 
			     bool antialiasing)
{
  TT_Instance        instance;
  TT_Glyph           glyph;
  TT_Glyph_Metrics   metrics;
  TT_Raster_Map      bitmap;
  GdkImage* gdkimage;
  int       error;

  error = TT_New_Instance(fd->ttdata.face, &instance);
  if (error) {
    add_error(fd, "Could not create instance");
    return NULL;
  }
  error = TT_Set_Instance_Resolutions(instance, xresolution, yresolution);
  if (error) {
    add_error(fd, "Could not set resolution");
    return NULL;
  }
  error = TT_Set_Instance_CharSize(instance, (TT_F26Dot6)(size*64));
  if (error) {
    add_error(fd, "Could not set size to %f", size);
    return NULL;
  }
  error = TT_New_Glyph(fd->ttdata.face, &glyph);
  if (error) {
    add_error(fd, "Could not create glyph object");
    return NULL;
  }
  error = TT_Load_Glyph(instance, glyph, index, TTLOAD_DEFAULT);
  if (error) {
    add_error(fd, "Could not load glyph");
    return NULL;
  }

  TT_Get_Glyph_Metrics(glyph, &metrics); // In 26.6 format and grid-fitted
  TT_BBox bbox = metrics.bbox;  
  int width  = (bbox.xMax-bbox.xMin)/64; // In pixels
  int height = (bbox.yMax-bbox.yMin)/64;

  bitmap.width = width;   // number of pixels per line
  bitmap.rows  = height;  // number of rows (pixels)
  bitmap.flow  = TT_Flow_Down;
  if (antialiasing) // number of columns (bytes) per row
    bitmap.cols = (bitmap.width+3) & -4; // 4 byte padding
  else
    bitmap.cols = (bitmap.width+7)/8;    // 1 byte padding
  bitmap.size = bitmap.rows * bitmap.cols; // bit/pixmap size in bytes
  bitmap.bitmap = calloc(1, bitmap.size);
  if (bitmap.bitmap == NULL) {
    fprintf(stderr, "Unable to reserve %ld bytes of memory; exiting ...", 
	    bitmap.size);
    exit(1);
  }

  if (antialiasing)
    error = TT_Get_Glyph_Pixmap(glyph, &bitmap, -bbox.xMin, -bbox.yMin);
  else
    error = TT_Get_Glyph_Bitmap(glyph, &bitmap, -bbox.xMin, -bbox.yMin);
  if (error) {
    add_error(fd, "Could not create glyph: %s", TT_ErrToString18(error));
    return NULL;
  }
  TT_Done_Glyph(glyph);
  TT_Done_Instance(instance);

  if (antialiasing) {
    int image_w = width;  // or bitmap.cols
    gdkimage = gdk_image_new(GDK_IMAGE_NORMAL, visual, image_w, height);
    // Translate the freetype gray values (0-4) to the pixel values contained in graypalette[]
    for (int y=0, bytesdone=0; y<height; y++, bytesdone+=bitmap.cols) {
      for (int x=0; x<image_w; x++) {
	guint8 pixel = ((char*)bitmap.bitmap)[bytesdone+x];  // 0 - 4
	gdk_image_put_pixel(gdkimage, x, y, graypalette[pixel].pixel);
      }
    }
    free(bitmap.bitmap);
  }
  else // gdk_image_destroy will free the bitmap 
    gdkimage = gdk_image_new_bitmap(visual, bitmap.bitmap, width, height);
  return gdkimage;
}




void tt_showchar(FontData* fd, guint character, double size)
{
  TT_CharMap charmap;
  int error;

  error = findcharmap(fd, &charmap);
  if (error) {
    errormsg("Could not find proper charmap");
    return;
  }
  TT_UShort index = TT_Char_Index(charmap, character);
  
  GdkImage* gdkimage = tt_makechar(fd, index, size, antialiasing);
  if (gdkimage == NULL) return;

  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(window, "char window");
  gtk_container_border_width(GTK_CONTAINER(window), 10);
  gtk_window_set_title(GTK_WINDOW(window), 
		       getfontname(fd, TT_NAME_ID_PS_NAME));

  GtkWidget* gtkimage = gtk_image_new(gdkimage, NULL);
  gtk_container_add(GTK_CONTAINER(window), gtkimage);
  gtk_signal_connect(GTK_OBJECT(window), "delete_event", 
		     GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     (GtkSignalFunc)delete_image, (gpointer)gdkimage);

  gtk_object_set_data(GTK_OBJECT(window), "image", (gpointer)gdkimage);
  GtkWidget *popupmenu = make_imagepopupmenu(window);
  gint event_mask = gtk_widget_get_events(window);
  event_mask |= GDK_BUTTON_PRESS_MASK; 
  gtk_widget_set_events(window, event_mask);
  gtk_signal_connect(GTK_OBJECT(window), "button_press_event",
		     GTK_SIGNAL_FUNC(image_clicked), GTK_OBJECT(popupmenu));

  int hid = gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
				      GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(window), "delete_handler", 
		      GINT_TO_POINTER(hid));
  gtk_widget_show_all(window);
}




static gint mouse_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  WInfo *winfo;
  FontData *fd;
  GdkModifierType mask;
  gint state;
  gint x, y;
  gint width_w, height_w;  // Of the drawing_area window
  gint width, height;      // Of a cell
  gint col, row;  // In drawing_area
  gint chrsize;

  static GString *info_string=NULL;  // Character names shown below drawing_area
  static GtkWidget *popup = NULL;    // Enlarged character window
  static guint buttonclicked = 0;    // Button causing popup to activate

  if (info_string == NULL) info_string = g_string_new(NULL);

  winfo = (WInfo*)data;
  GtkNotebook *notebook = GTK_NOTEBOOK(winfo->notebook);  
  int page = gtk_notebook_get_current_page(notebook);

  switch(event->type) {

  case GDK_LEAVE_NOTIFY:
    // Delete info_label if no mouse key pressed
    state = event->crossing.state;
    if (state&GDK_BUTTON1_MASK || state&GDK_BUTTON2_MASK || 
	state&GDK_BUTTON3_MASK)
      return FALSE;  
    g_string_truncate(info_string, 0);
    gdk_window_clear(winfo->info_label->window);  // Delete old info string
    return FALSE;

  case GDK_MOTION_NOTIFY:
    {
      if (event->motion.is_hint)
	gdk_window_get_pointer(event->motion.window, &x, &y, &mask);
      // Do not go further if mouse moved with a mouse key pressed
      if (mask&GDK_BUTTON1_MASK || mask&GDK_BUTTON2_MASK || 
	  mask&GDK_BUTTON3_MASK)
	return FALSE;  
      width = winfo->cellwidth;  height = winfo->cellheight;
      col = x/width;  row = y/height;
      TT_UShort index = 256*page + 16*row + col;  // Glyph number in the file  
      if (col<0 || col>=16 || row<0 || row>=16 || 
	  index>=winfo->fd->num_glyphs) {
	g_string_truncate(info_string, 0);
	gdk_window_clear(winfo->info_label->window);  // Delete old info string
	return FALSE; // Out of the grid
      }
      
      char *ps_name;
      if (winfo->fd->ttdata.PSloaded == FALSE) return FALSE;
      int error = TT_Get_PS_Name(winfo->fd->ttdata.face, index, &ps_name);
      if (error) return FALSE;
      if (strcmp(ps_name, info_string->str) == 0) return FALSE; // info_string not changed
      
      g_string_assign(info_string, ps_name);
      gdk_window_clear(winfo->info_label->window);  // Delete old info string
      gdk_draw_string(winfo->info_label->window, 
		      winfo->info_label->style->font, 
		      winfo->info_label->style->fg_gc[GTK_STATE_NORMAL], 
		      2, 2 + winfo->info_label->style->font->ascent, 
		      info_string->str);
      return FALSE; 
    }

  case GDK_BUTTON_PRESS:
    {
      fd = winfo->fd;
      if (buttonclicked != 0) return FALSE;  // Do not allow a second parallel click
      buttonclicked = event->button.button;  // Record which button was clicked
      switch (event->button.button) {
      case 1: chrsize = 100; break;
      case 2: chrsize = 200; break;
      case 3: chrsize = 300; break;
      default: // sorry, scrolling wheels don't do anything here 
	return FALSE;
      }
      width = winfo->cellwidth;
      height = winfo->cellheight;
      col = ((int)event->button.x)/width;
      row = ((int)event->button.y)/height;
      if (col<0 || col>=16 || row<0 || row>=16) return FALSE; // Out of the grid
      
      TT_UShort index = 256*page + 16*row + col;  // Glyph number in the file  
      if (index >= fd->num_glyphs) return FALSE;  // No such glyph
      GdkImage* gdkimage = tt_makechar(fd, index, chrsize, FALSE);  // No antialiasing
      if (gdkimage == NULL) return FALSE;
      GtkWidget* gtkimage = gtk_image_new(gdkimage, NULL);
      
      // Make popup window with the image
      popup = gtk_window_new(GTK_WINDOW_POPUP);
      gtk_container_border_width(GTK_CONTAINER(popup), 10);
      gtk_container_add(GTK_CONTAINER(popup), gtkimage);
      // Calculate position of the popup in the screen
      int pos_x = (int)event->button.x_root;
      int pos_y = (int)event->button.y_root;
      if (pos_x+gdkimage->width > gdk_screen_width()) 
	pos_x -= gdkimage->width;
      if (pos_y+gdkimage->height > gdk_screen_height()) 
	pos_y -= gdkimage->height;
      gtk_widget_set_uposition(popup, pos_x, pos_y);
      
      gtk_signal_connect(GTK_OBJECT(popup), "delete_event", 
			 GTK_SIGNAL_FUNC(gtk_false), NULL);
      gtk_signal_connect(GTK_OBJECT(popup), "destroy", 
			 (GtkSignalFunc)delete_image, (gpointer)gdkimage);
      gtk_widget_show_all(popup);
      gtk_grab_add(widget);
      return FALSE;
    }

  case GDK_BUTTON_RELEASE:
    if (event->button.button != buttonclicked) return FALSE;
    buttonclicked = 0;
    if (popup) {
      gtk_widget_destroy(popup);
      gtk_widget_destroyed(popup, &popup);  // Sets popup to NULL (2nd argument)
      gtk_grab_remove(widget);
      g_string_truncate(info_string, 0);
      gdk_window_clear(winfo->info_label->window);  // Delete old info string
    }
    return FALSE;

  default: break;
  }
  return FALSE;
}




static void destroy_fonttable(GtkWidget *widget, gpointer data) 
{
  WInfo *winfo = (WInfo*)data;
  FontData *fd = winfo->fd;
  fd->refcount--;  // FontData* no longer used by this window
  // If the font is no longer being displayed in the font list, try to delete it
  if (!fd->visible) destroyfont(fd);
  TT_Done_Instance(winfo->instance);

// Deletes the pixmap list associated to a fontmap window (one pixmap for each notebook page)
  GSList* pixmaplist = winfo->pixmaplist;
  while (pixmaplist) {
    gdk_pixmap_unref((GdkPixmap*)pixmaplist->data);
    pixmaplist = pixmaplist->next;
  }
  g_slist_free(pixmaplist);
  g_free(winfo);
}




static void saveasGIF(gpointer callback_data,
		      guint    callback_action,
		      GtkWidget *widget) 
{
  
  GtkWidget *window = (GtkWidget*)callback_data;
  WInfo *winfo = (WInfo*)gtk_object_get_data(GTK_OBJECT(window), "winfo");

  GtkNotebook *notebook = GTK_NOTEBOOK(winfo->notebook);  
  int pageno = gtk_notebook_get_current_page(notebook);
  GdkPixmap *pixmap = (GdkPixmap*)g_slist_nth(winfo->pixmaplist, pageno)->data;

  /* Get image from pixmap and put the reference to it 
     in the 'window' object */
  GdkImage *image = gdk_image_get(pixmap, 0, 0, 
				  16*winfo->cellwidth, 
				  16*winfo->cellheight);
  gtk_object_set_data(GTK_OBJECT(window), "image", (gpointer)image); 
  
  /* The 'image' and 'delete_handler' in window are needed by save_asgif */
  save_asgif(window);
  gdk_image_destroy(image); 
}



/* 
   Actually draw the contents of the fonttable in the pixmap

 */
static void fill_pixmap(WInfo *winfo, int page)
{
  FontData *fd = winfo->fd;
  int width = winfo->cellwidth;
  int height = winfo->cellheight;
  double size = winfo->fontsize;
  int width_w = 16*width;
  int height_w = 16*height;
  int maxnum = MIN(256, fd->num_glyphs - page*256);  // Number of glyphs on this page
  int glyphsdone = 256*page;
  double xscale = xresolution/72.0;
  double yscale = yresolution/72.0;
  GdkImage *gdkimage;
  int error;

  TT_Instance   instance = winfo->instance;
  TT_Header     *header = fd->ttdata.properties.header;
  TT_Glyph      glyph;
  TT_Raster_Map bitmap;
  TT_BBox       bbox;

  bbox.xMin = header->xMin;
  bbox.yMin = header->yMin;
  bbox.xMax = header->xMax;
  bbox.yMax = header->yMax;
  // In pixels
  int maxdescent = (int)(bbox.yMin*size*yscale/header->Units_Per_EM);  
  int maxleft = (int)(bbox.xMin*size*xscale/header->Units_Per_EM);

  bitmap.width = width_w;  // number of pixels per line
  bitmap.rows = height_w;  // number of rows (pixels)
  bitmap.flow = TT_Flow_Down;
  if (winfo->antialiasing) // number of columns (bytes) per row
    bitmap.cols = (bitmap.width+3) & -4; // 4 byte padding
  else
    bitmap.cols = (bitmap.width+7)/8;    // 1 byte padding
  bitmap.size = bitmap.rows * bitmap.cols; // bitmap size in bytes
  bitmap.bitmap = calloc(1, bitmap.size);
  if (bitmap.bitmap == NULL) {
    fprintf(stderr, "Unable to reserve %ld bytes of memory; exiting ...", 
	    bitmap.size);
    exit(1);
  }

  error = TT_New_Glyph(fd->ttdata.face, &glyph);
  if (error) {
    errormsg("Could not create glyph object");
    return;
  }

  // Put glyphs; i=column, j=row
  for (int j=0; j<=(maxnum-1)/16; j++)  
    for (int i=0; i<MIN(16, maxnum-j*16); i++) {
      
      gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 
			      double(++glyphsdone)/fd->num_glyphs);
      while (gtk_events_pending()) gtk_main_iteration();
      
      TT_UShort idx = 256*page + 16*j + i; 
      TT_UShort flags = TTLOAD_SCALE_GLYPH;
      if (winfo->hinting) flags |= TTLOAD_HINT_GLYPH;
      error = TT_Load_Glyph(instance, glyph, idx, flags);
      if (error) {
	add_error(fd, "Could not load glyph %x", idx);
	continue;
      }
      if (winfo->antialiasing)
	error = TT_Get_Glyph_Pixmap(glyph, &bitmap, 
				    (width*i-maxleft+width/12)*64, 
				    (height*(15-j) - 
				     maxdescent+height/12)*64); 
      else
	error = TT_Get_Glyph_Bitmap(glyph, &bitmap, 
				    (width*i-maxleft+width/12)*64, 
				    (height*(15-j) - 
				     maxdescent+height/12)*64); 
      if (error) {
	add_error(fd, "Could not create glyph: %s", 
		  TT_ErrToString18(error));
	continue;
      }
    }

  if (winfo->antialiasing) {
    gdkimage = gdk_image_new(GDK_IMAGE_NORMAL, visual, width_w, height_w);
    /* Translate the freetype gray values (0-4) to the pixel values 
       contained in graypalette[] */
    for (int y=0, bytesdone=0; y<height_w; y++, bytesdone+=bitmap.cols) {
      for (int x=0; x<width_w; x++) {
	guint8 pixel = ((char*)bitmap.bitmap)[bytesdone+x];  // 0 - 4
	gdk_image_put_pixel(gdkimage, x, y, graypalette[pixel].pixel);
      }
    }
  }
  else {
    gdkimage = gdk_image_new_bitmap(visual, bitmap.bitmap, 
				    width_w, height_w);
  }

  GdkPixmap *pixmap = (GdkPixmap*)g_slist_nth(winfo->pixmaplist, page)->data;
  gdk_draw_image(pixmap, bwGC, gdkimage, 0, 0, 0, 0, width_w, height_w);
  for (int i=1; i<=16; i++) {  // Draw grid
    gdk_draw_line(pixmap, defGC, 0, height*i, width_w, height*i);
    gdk_draw_line(pixmap, defGC, width*i, 0, width*i, height_w);
  }
  
  gdk_image_destroy(gdkimage);
  if (antialiasing) free(bitmap.bitmap);
  TT_Done_Glyph(glyph);
}




/* 
   Called to toggle the representation of the map in the 
   font table window
*/
static void switch_options(gpointer callback_data,
			   guint    callback_action,
			   GtkWidget *widget) 
{
  GtkWidget *item;
  GtkWidget *window = (GtkWidget*)callback_data;

  set_window_busy(window, TRUE);
  WInfo *winfo = (WInfo*)gtk_object_get_data(GTK_OBJECT(window), "winfo");

  item = gtk_item_factory_get_item(winfo->item_factory, 
				   "/View/Antialiasing");
  winfo->antialiasing = GTK_CHECK_MENU_ITEM(item)->active;
  item = gtk_item_factory_get_item(winfo->item_factory, 
				   "/View/Hinting");
  winfo->hinting = GTK_CHECK_MENU_ITEM(item)->active;

  GSList *plist = winfo->pixmaplist;
  for (int page=0; plist; plist=plist->next, page++) {
    fill_pixmap(winfo, page); 
  }
  gtk_widget_draw(winfo->notebook, NULL);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0.0);
  set_window_busy(window, FALSE);
}



static GtkItemFactoryEntry ftmenu_items[] = { 
  // path accelerator callback callback_action item_type
{ "/_File",      NULL,           NULL, 0, "<Branch>" },
{ "/File/_Save as GIF",  "<control>S",  (GtkItemFactoryCallback)saveasGIF,  0, NULL },
{ "/File/_Close",  "<control>C",  (GtkItemFactoryCallback)gtk_widget_destroy,  0, NULL },
{ "/_View",      NULL,           NULL, 0, "<Branch>" },
{ "/View/_Antialiasing",  "<control>A",  (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"},
{ "/View/_Hinting",  "<control>H",  (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"}
};




void tt_fonttable(FontData* fd, double size)
{
  TT_BBox            bbox;
  TT_Post            post;
  int error;
  
  if (fd->num_glyphs < 1) return;  // Ill case
  WInfo *winfo = g_new(WInfo, 1);

  error = TT_New_Instance(fd->ttdata.face, &winfo->instance);
  if (error) {
    errormsg("Could not create instance");
    return;
  }
  error = TT_Set_Instance_Resolutions(winfo->instance, 
				      xresolution, yresolution);
  if (error) {
    errormsg("Could not set resolution");
    return;
  }
  error = TT_Set_Instance_CharSize(winfo->instance, (TT_F26Dot6)(size*64));
  if (error) {
    errormsg("Could not set size to %f", size);
    return;
  }

  if (fd->ttdata.PSloaded == FALSE) {
    error = TT_Load_PS_Names(fd->ttdata.face, &post);
    if (error) 
      add_error(fd, "Cannot load PostScript name table: %s", 
		TT_ErrToString18(error));
    else 
      fd->ttdata.PSloaded = TRUE;
  }

  TT_Header *header = fd->ttdata.properties.header;
  bbox.xMin = header->xMin;
  bbox.yMin = header->yMin;
  bbox.xMax = header->xMax;
  bbox.yMax = header->yMax;

  int sizex = bbox.xMax - bbox.xMin;  // In font units
  int sizey = bbox.yMax - bbox.yMin;
  if (sizex<=0 || sizey<=0) {
      errormsg("Invalid bounding box in font");
      return;
  }
  double xscale = xresolution/72.0;
  double yscale = yresolution/72.0;

  // Get the size of the cell surrounding the glyph: it should be 1.2 times the
  // size of the glyph
  int width = (int)((sizex*size*xscale*1.2)/header->Units_Per_EM);  // Pixels
  int height = (int)((sizey*size*yscale*1.2)/header->Units_Per_EM);
  int width_w = width*16 + 1;   // 16 cells in a row or column
  int height_w = height*16 + 1;

    // Create window and image
  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(window, "fonttable window");
  gtk_window_set_title(GTK_WINDOW(window), 
		       getfontname(fd, TT_NAME_ID_FULL_NAME));
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
  int hid = gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
				      GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(window), "delete_handler", 
		      GINT_TO_POINTER(hid));
  gtk_widget_realize(window);


  // Global vertical container
  GtkWidget *gvbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), gvbox);

  // Menubar
  GtkAccelGroup *accel_group = gtk_accel_group_new();
  GtkItemFactory *item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, 
						      "<main>", accel_group);
  gint nmenu_items = sizeof(ftmenu_items) / sizeof (ftmenu_items[0]);
  // Pass window as callback_data
  gtk_item_factory_create_items(item_factory, nmenu_items, 
				ftmenu_items, window);
  GtkWidget *menubar = gtk_item_factory_get_widget(item_factory, "<main>");
  gtk_accel_group_attach(accel_group, GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(gvbox), menubar, FALSE, FALSE, 0);

  // Set antialias menu item to current state of check button
  GtkWidget *aaliasmenu = gtk_item_factory_get_item(item_factory, 
						    "/View/Antialiasing");
  GTK_CHECK_MENU_ITEM(aaliasmenu)->active = antialiasing;

  // Set hinting menu item to TRUE
  GtkWidget *hintmenu = gtk_item_factory_get_item(item_factory, 
						    "/View/Hinting");
  GTK_CHECK_MENU_ITEM(hintmenu)->active = TRUE;

  // Vertical container
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(gvbox), vbox, TRUE, TRUE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 5);

  // Notebook
  GtkWidget *notebook = gtk_notebook_new();
  gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);
  gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

  // Info label
  GtkWidget *evb = gtk_event_box_new();
  gtk_widget_set_name(evb, "info label");
  gtk_box_pack_end(GTK_BOX(vbox), evb, FALSE, FALSE, 0);
  GtkWidget *label = gtk_label_new(NULL);
  gtk_widget_set_name(label, "info label");
  gtk_misc_set_padding(GTK_MISC(label), 0, 2);
  gtk_container_add(GTK_CONTAINER(evb), label);

  // Set infos for the mouse_event function (called when mouse clicked on drawing area)
  winfo->fd = fd;
  winfo->fontsize = size;
  winfo->antialiasing = antialiasing;
  winfo->hinting = GTK_CHECK_MENU_ITEM(hintmenu)->active;
  winfo->notebook = notebook;
  winfo->item_factory = item_factory;
  winfo->cellwidth = width;
  winfo->cellheight = height;
  winfo->pixmaplist = NULL;   // This holds all pixmaps in the window (one for each notebook tab)
  winfo->info_label = evb;
  gtk_object_set_data(GTK_OBJECT(window), "winfo", winfo);

  /* Loop now for all pages in the notebook (256 glyphs per page) */
  for (int page=0; page<=(fd->num_glyphs-1)/256; page++) {
    char labelstr[128];
    sprintf(labelstr, "Page %d", page+1);
    
    GdkPixmap* pixmap = gdk_pixmap_new(window->window, 
				       width_w, height_w, -1);
    winfo->pixmaplist = g_slist_append(winfo->pixmaplist, (gpointer)pixmap);
    
    // Make pixmap white
    gdk_draw_rectangle(pixmap, window->style->white_gc, TRUE, 0, 0, -1, -1); 
    
    GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_border_width(GTK_CONTAINER(scwindow), 10);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				   GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
    GtkWidget *label = gtk_label_new(labelstr);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), scwindow, label);
    
    GtkWidget* drawing_area = gtk_drawing_area_new();
    gtk_widget_set_events(drawing_area, GDK_EXPOSURE_MASK | 
			  GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK |
			  GDK_POINTER_MOTION_MASK | 
			  GDK_POINTER_MOTION_HINT_MASK |
			  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwindow), 
					  drawing_area);
    gtk_signal_connect(GTK_OBJECT(drawing_area), "expose_event", 
		       (GtkSignalFunc)expose_event, (gpointer)pixmap);
    gtk_signal_connect(GTK_OBJECT(drawing_area), "motion_notify_event", 
		       (GtkSignalFunc)mouse_event, (gpointer)winfo);
    gtk_signal_connect(GTK_OBJECT(drawing_area), "leave_notify_event", 
		       (GtkSignalFunc)mouse_event, (gpointer)winfo);
    
    // Set the size of the area so the scwindow knows
    gtk_drawing_area_size(GTK_DRAWING_AREA(drawing_area), width_w, height_w);
    /* And the size of the scwindow to request a view of the whole pixmap;
       if the dimensions of the screen are exceeded, it will be limited in
       t1_fonttable()
    */
    gtk_widget_set_usize(GTK_BIN(scwindow)->child, width_w, height_w);
    
    fill_pixmap(winfo, page);
    
    if (size<100.0) {  // Expansion window only for small sizes
      gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event", 
			 (GtkSignalFunc)mouse_event, (gpointer)winfo);
      gtk_signal_connect(GTK_OBJECT(drawing_area), "button_release_event", 
			 (GtkSignalFunc)mouse_event, (gpointer)winfo);
    }
    
  } // Page drawing
  fd->refcount++;  // FontData* used by this window
  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0.0);

  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     (GtkSignalFunc)destroy_fonttable, (gpointer)winfo);

  /* Compute the size requested by the widgets in the window */
  GtkRequisition req;
  gtk_widget_show_all(gvbox);
  gtk_widget_size_request(window, &req);

  /* Set the size so that the complete fontmap is visible on screen, 
     but only up to the screen size */
  gtk_widget_set_usize(window, 
		       MIN(req.width,  gdk_screen_width() -30),
  		       MIN(req.height, gdk_screen_height()-40));
  gtk_widget_show(window);
}




static inline guint32 val(const char* ptr)
{
  guint16 l = ((guint16*)ptr)[0];
  guint16 r = ((guint16*)ptr)[1];
  return (((guint32)l)<<16) + (guint32)r;
}


// Returns kerning value in FUnits
static TT_FWord getkerning(FontData* fd, TT_UShort ch1, TT_UShort ch2)
{
  int kerntable = fd->ttdata.kerntable;
  guint32 target = (((guint32)ch1)<<16) + (guint32)ch2;

  if (ch2 == 0) return 0;   // Save time: end of string
  TT_UShort searchRange = fd->ttdata.kerndir.tables[kerntable].t.kern0.searchRange;
  TT_UShort rangeShift  = fd->ttdata.kerndir.tables[kerntable].t.kern0.rangeShift;
  char* base = (char*)(fd->ttdata.kerndir.tables[kerntable].t.kern0.pairs);
  char* top  = base + searchRange;

  if (target < val(base)) return 0;  // value < lower limit
  if (searchRange>6 && target>val(top-6)) {  // if value > upper limit
    // Should also be done with top = base if searchRange/6 is not a power of 2
    // Linear search in this zone
    for (char* ptr = top; ptr < top+rangeShift; ptr += 6) 
      if (target == val(ptr)) return ((TT_FWord*)ptr)[2];
    return 0;  // Not found
  }  
  if (target==val(base)) return ((TT_FWord*)base)[2];

  // Binary search between base and top (open interval)
  while (searchRange>6) {  // Each element in the table has 6 bytes
    searchRange /= 2;
    char *next = base + searchRange;
    if (target>val(next)) base = next;
    else { 
      if (target<val(next)) 
	top = next;
      else 
	return ((TT_FWord*)next)[2];   // Found!!
    }
  }
  return 0;  // No such kerning pair
}




static GdkImage* tt_makestring(FontData* fd, const char* string, double size, 
			       bool kerning, bool antialiasing)
{
  TT_Instance        instance;
  TT_Glyph           glyph;

  TT_Glyph_Metrics  metrics;
  TT_Raster_Map     bitmap;
  TT_BBox           bbox;
  TT_CharMap        charmap;
  GdkImage *gdkimage, *gdkimagebig;
  int error;

  error = findtextcharmap(fd, &charmap);
  if (error) {
    errormsg("Could not find proper text charmap");
    return NULL;
  }
  error = TT_New_Instance(fd->ttdata.face, &instance);
  if (error) {
    errormsg("Could not create instance");
    return NULL;
  }
  error = TT_Set_Instance_Resolutions(instance, xresolution, yresolution);
  if (error) {
    errormsg("Could not set resolution");
    return NULL;
  }
  error = TT_Set_Instance_CharSize(instance, (TT_F26Dot6)(size*64));
  if (error) {
    errormsg("Could not set size to %f", size);
    return NULL;
  }
  error = TT_New_Glyph(fd->ttdata.face, &glyph);
  if (error) {
    errormsg("Could not create glyph object");
    return NULL;
  }

  TT_Header *header = fd->ttdata.properties.header;
  bbox.xMin = header->xMin;  // BBox of the whole font, in Font Units
  bbox.yMin = header->yMin;
  bbox.xMax = header->xMax;
  bbox.yMax = header->yMax;

  int sizex = bbox.xMax - bbox.xMin;  // In font units
  int sizey = bbox.yMax - bbox.yMin;
  if (sizex<=0 || sizey<=0) {
    errormsg("Invalid bounding box in font");
    return NULL;
  }
  double xscale = xresolution/72.0;
  double yscale = yresolution/72.0;

  TT_F26Dot6 maxdescent = (TT_F26Dot6)(bbox.yMin*size*yscale/header->Units_Per_EM)*64;  // In 26.6 format 
  // Get the maximum size of the cell surrounding the glyph
  int width = (int)((sizex*size*xscale)/header->Units_Per_EM);  // Pixels
  int height = (int)((sizey*size*yscale)/header->Units_Per_EM);
  width *= strlen(string);  // Maximum size of the whole string

  bitmap.width = width;   // number of pixels per line
  bitmap.rows  = height;  // number of rows (pixels)
  bitmap.flow = TT_Flow_Down;
  if (antialiasing) // number of columns (bytes) per row
    bitmap.cols = (bitmap.width+3) & -4; // 4 byte padding
  else
    bitmap.cols = (bitmap.width+7)/8;    // 1 byte padding
  bitmap.size = bitmap.rows * bitmap.cols; // bit/pixmap size in bytes
  bitmap.bitmap = calloc(1, bitmap.size);  // Initialised to zeroes
  if (bitmap.bitmap == NULL) {
    fprintf(stderr, "Unable to reserve %ld bytes of memory; exiting ...", 
	    bitmap.size);
    exit(1);
  }

  TT_F26Dot6 pen_x = 0;    // Pen positions in the bitmap in 26.6 format
  TT_F26Dot6 pen_y = -maxdescent;
  TT_F26Dot6 maxy = 0;  // Maximal value of the yMax of the bbox of all characters
  TT_F26Dot6 miny = 0;
  TT_F26Dot6 mx = 0; // Maximal value of x where a pixel in the bitmap is written
  for (const guchar *ptr=(guchar*)string; *ptr; ptr++) { // ptr must be unsigned!
    TT_UShort index = TT_Char_Index(charmap, *ptr);
    error = TT_Load_Glyph(instance, glyph, index, TTLOAD_DEFAULT);
    if (error) {
      add_error(fd, "Could not load glyph %hx", index);
      continue;
    }
    TT_Get_Glyph_Metrics(glyph, &metrics); // In 26.6 format and grid-fitted
    if (ptr==(guchar*)string) pen_x = -metrics.bbox.xMin; // For the first character
    maxy = MAX(maxy, metrics.bbox.yMax);
    miny = MIN(miny, metrics.bbox.yMin);
    mx   = MAX(mx, pen_x + metrics.bbox.xMax);
    if (antialiasing)
      error = TT_Get_Glyph_Pixmap(glyph, &bitmap, pen_x, pen_y);  
    else
      error = TT_Get_Glyph_Bitmap(glyph, &bitmap, pen_x, pen_y);  
    if (error) {
      add_error(fd, "Could not create glyph %c: %s", *ptr, 
		TT_ErrToString18(error));
      continue;
    }
    pen_x += metrics.advance;
    if (kerning && fd->ttdata.kerndir.nTables>0) {
      double kern = getkerning(fd, index, TT_Char_Index(charmap, ptr[1]))*size*xscale/header->Units_Per_EM;  // In pixels
      pen_x += ((TT_F26Dot6)(64*kern)+32)&-64;  // Round it correctly
    }
  }
  TT_Done_Glyph(glyph);
  TT_Done_Instance(instance);

  if (antialiasing) {
    gdkimagebig = gdk_image_new(GDK_IMAGE_NORMAL, visual, width, height);
    // Translate the freetype gray values (0-4) to the pixel values contained in graypalette[]
    for (int y=0, bytesdone=0; y<height; y++, bytesdone+=bitmap.cols) {
      for (int x=0; x<width; x++) {
	guint8 pixel = ((char*)bitmap.bitmap)[bytesdone+x];  // 0 - 4
	gdk_image_put_pixel(gdkimagebig, x, y, graypalette[pixel].pixel);
      }
    }
    free(bitmap.bitmap);
  }
  else // no antialiasing: gdk_image_destroy will free the bitmap 
    gdkimagebig = gdk_image_new_bitmap(visual, bitmap.bitmap, width, height);

  // Now get the string without margins
  gdkimage = gdk_image_subimage(gdkimagebig, 0, height-(pen_y+maxy)/64, 
				mx/64, (maxy-miny)/64); 
  gdk_image_destroy(gdkimagebig);
  return gdkimage;
}




void tt_drawsample(FontData* fd, const char *msg, double size)
{
  gdk_window_clear(samplet->window);  // Clear old font sample
  GdkImage* image = tt_makestring(fd, msg, size, FALSE, FALSE); // No kerning, no antialiasing
  if (image == NULL) {  // For whatever reason
    gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL);  
    return;  
  }
  GtkStyle* style = gtk_widget_get_style(mainwindow);
  GdkPixmap* pixmap = gdk_pixmap_new(mainwindow->window, 
				     image->width, image->height, -1);
  gdk_draw_image(pixmap, defGC, image, 0, 0, 0, 0, -1, -1);
  gdk_image_destroy(image);

  gtk_pixmap_set(GTK_PIXMAP(samplet), pixmap, NULL);  // Deletes old pixmap too
}



void tt_showstring(FontData* fd, const char* string, double size)
{
  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(window, "string window");
  gtk_container_border_width(GTK_CONTAINER(window), 10);
  gtk_window_set_title(GTK_WINDOW(window), 
		       getfontname(fd, TT_NAME_ID_PS_NAME));

  GdkImage* gdkimage = tt_makestring(fd, string, size, kerning, antialiasing);
  if (gdkimage == NULL) return;  // For whatever reason

  GtkWidget* gtkimage = gtk_image_new(gdkimage, NULL);
  gtk_container_add(GTK_CONTAINER(window), gtkimage);
  gtk_signal_connect(GTK_OBJECT(window), "delete_event", 
		     GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     GTK_SIGNAL_FUNC(delete_image), (gpointer)gdkimage);

  gtk_object_set_data(GTK_OBJECT(window), "image", (gpointer)gdkimage);
  GtkWidget *popupmenu = make_imagepopupmenu(window);
  gint event_mask = gtk_widget_get_events(window);
  event_mask |= GDK_BUTTON_PRESS_MASK; 
  gtk_widget_set_events(window, event_mask);
  gtk_signal_connect(GTK_OBJECT(window), "button_press_event",
		     GTK_SIGNAL_FUNC(image_clicked), GTK_OBJECT(popupmenu));

  int hid = gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
				      GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(window), "delete_handler", 
		      GINT_TO_POINTER(hid));
  gtk_widget_show_all(window);
}




// ********************* Section handling the download of T42 fonts

static GSList *table_list;   // Global; makes things easier, so please allow it
static const int LINE = 36;  // Bytes per line
static const guint MAXSTRING = 65534;  // Maximal length of a PostScript string


inline int scale(FontData* fd, TT_Pos v)
{
  return (v*1000)/fd->ttdata.properties.header->Units_Per_EM;
}



// Makes list of tables to download available in table_list
// Return TRUE for success, FALSE for failure
static bool get_table_list(FontData *fd)
{
  static const char *table_name[] = {"cvt ", "fpgm", "glyf", "head", "hhea", "hmtx", "loca", "maxp", "prep"};  // Sorted list!!
  TT_Long tlen;

  // Try to load all tables in table_name
  for (guint i=0; i<sizeof(table_name)/sizeof(table_name[0]); i++) {
    TT_Long tag = MAKE_TT_TAG(table_name[i][0],table_name[i][1],table_name[i][2],table_name[i][3]);
    tlen = 0;
    // Get length of table
    int error = TT_Get_Font_Data(fd->ttdata.face, tag, 0, NULL, &tlen);
    if (error) 
      if (strcmp(table_name[i], "cvt ")==0 || 
	  strcmp(table_name[i], "fpgm")==0 || 
	  strcmp(table_name[i], "prep")==0) {  // These 3 tables are optional
	add_error(fd, "Warning: table %s not present", table_name[i]);
	continue;
      }
      else {
	errormsg("Invalid font: Table %s is missing", table_name[i]);
	return FALSE;
      }
    // Check length of table; only glyf can be larger than MAXSTRING
    if (tlen>(TT_Long)MAXSTRING && strcmp(table_name[i], "glyf")!=0) {  
      errormsg("Invalid font: Table %s is longer than %d bytes", 
	       table_name[i], MAXSTRING);
      return FALSE;
    }

    TT_Table *table = g_new(TT_Table, 1);
    if (table) table->data = g_new(guchar, tlen);
    if (table==NULL || table->data==NULL) {
      fprintf(stderr, "Unable to reserve memory; exiting ...");
      exit(1);
    }
    // Load table into memory
    error = TT_Get_Font_Data(fd->ttdata.face, tag, 0, table->data, &tlen);
    if (error) {
      errormsg("Invalid font: Cannot load table %s", table_list[i]);
      return FALSE;
    }
    table->name = table_name[i];
    table->tag = tag;
    table->length = tlen;
    table_list = g_slist_append(table_list, table);
  }

  // Read table checksums from table directory (rather than calculate ourselves)
  guchar aux[6];
  tlen = 6;
  TT_Get_Font_Data(fd->ttdata.face, 0, 0, aux, &tlen);  // Read beginning of font file
  guint num_tables = (aux[4]<<8) + aux[5];  // Number of tables in original font
  tlen = 16*num_tables;   // Length of table directory
  guchar *table_dir = g_new(guchar, tlen);
  if (table_dir == NULL) {
    fprintf(stderr, "Unable to reserve memory; exiting ...");
    exit(1);
  }
  TT_Get_Font_Data(fd->ttdata.face, 0, 12, table_dir, &tlen); // Read table directory

  // For all tables we are going to download, store checksum and offset
  num_tables = g_slist_length(table_list);  // Number of tables in downloaded font
  TT_ULong offset = 12 + 16*num_tables;     // Begin + table directory
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    // Find checksum in table directory
    for (guchar *p = table_dir; p<table_dir+tlen; p += 16) {
      TT_ULong tag = MAKE_TT_TAG(p[0], p[1], p[2], p[3]);
      TT_ULong checksum = MAKE_TT_TAG(p[4], p[5], p[6], p[7]);
      if (tb->tag == tag) tb->checksum = checksum;
    }
    // Offset
    tb->offset = offset;
    offset += (tb->length+3) & ~3;  // a table must start at a 4 byte boundary
  }  
  g_free(table_dir);  // Usar y tirar
  return TRUE;
}


// Free the table_list list made by get_table_list
static void free_table_list(void)
{
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    g_free(tb->data);
    g_free(tb);
  }
  g_slist_free(table_list);
  table_list = NULL;
}



// Write a table piece (a string for Adobe)
static void write_string(FILE *ofp, bool pad, TT_Table *tb, 
			 TT_ULong offset, TT_ULong length)
{
  static const char *hexchar = "0123456789ABCDEF";
  int pos = 0;  // Counts the number of bytes so far in the line

  fputs("<\n", ofp);
  for (guchar *p=tb->data+offset; p<tb->data+offset+length; p++) {
    if (pos++ == LINE) {
      fputc('\n', ofp);
      pos = 1;
    }
    fputc(hexchar[(*p>>4)&0xf], ofp); // Translate the byte at p to hex
    fputc(hexchar[*p&0xf], ofp);
  }
  if (pad) {
    TT_ULong nbytes = ((tb->length+3)&~3) - tb->length;  // 0-3
    while (nbytes--) fputs("00", ofp);  // Pad table with zeros to a 4 byte boundary
  }
  fputs("\n00>", ofp);  // Padding byte for compatibility with PS versions prior to 2013
}



static TT_ULong get_glyph_offset(TT_Table *loca, int ch, 
				 int Index_To_Loc_Format)
{
  TT_ULong pos, offset;

  switch (Index_To_Loc_Format) {
  case 0:  // Short format
    pos = sizeof(TT_UShort) * ch;
    offset = 2*((loca->data[pos] << 8) + loca->data[pos+1]);
    break;
  case 1:  // Long format
    pos = sizeof(TT_ULong) * ch;
    offset = MAKE_TT_TAG(loca->data[pos], loca->data[pos+1], loca->data[pos+2], loca->data[pos+3]);
    break;
  }
  return offset;
}



static void make_offset_table(FontData *fd, char *ot, TT_UShort numTables)
{
  TT_UShort searchRange=1, entrySelector=0, rangeShift=0;
  
  while(searchRange <= numTables) {
    searchRange *= 2;
    entrySelector++;
  } 
  searchRange = 16*(searchRange/2);
  entrySelector--;
  rangeShift = 16*numTables - searchRange;

  sprintf(ot, "00010000%02hX%02hX%02hX%02hX%02hX%02hX%02hX%02hX", numTables>>8, numTables&0xff, searchRange>>8, searchRange&0xff, entrySelector>>8, entrySelector&0xff, rangeShift>>8, rangeShift&0xff);
}



/*
  Download the Type 42 font onto the given file (for printing)
  Return success as boolean
  Refer to 'The Type 42 font format specification', Adobe technical note #5012,
  if you want to understand the code
  The code in 'ttftot42' by Dieter Baron (dillo@giga.or.at) was used as example
*/
bool tt_downloadfont(FILE *ofp, FontData* fd)
{
  int error;
  TT_Post   post;
  TT_Header *header = fd->ttdata.properties.header;
  TT_Postscript *ps = fd->ttdata.properties.postscript;
  TT_OS2       *os2 = fd->ttdata.properties.os2;

  if (fd->ttdata.PSloaded == FALSE) {
    error = TT_Load_PS_Names(fd->ttdata.face, &post);
    if (error) {
      errormsg("Invalid font\nCannot load PostScript names: %s", 
	       TT_ErrToString18(error));
      return FALSE;
    }
    fd->ttdata.PSloaded = TRUE;
  }
  // Compute the tables to write in the T42 font
  bool rc = get_table_list(fd);  // Sets table_list
  if (rc == FALSE) {
    free_table_list();
    return FALSE;
  }

  // Start font
  fprintf(ofp, "%%!PS-TrueTypeFont-%ld.%ld-%ld.%ld\n", 
	  header->Table_Version >> 16, 
	  ((header->Table_Version & 0xFFFF)*1000)/0x10000, 
	  header->Font_Revision >> 16, 
	  ((header->Font_Revision & 0xFFFF)*1000)/0x10000);
  fprintf(ofp, "%%%%Creator: GFontView V%s (from %s)\n", 
	  VERSION, g_basename(fd->fontFile)); 
  if (ps->minMemType42!=0 && ps->maxMemType42!=0) 
    fprintf(ofp, "%%%%VMusage: %lu %lu\n", ps->minMemType42, ps->maxMemType42);
  
  // Dictionary
  // Entries in all types of font dictionaries
  fprintf(ofp, "11 dict begin\n");
  fprintf(ofp, "  /FontName (%s) def\n", getfontname(fd, TT_NAME_ID_PS_NAME));
  fprintf(ofp, "  /FontType 42 def\n");
  fprintf(ofp, "  /FontMatrix [1 0 0 1 0 0] def\n");
  fprintf(ofp, "  /FontInfo 9 dict dup begin\n");
  fprintf(ofp, "    /version (%03ld.%03ld) readonly def \n", 
	  header->Font_Revision >> 16, 
	  ((header->Font_Revision & 0xFFFF)*1000)/0x10000);
  fprintf(ofp, "    /FullName (%s) readonly def\n", 
	  getfontname(fd, TT_NAME_ID_FULL_NAME));
  fprintf(ofp, "    /FamilyName (%s) readonly def\n", 
	  getfontname(fd, TT_NAME_ID_FONT_FAMILY));
  fprintf(ofp, "    /Notice (%s) readonly def\n", 
	  getfontname(fd, TT_NAME_ID_COPYRIGHT));
  guint weight = os2->usWeightClass/100-1;
  if (os2->usWeightClass<100 || os2->usWeightClass>900) {
    add_error(fd, "Invalid weight value in OS/2 table: %hu", 
	      os2->usWeightClass);
    weight = 3; // Regular
  }
  fprintf(ofp, "    /Weight (%s) readonly def\n", weight_table[weight]);
  fprintf(ofp, "    /ItalicAngle %.1f def\n", 
	  (ps->italicAngle>>16) + (ps->italicAngle&0xFFFF)/65536.0);
  fprintf(ofp, "    /isFixedPitch %s def\n", ps->isFixedPitch?"true":"false");
  fprintf(ofp, "    /UnderlinePosition %hd def\n", ps->underlinePosition);
  fprintf(ofp, "    /UnderlineThickness %hd def\n", ps->underlineThickness);
  fprintf(ofp, "  end readonly def\n");

  // Additional entries in all base fonts (FontType not 0)
  fprintf(ofp, "  /Encoding StandardEncoding def\n");  // Enough for our purpose
  fprintf(ofp, "  /FontBBox [%d %d %d %d] def\n", 
	  scale(fd, header->xMin), scale(fd, header->yMin), 
	  scale(fd, header->xMax), scale(fd, header->yMax));

  // Additional and modified entries in Type 42 fonts
  fprintf(ofp, "  /PaintType 0 def\n");  // Filled glyphs (2 for stroked)
  fprintf(ofp, "  /CharStrings %d dict dup begin\n", fd->num_glyphs);
  for (int i=0; i<fd->num_glyphs; i++) {
    char *psname;
    error = TT_Get_PS_Name(fd->ttdata.face, i, &psname);
    if (error) {
      add_error(fd, "Cannot get PostScript name for glyph %d: %s", i, 
		TT_ErrToString18(error));
      continue;
    }
    fprintf(ofp, "    /%s %d def\n", psname, i);
  }
  fprintf(ofp, "  end readonly def\n");

  /* The TrueType tables follow
   They consist of a series of PS strings (enclosed in <>).
   Each string must begin at TrueType table boundaries or at individual 
   glyph boundaries within the glyf table.
   Each string must have one additional padding byte appended: '00'
   The strings should consist of lines of constant n characters, separated by m
   characters of white space and/or control characters.
   Each line should not be greater than 255 characters long.
   Each string can only be 65534 bytes long (plus 1 byte padding)

   The possibly used TT tables are head, hhea, hmtx, loca, prep, fpgm, glyf,
   cvt_, maxp; from PS version 3011 on, also vhea, vmtx.
   Of these, cvt_, fpgm and prep might not be present in the ttf font
  */

  // Find the 'head' table (to store the font checksum)
  TT_Table *head = NULL;
  for (GSList *elem = table_list; elem; elem = elem->next) {
    head = (TT_Table*)elem->data;
    if (strcmp(head->name, "head")==0) break;
  }
  
  // Find the 'loca' table (so that we can split the glyf table)
  TT_Table *loca = NULL;
  for (GSList *elem = table_list; elem; elem = elem->next) {
    loca = (TT_Table*)elem->data;
    if (strcmp(loca->name, "loca")==0) break;
  }

  char ot[32];  // Offset table (Why is it called like that?)
  fprintf(ofp, "%% Font download\n");
  make_offset_table(fd, ot, g_slist_length(table_list));
  fprintf(ofp, "/sfnts [ <\n%s\n", ot);  // font header
  TT_ULong checksum = MAKE_TT_TAG(ot[0], ot[1], ot[2], ot[3]) + 
    MAKE_TT_TAG(ot[4], ot[5], ot[6], ot[7]) + 
    MAKE_TT_TAG(ot[8], ot[9], ot[10], ot[11]);

  // Write table directory
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    checksum += tb->tag + 2*tb->checksum + tb->offset + tb->length;
    fprintf(ofp, "%08.8lX%08.8lX%08.8lX%08.8lX\n", tb->tag, tb->checksum, 
	    tb->offset, tb->length);
  }
  fprintf(ofp, ">");

  // Write the computed checksum in the 'head' table
  checksum = 0xb1b0afba - checksum;
  head->data[8]  = (checksum>>24)&0xff;
  head->data[9]  = (checksum>>16)&0xff;
  head->data[10] = (checksum>>8)&0xff;
  head->data[11] = checksum&0xff;

  // Go table by table and download it
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    fprintf(ofp, "\n%% %s table\n", tb->name);
    if (strcmp(tb->name, "glyf")!=0) {
      // All but the glyf table (already checked length of table)
      write_string(ofp, TRUE, tb, 0, tb->length);
      continue;
    }
    // The glyf table (has to be splitted probably)
    for (int ch=0; ch<=fd->num_glyphs; ch++) {
      TT_ULong offset_now, offset_next, offset_lastwritten;
      if (ch==0) offset_now = get_glyph_offset(loca, ch, 
					       header->Index_To_Loc_Format);
      else offset_now = offset_next;
      if (ch == fd->num_glyphs) {
	// Reached last character, so flush all not written glyphs
	write_string(ofp, TRUE, tb, offset_lastwritten, 
		     offset_now - offset_lastwritten);
	break;
      }
      offset_next = get_glyph_offset(loca, ch + 1, 
				     header->Index_To_Loc_Format);
      TT_ULong chlength = offset_next - offset_now;
      if (ch==0) offset_lastwritten = offset_now;
      if (offset_now - offset_lastwritten + chlength > MAXSTRING) {
	write_string(ofp, FALSE, tb, offset_lastwritten, 
		     offset_now - offset_lastwritten);
	offset_lastwritten = offset_now;
      }
    }  
  }
  fprintf(ofp, " ] def\n");
  
  // Buf, done
  fprintf(ofp, "FontName currentdict end definefont pop\n\n");
  free_table_list();
  return TRUE;
}





// *******************  Functions to show properties of the font

static void make_entries(GtkWidget *frame, struct TT_Info *props, int num)
{
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_border_width(GTK_CONTAINER(vbox), 5);

  GtkWidget *table = gtk_table_new(num, 2, FALSE); // A table with num rows, 2 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

  for (int i=0; i<num; i++) {
    GtkWidget *label = props[i].property;
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_widget_set_style(label, stylebold);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i+1, 
		     GTK_FILL, GTK_EXPAND, 0, 0);
    label = props[i].value;
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, i, i+1, 
		     (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), 
		     (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), 0, 0);
  }
}


static GtkWidget* specific_info(FontData *fd)
{
  struct TT_Info props[20];
  char buf[50], *str;
  int nprops = 0;

  TT_Header *header = fd->ttdata.properties.header;
  TT_OS2 *os2 = fd->ttdata.properties.os2;
  TT_Postscript *ps = fd->ttdata.properties.postscript;

  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);

  props[nprops].property = gtk_label_new("Units per EM:");
  sprintf(buf, "%hu", header->Units_Per_EM);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Average width:");
  sprintf(buf, "%hd", os2->xAvgCharWidth);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Weight:");
  if (os2->usWeightClass>=100 && os2->usWeightClass<=900 && 
      os2->usWeightClass==(os2->usWeightClass/100)*100)
    sprintf(buf, "%s", weight_table[os2->usWeightClass/100-1]);
  else
    sprintf(buf, "Invalid (%hu)", os2->usWeightClass);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Width:");
  if (os2->usWidthClass>=1 && os2->usWidthClass<=9)
    sprintf(buf, "%s", width_table[os2->usWidthClass-1]);
  else
    sprintf(buf, "Invalid (%hu)", os2->usWidthClass);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Italic angle:");
  sprintf(buf, "%.1f", (ps->italicAngle>>16) + (ps->italicAngle&0xFFFF)/65536.0);
  props[nprops++].value = gtk_label_new(buf);
  
  props[nprops].property = gtk_label_new("Monospaced:");
  sprintf(buf, "%s", ps->isFixedPitch?"yes":"no");
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Character maps:");
  sprintf(buf, "%d", fd->ttdata.properties.num_CharMaps);
  props[nprops++].value = gtk_label_new(buf);

  make_entries(frame, props, nprops);
  gtk_widget_show_all(frame);
  return frame;
}


static GtkWidget* general_info(FontData *fd)
{
  struct TT_Info props[20];
  char buf[25], *str;
  int nprops = 0;

  TT_OS2 *os2 = fd->ttdata.properties.os2;

  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);

  props[nprops].property = gtk_label_new("File name:");
  props[nprops++].value = gtk_label_new(g_basename(fd->fontFile));

  props[nprops].property = gtk_label_new("Font family:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_FONT_FAMILY));

  props[nprops].property = gtk_label_new("Font subfamily:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_FONT_SUBFAMILY));

  props[nprops].property = gtk_label_new("Font full name:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_FULL_NAME));

  props[nprops].property = gtk_label_new("Font PS-name:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_PS_NAME));

  props[nprops].property = gtk_label_new("Font version:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_VERSION_STRING));

  props[nprops].property = gtk_label_new("Copyright:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_COPYRIGHT));

  props[nprops].property = gtk_label_new("Trademark:");
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_TRADEMARK));

  props[nprops].property = gtk_label_new("Embedding\nlicense:");
  if (os2->fsType==0x0000) 
    str = "No restrictions";
  else if (os2->fsType&0x0008)
    str = "Editable embedding allowed";
  else if (os2->fsType&0x0004)
    str = "Read-only embedding allowed";
  else if (os2->fsType&0x0002)
    str = "No embedding allowed";
  else
    str = "Unknown";
  props[nprops++].value = gtk_label_new(str);

  props[nprops].property = gtk_label_new("Glyphs:");
  sprintf(buf, "%u", fd->num_glyphs);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Kern pairs:");
  sprintf(buf, "%u", fd->ttdata.kernpairs);
  props[nprops++].value = gtk_label_new(buf);

  make_entries(frame, props, nprops);
  gtk_widget_show_all(frame);
  return frame;
}




static void page_switch(GtkWidget *notebook, GtkNotebookPage *page, 
			guint page_num)
{
  printf("Page: %u\n", page_num);
  GtkWidget *frame = page->child;


}



void tt_showproperties(FontData *fd, GtkWidget *window)
{
  GtkWidget *frame, *label;

  GtkWidget *notebook = gtk_notebook_new();
  gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);
  /*
  gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
		     GTK_SIGNAL_FUNC(page_switch), NULL);
  */
  gtk_container_add(GTK_CONTAINER(window), notebook);

  // General info
  frame = general_info(fd);
  label = gtk_label_new("General");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  // More specific
  frame = specific_info(fd);
  label = gtk_label_new("Specific");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  gtk_widget_show(notebook);
  gtk_widget_show(window);
}




