/*
   schema.c -
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: schema.c,v 1.15 2001/06/19 20:48:10 reinhard Exp $
 
*/

#include "config.h"

#include "geas-server.h"
#include "geas-skeleton.h"
#include "geas.h"

#include "classdata.h"
#include "schema.h"
#include "exceptions.h"

/** \brief List all fields to return
 *  \return GList of pointers to odl_field instances
 *  \param id Identify of object making the method call
 *  \param c odl_class instance to process
 *  \param includeinherited TRUE if inherited fields should be included
 */
static GList *schema_list_class_fields (GEAS_object_reference * id,
                                        odl_class * c,
                                        gboolean includeinherited);

/** \brief Convert ODL datatype to the enum fields in typedefs.idl
 */
static GEAS_Datatype odl_datatype_to_geas_datatype (enum odl_datatype dt);

GEAS_classnames *
schema_list_classes (GEAS_object_reference * id, CORBA_Environment * ev)
{
  odl_namelist *classes;
  GList *l;
  GEAS_classnames *retval;
  int i, count;
  odl_class *c;

  classes = odl_tree_list_classes (all_classes);
  count = g_list_length (classes);
  retval = GEAS_classnames__alloc ();
  if (!retval)
    {
      return (NULL);
    }
  CORBA_sequence_set_release (retval, CORBA_TRUE);
  retval->_maximum = count;
  retval->_buffer = CORBA_sequence_CORBA_string_allocbuf (retval->_maximum);
  l = classes;
  i = 0;
  while (l)
    {
      c = odl_find_class (all_classes, l->data, NULL);
      if (c && c->base.access == ODL_ACCESS_PUBLIC)
        {
          if (strncmp (l->data, "root::", 6) == 0)
            {
              retval->_buffer[i] = CORBA_string_dup (&((char *)(l->data))[6]);
            }
          else
            {
              retval->_buffer[i] = CORBA_string_dup (l->data);
            }
          i++;
        }
      l = g_list_next (l);
    }
  if (i > 0)
    {
      retval->_length = i;
    }
  else
    {
      retval->_length = 0;
    }
  if (classes)
    {
      odl_namelist_free (classes);
    }
  return retval;
}

/* Raises: UnknownClass, ServerError */
GEAS_ClassDefinition *
schema_get_class (GEAS_object_reference * id, const char *name,
                  gboolean includeinherited, CORBA_Environment * ev)
{
  GEAS_ClassDefinition *retval = NULL;
  odl_class *c;
  GList *l, *fields;
  int count, i;

  /* find class */
  c = odl_find_class (all_classes, name, NULL);
  if (!c)
    {
      make_UnknownClass_exception (ev, "Could not find class %s", name);
      return (NULL);
    }

  /* allocate return value */
  retval = GEAS_ClassDefinition__alloc ();
  if (!retval)
    {
      make_ServerError_exception (ev, "Out of memory.");
      return (NULL);
    }

  /* class name */
  retval->name = CORBA_string_dup (odl_class_get_full_name (c));

  /* parents */
  l = c->parents;
  count = g_list_length (l);
  CORBA_sequence_set_release (&retval->parents, CORBA_TRUE);
  retval->parents._maximum = count;
  retval->parents._length = count;
  retval->parents._buffer = CORBA_sequence_CORBA_string_allocbuf (count);
  for (i = 0; i < count; i++, l = g_list_next (l))
    {
      retval->parents._buffer[i] = CORBA_string_dup (l->data);
    }

  /* member fields */
  fields = schema_list_class_fields (id, c, includeinherited);
  count = g_list_length (fields);
  CORBA_sequence_set_release (&retval->fields, CORBA_TRUE);
  retval->fields._maximum = count;
  retval->fields._length = count;
  retval->fields._buffer =
    CORBA_sequence_GEAS_FieldDefinition_allocbuf (count);
  for (i = 0, l = fields; l != NULL; l = l->next, i++)
    {
      /* add field to data structure */
      GEAS_FieldDefinition *fd = &retval->fields._buffer[i];
      odl_field *f = (odl_field *) l->data;
      odl_class *cd;

      fd->name = CORBA_string_dup (odl_field_get_name (f));
      cd = (odl_class *) f->base.parent;
      fd->classname = CORBA_string_dup (odl_class_get_full_name (cd));

      if (f->format)
        {
          fd->format = CORBA_string_dup (f->format);
        }
      else
        {
          fd->format = CORBA_string_dup ("");
        }
      if (f->datatypeclass)
        {
          fd->datatypeclass = CORBA_string_dup (f->datatypeclass);
        }
      else
        {
          fd->datatypeclass = CORBA_string_dup ("");
        }

      if (f->defaultval)
        {
          fd->defaultvalue = CORBA_string_dup (f->defaultval);
        }
      else
        {
          fd->defaultvalue = CORBA_string_dup ("");
        }
      fd->datatype = odl_datatype_to_geas_datatype (f->datatype);
      fd->notnull =
        (odl_field_has_property (f, ODL_PROP_NOTNULL) ? CORBA_TRUE :
         CORBA_FALSE);
      fd->isreadonly =
        (odl_field_has_property (f, ODL_PROP_READONLY) ? CORBA_TRUE
         : CORBA_FALSE);

      switch (f->fieldtype)
        {
        case FT_basic:
          fd->type = GEAS_basic;
          break;
        case FT_lookup:
          fd->type = GEAS_lookup;
          break;
        case FT_reference:
          fd->type = GEAS_reference;
          break;
        case FT_list:
          fd->type = GEAS_list;
          break;
        case FT_method:
          fd->type = GEAS_method;
          break;
        case FT_calculated:
          fd->type = GEAS_calculated;
          break;
        case FT_readonly:
          fd->type = GEAS_nowrite;
          break;
        default:
          fd->type = GEAS_unknown;
          break;
        }

    }

  /* does this def. include all fields? */
  if (includeinherited)
    {
      retval->allfields = CORBA_TRUE;
    }
  else
    {
      retval->allfields = CORBA_TRUE;
    }
  /* done */
  return (retval);
}

GEAS_classlist *
schema_get_all (GEAS_object_reference * id, CORBA_Environment * ev)
{
  fatal_error ("Really a TODO here...");
  return NULL;
}

/** \brief Make a list of all fields in a class
 */
static GList *
schema_list_class_fields (GEAS_object_reference * id, odl_class * c,
                          gboolean includeinherited)
{
  GList *retval = NULL;
  GList *l;
  gboolean allowed;

  for (l = c->contents; l != NULL; l = g_list_next (l))
    {
      odl_field *f = (odl_field *) l->data;

      if (f->base.type == IT_field &&   /* a field */
          f->base.access == ODL_ACCESS_PUBLIC)  /* is accessible */
        {
          allowed = FALSE;

          switch (f->fieldtype)
            {
            case FT_basic:
              if (f->datatype != DT_unknown)
                allowed = TRUE;
              break;
            case FT_lookup:
            case FT_reference:
            case FT_list:
            case FT_method:
            case FT_calculated:
            case FT_readonly:
              allowed = TRUE;
              break;
            case FT_unknown:
            }

          /* add field to list */
          if (allowed)
            {
              retval = g_list_append (retval, f);
            }
        }
    }

  /* stop now, unless we want inherited fields and have parent classes */
  if (!includeinherited || c->parents == NULL)
    {
      return (retval);
    }
  /* add any parent fields */
  for (l = c->parents; l != NULL; l = g_list_next (l))
    {
      GList *tmp;
      odl_class *parent = odl_find_class (all_classes, l->data, NULL);

      if (parent)
        {
          /* get fields in just the named class */
          tmp = schema_list_class_fields (id, c, FALSE);
          if (tmp)
            {
              retval = g_list_concat (retval, tmp);
            }
        }
    }

  return (retval);
}

static GEAS_Datatype
odl_datatype_to_geas_datatype (enum odl_datatype dt)
{
  switch (dt)
    {
    case DT_char:
      return (GEAS_Char);
      break;
    case DT_int:
      return (GEAS_Int);
      break;
    case DT_int16:
      return (GEAS_Int16);
      break;
    case DT_int32:
      return (GEAS_Int32);
      break;
    case DT_int64:
      return (GEAS_Int64);
      break;
    case DT_boolean:
      return (GEAS_Boolean);
      break;
    case DT_text:
      return (GEAS_Text);
      break;
    case DT_class:
      return (GEAS_Class);
      break;
    case DT_date:
      return (GEAS_Date);
      break;
    case DT_time:
      return (GEAS_Time);
      break;
    case DT_datetime:
      return (GEAS_DateTime);
      break;
    case DT_bool:
      return (GEAS_Bool);
      break;
    case DT_float:
      return (GEAS_FloatValue);
      break;
    case DT_void:
      return (GEAS_Void);
      break;
    case DT_object:
      return (GEAS_ObjectRef);
      break;
    case DT_enum:
      return (GEAS_Enum);
      break;
    case DT_unsignedint:
      return (GEAS_UnsignedInt);
      break;
    case DT_unknown:
      return (GEAS_UnknownDatatype);
      break;
    case DT_type:
      /* TODO - need to handle new types. */
      break;
    }

  return (GEAS_UnknownDatatype);
}

GEAS_fieldlist *
schema_get_fields (GEAS_object_reference * id,
                   const char *name, gboolean includeinherited,
                   CORBA_Environment * ev)
{
  GEAS_fieldlist *retval;
  GList *l, *fields;
  int i, count;
  odl_class *c;

  c = odl_find_class (all_classes, name, NULL);
  if (!c)
    {
      make_UnknownClass_exception (ev, "Class '%s' was not found", name);
      return (NULL);
    }

  retval = GEAS_fieldlist__alloc ();
  if (!retval)
    {
      return (NULL);
    }
  CORBA_sequence_set_release (retval, CORBA_TRUE);

  fields = schema_list_class_fields (id, c, includeinherited);
  count = g_list_length (fields);
  retval->_maximum = count;
  retval->_length = count;
  retval->_buffer = CORBA_sequence_GEAS_FieldDefinition_allocbuf (count);
  for (i = 0, l = fields; l != NULL; l = l->next, i++)
    {
      GEAS_FieldDefinition *fd = &retval->_buffer[i];
      odl_field *f = (odl_field *) l->data;
      odl_class *cd;

      fd->name = CORBA_string_dup (odl_field_get_name (f));
      cd = (odl_class *) f->base.parent;
      fd->classname = CORBA_string_dup (odl_class_get_full_name (cd));

      if (f->format)
        {
          fd->format = CORBA_string_dup (f->format);
        }
      else
        {
          fd->format = CORBA_string_dup ("");
        }
      if (f->datatypeclass)
        {
          fd->datatypeclass = CORBA_string_dup (f->datatypeclass);
        }
      else
        {
          fd->datatypeclass = CORBA_string_dup ("");
        }
      if (f->defaultval)
        {
          fd->defaultvalue = CORBA_string_dup (f->defaultval);
        }
      else
        {
          fd->defaultvalue = CORBA_string_dup ("");
        }
      fd->datatype = odl_datatype_to_geas_datatype (f->datatype);
      fd->notnull =
        (odl_field_has_property (f, ODL_PROP_NOTNULL) ? CORBA_TRUE :
         CORBA_FALSE);
      fd->isreadonly =
        (odl_field_has_property (f, ODL_PROP_READONLY) ? CORBA_TRUE
         : CORBA_FALSE);
      switch (f->fieldtype)
        {
        case FT_basic:
          fd->type = GEAS_basic;
          break;
        case FT_lookup:
          fd->type = GEAS_lookup;
          break;
        case FT_reference:
          fd->type = GEAS_reference;
          break;
        case FT_list:
          fd->type = GEAS_list;
          break;
        case FT_method:
          fd->type = GEAS_method;
          break;
        case FT_calculated:
          fd->type = GEAS_calculated;
          break;
        case FT_readonly:
          fd->type = GEAS_nowrite;
          break;
        default:
          fd->type = GEAS_unknown;
          break;
        }
    }
  return (retval);
}
